
/* Web Polygraph       http://www.web-polygraph.org/
 * (C) 2003-2006 The Measurement Factory
 * Licensed under the Apache License, Version 2.0 */

#include "base/polygraph.h"

#include "xstd/h/iostream.h"
#include <fstream>
#include "xstd/h/sstream.h"
#include "xstd/h/iomanip.h"

#include "xstd/gadgets.h"
#include "base/BStream.h"
#include "base/polyVersion.h"
#include "base/AnyToString.h"
#include "csm/ContentDbase.h"
#include "csm/cdbEntries.h"
#include "csm/cdbBuilders.h"
#include "csm/XmlParser.h"

static String ThePrgName = "";
static enum { fmtVerbatim, fmtMarkup, fmtLinkOnly } TheFormat = fmtMarkup;
static int TheTotalLinkCount = 0;

static
bool readCont(ContentDbase *cdb, const String &fname) {
	CdbBuilder::TheLinkCount = 0;

	const String hname = fname == "-" ? String("stdin") : fname;
	clog << ThePrgName << ": adding "
		<< (TheFormat == fmtMarkup ? "objects from " : "entire ")
		<< "'" << hname << "' ... ";

	const int savedCount = cdb->count();
	CdbBuilder *b = TheFormat == fmtMarkup ? (CdbBuilder*) new MarkupParser :
		(TheFormat == fmtLinkOnly ? (CdbBuilder*) new LinkOnlyParser :
		(CdbBuilder*) new VerbatimParser);

	istream *is = fname == "-" ? 
		(istream*)&cin : (istream*)new ifstream(fname.cstr());
	b->db(cdb);

	unsigned long bufsize = 128*1024;
	unsigned long bufpos = 0;
	char * buf = new char[bufsize];

	while(is->good())
	{
	    if (bufpos == bufsize)
	    {
		char * temp = new char[bufsize*2];
		memcpy(temp, buf, bufsize);
		bufsize += bufsize;
		delete buf;
		buf = temp;
	    }
	    is->read(buf + bufpos, bufsize - bufpos);
	    bufpos += is->gcount();
	}

	b->configure(fname, buf, buf + bufpos);
	const bool res = b->parse();
	if (is != &cin)
		delete is;

	delete b;
	delete buf;

	if (res) {
		TheTotalLinkCount += CdbBuilder::TheLinkCount;
		clog << "\t " << setw(6) << (cdb->count() - savedCount) <<
			" object(s) and " << CdbBuilder::TheLinkCount << " link(s)";
	}
	clog << endl;
	return res;
}

static
int doShow(const String &dbName) {
	ContentDbase *cdb = new ContentDbase;
	ifstream f(dbName.cstr());
	IBStream is;
	is.configure(&f, dbName);
	cdb->load(is);
	if (is.good()) {
		cdb->print(cout);
		return 0;
	}

	cerr << ThePrgName << ": error loading db from `" << dbName << "'" << endl;
	return -2;
}

static
int doAdd(const String &dbName, int fcount, char *fnames[]) {
	ContentDbase *cdb = new ContentDbase;

	// we are using fstream for both reading and writing, and
	// in this case we need this additional code to make sure
	// that the file gets created if it does not exist yet
	{
	    // should be enough to create new file or keep existing
	    // one as it is
	    ofstream out(dbName.cstr(), ios::app);
	}

	fstream f(dbName.cstr(), ios::in|ios::out);

	{
		clog << ThePrgName << ": loading db from " << dbName << " ... ";
		IBStream is;
		is.configure(&f, dbName);
		cdb->load(is);
		clog << "\t " << setw(6) << cdb->count() << " objects" << endl;
	}

	if (fcount) {
		for (int i = 0; i < fcount; ++i)
			Must(readCont(cdb, fnames[i]));
	} else {
		Must(readCont(cdb, "-"));
	}

	clog << ThePrgName << ": got " << fcount << " files and " <<
		TheTotalLinkCount << " links" << endl;

	{
		clog << ThePrgName << ": storing db in " << dbName << " ... ";
		f.clear();
		f.seekg(0);
		OBStream os;
		os.configure(&f, dbName);
		cdb->store(os);
		if (os.good())
			clog << "\t " << setw(6) << cdb->count() << " objects" << endl;
		else
			clog << "error: " << Error::Last() << endl;
	}
	f.close();
	return 0;
}

static
int usage(std::ostream &os) {
	(void)PolyVersion();
	os << "Usage: " << ThePrgName << " <database.cdb> <command> [--option] [filename ...]" << endl
		<< "commands:" << endl
		<< "  show          dump database contents to stdout" << endl
		<< "  add           add file(s) contents to the database" << endl
		<< "options:" << endl
		<< "  format <markup|linkonly|verbatim>  split each file into markup tags, adjust links, or add as is"
		<< endl;
	return 0;
}

int main(int argc, char *argv[]) {
	ThePrgName = argv[0];

	if (argc == 2 && String("--help") == argv[1])
		return usage(cout);

	if (argc < 3)
		return usage(cerr);

	int argOff = 1;
	const String dbName = argv[argOff++];
	const String cmd = argv[argOff++];

	if (cmd != "show" && cmd != "add") {
		usage(cerr);
		cerr << "unknown command `" << cmd << "', expected `show' or `add'" << endl;
		return -1;
	}

	if (argOff < argc && strstr(argv[argOff], "--") == argv[argOff]) {
		const String option = argv[argOff++];
		if (option != "--format") {
			usage(cerr);
			cerr << "unknown option '" << option << "', expected 'format'" << endl;
			return -1;
		}
			
		const String format = argv[argOff++];
		if (format == "markup")
			TheFormat = fmtMarkup;
		else
		if (format == "linkonly")
			TheFormat = fmtLinkOnly;
		else
		if (format == "verbatim")
			TheFormat = fmtVerbatim;
		else {
			usage(cerr);
			cerr << "unknown format '" << format << "', expected 'markup' or 'verbatim'" << endl;
			return -1;
		}
	}

	if (cmd == "show")
		return doShow(dbName);
	else
	if (cmd == "add")
		return doAdd(dbName, argc-argOff, argv+argOff);

	usage(cerr);
	cerr << "unknown command `" << cmd << "', expected `show' or `add'" << endl;
	return -1;
}
