/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Cedric Pinson <cpinson@freesheep.org>
 *
 */

#include "ugameStdAfx.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#ifdef WIN32
#include "config_win32.h"
#endif

#include <osg/PositionAttitudeTransform>

#include <cal3d/cal3d.h>
#include <cal3d/model.h>
#include <cal3d/scheduler.h>

#include <ugame/animated.h>
#include <maf/application.h>
#include <maf/data.h>

static std::string getCal3DError(void) {
  char tmp[32];
  sprintf(tmp, "%d", CalError::getLastErrorLine());
  std::string error = CalError::getLastErrorDescription() + " " +
    CalError::getLastErrorText() + " " +
    CalError::getLastErrorFile() + ":" + tmp;
  return error;
}

// Model

UGAMEAnimatedModel::UGAMEAnimatedModel() :
  mOsgCalModel(new osgCal::Model)
{
}

UGAMEAnimatedModel::~UGAMEAnimatedModel()
{
}

void UGAMEAnimatedModel::Init()
{
  UGAMEArtefactModel::Init();
  g_assert(GetData() != 0 && GetData()->GetCoreModel() != 0);

  _init();
}

void UGAMEAnimatedModel::_init()
{
  // mixer
  CalScheduler* scheduler = new CalScheduler;
  mOsgCalModel->getCalModel()->setAbstractMixer(scheduler);
  mOsgCalModel->setUseColorOrTexture(false); // color and texture can be used simultaneously
  // create the calmodel from the calcoremodel
  mOsgCalModel->setCoreModel(GetData()->GetCoreModel());

  // textures that must not be shared
  if(!mOsgCalModel->setPrivateTextures(mPrivateTextures))
    g_critical("UGAMEAnimatedModel::Init: setPrivateTextures(%s ...) failed %s", mPrivateTextures[0].c_str(), getCal3DError().c_str());
  // activate meshes
  MAFCal3dData::Outfit* outfit = GetData()->GetOutfit(mOutfit);
  if(!outfit) {

    // we can't manage a default outfit (because of male and female and so on). So if an outfit is not found
    // we use the first outfit from the list as default.
    // A better behaviour should be to try outfit from 0 to nb outfit if there is an error in the default outfit ( index 0)

//     g_warning("UGAMEAnimatedModel::Init: revert to default outfit");
//     outfit = GetData()->GetOutfit(mDefaultOutfit);
    g_warning("UGAMEAnimatedModel::Init: outfit not found use the first instead");
    mOutfit=GetData()->getOutfitNameByIndex(0);
    outfit = GetData()->GetOutfit(mOutfit);
  }
  if(outfit) {
    for(MAFCal3dData::Outfit::iterator meshDescription = outfit->begin();
	meshDescription != outfit->end();
	meshDescription++) {
      const std::string& meshName = (*meshDescription)["lib"];
      if(!mOsgCalModel->setActiveMesh(meshName))
	g_critical("UGAMEAnimatedModel::Init: setActiveMesh(%s) failed %s", meshName.c_str(), getCal3DError().c_str());

      if(meshDescription->find("collision") != meshDescription->end()) {
	if(!mOsgCalModel->setInvisibleMesh(meshName))
	  g_critical("UGAMEAnimatedModel::Init: setInvisibleMesh(%s) failed %s", meshName.c_str(), getCal3DError().c_str());
	if(!mOsgCalModel->setCollisionMesh(meshName))
	  g_critical("UGAMEAnimatedModel::Init: setCollisionMesh(%s) failed %s", meshName.c_str(), getCal3DError().c_str());
      }
      std::vector<std::string> materialNames;
      for(MAFCal3dData::MeshDescription::iterator pair = meshDescription->begin();
	  pair != meshDescription->end();
	  pair++) {
	const std::string& variable = pair->first;
	const std::string& value = pair->second;
	if(variable.substr(0, 8) == "material")
	  materialNames.push_back(value);
      }
      if(!mOsgCalModel->bindMaterials(meshName, materialNames))
	g_critical("UGAMEAnimatedModel::Init: bindMaterials(%s) failed %s (hint: check the consistency of the corresponding .CMF file with cal3d_converter)", meshName.c_str(), getCal3DError().c_str());
    }
    delete outfit;
  }
  // setup

  if(!mOsgCalModel->create())
    g_critical("UGAMEAnimatedModel::Init: create failed %s", getCal3DError().c_str());

  SetArtefact(mOsgCalModel.get());
}

void UGAMEAnimatedModel::reinit()
{
	CalScheduler* scheduler = (CalScheduler*) mOsgCalModel->getCalModel()->getAbstractMixer();
	if (scheduler)
		delete scheduler;

	GetPAT()->removeChild(mOsgCalModel.get());
	mOsgCalModel.release();

	mOsgCalModel = new osgCal::Model();

	_init();
}

osgCal::Model* UGAMEAnimatedModel::GetArtefact()
{
  return mOsgCalModel.get();
}

CalCoreBone* UGAMEAnimatedModel::GetCoreBone(const std::string& name)
{
  g_assert(mOsgCalModel.get() != 0);
  g_assert(mOsgCalModel->getCalModel() != 0);
  g_assert(mOsgCalModel->getCalModel()->getSkeleton() != 0);
  g_assert(mOsgCalModel->getCalModel()->getSkeleton()->getCoreSkeleton() != 0);

  CalCoreSkeleton* skeleton = mOsgCalModel->getCalModel()->getSkeleton()->getCoreSkeleton();
  int id = skeleton->getCoreBoneId(name);
  return skeleton->getCoreBone(id);
}

int UGAMEAnimatedModel::GetCoreBoneId(const std::string& name)
{
  g_assert(mOsgCalModel.get() != 0);
  g_assert(mOsgCalModel->getCalModel() != 0);
  g_assert(mOsgCalModel->getCalModel()->getSkeleton() != 0);
  g_assert(mOsgCalModel->getCalModel()->getSkeleton()->getCoreSkeleton() != 0);

  return mOsgCalModel->getCalModel()->getSkeleton()->getCoreSkeleton()->getCoreBoneId(name);
}

CalBone* UGAMEAnimatedModel::GetBone(const std::string& name)
{
  g_assert(mOsgCalModel.get() != 0);
  g_assert(mOsgCalModel->getCalModel() != 0);
  g_assert(mOsgCalModel->getCalModel()->getSkeleton() != 0);
  g_assert(mOsgCalModel->getCalModel()->getSkeleton()->getCoreSkeleton() != 0);

  CalSkeleton* skeleton = mOsgCalModel->getCalModel()->getSkeleton();
  int id = skeleton->getCoreSkeleton()->getCoreBoneId(name);
  return skeleton->getBone(id);
}

CalModel* UGAMEAnimatedModel::GetCalModel(void) {
  g_assert(mOsgCalModel.get() != 0);
  return mOsgCalModel->getCalModel();
}

CalScheduler* UGAMEAnimatedModel::GetScheduler(void) {
  CalModel* model = GetCalModel();
  g_assert(model != 0 && model->getAbstractMixer() != 0);
  return (CalScheduler*)(model->getAbstractMixer());
}

CalCoreAnimation* UGAMEAnimatedModel::GetCoreAnimation(const std::string& name)
{
  return GetCoreAnimation(GetCoreAnimationId(name));
}

CalCoreAnimation* UGAMEAnimatedModel::GetCoreAnimation(int id)
{
  g_assert(mOsgCalModel.get() != 0);
  g_assert(mOsgCalModel->getCalCoreModel() != 0);
  return mOsgCalModel->getCalCoreModel()->getCoreAnimation(id);
}

double UGAMEAnimatedModel::GetDuration(int id)
{
  g_assert(id >= 0);
  g_assert(mOsgCalModel.get() != 0);
  g_assert(mOsgCalModel->getCalCoreModel() != 0);
  g_assert(mOsgCalModel->getCalCoreModel()->getCoreAnimation(id) != 0);
  return mOsgCalModel->getCalCoreModel()->getCoreAnimation(id)->getDuration();
}

double UGAMEAnimatedModel::GetDuration(const std::string& name)
{
  return GetDuration(GetCoreAnimationId(name));
}

int UGAMEAnimatedModel::GetCoreAnimationId(const std::string& name)
{
  g_assert(mOsgCalModel.get() != 0);
  g_assert(mOsgCalModel->getCalCoreModel() != 0);
  int a=mOsgCalModel->getCalCoreModel()->getCoreAnimationId(name);
  if (a<0)
    g_critical("UGAMEAnimatedModel::GetCoreAnimationId id not found for animation %s",name.c_str());
  return a;
}

UGAMEAnimatedModel::Drawables* UGAMEAnimatedModel::GetDrawables(const std::string& materialName)
{
  g_assert(mOsgCalModel.get() != 0);
  return mOsgCalModel->getDrawables(materialName);
}

// Controller

void UGAMEAnimatedController::Init( void ) 
{
  if(!GetModel())
    SetModel(new UGAMEAnimatedModel);
  UGAMEArtefactController::Init();
}
