/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *
 */

#include "mafStdAfx.h"

#ifndef MAF_USE_VS_PCH

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifdef WIN32
#include "config_win32.h"
#endif

#include <maf/vision.h>
#include <maf/maferror.h>

#include <osg/Transform>
#include <osg/Geode>

#endif

class BBVisitor : public osg::NodeVisitor {
public:
  BBVisitor() : osg::NodeVisitor(osg::NodeVisitor::TRAVERSE_ALL_CHILDREN) {
    setNodeMaskOverride(0xffffffff);
  }

  virtual void apply(osg::Transform& node) {
    osg::NodeVisitor::apply((osg::Group&)node);
    if(mBB.valid()) {
      osg::Matrix matrix;
      node.computeLocalToWorldMatrix(matrix, this);
      mBB._min = matrix * mBB._min;
      mBB._max = matrix * mBB._max;
      if(mBB._max < mBB._min) {
	osg::Vec3 tmp;
	tmp = mBB._min;
	mBB._min = mBB._max;
	mBB._max = tmp;
      }
    }
  }

  virtual void apply(osg::Geode& node) {
    unsigned int num_drawables = node.getNumDrawables();
    for(unsigned int i = 0; i < num_drawables; i++) {
      osg::Drawable* drawable = node.getDrawable(i);
      osg::BoundingBox f = drawable->getBound();
      mBB.expandBy(drawable->getBound());
    }
  }

  osg::BoundingBox mBB;
};

osg::BoundingBox GetBoundingBox(osg::Node* node) {
  BBVisitor visitor;
  node->accept(visitor);
  return visitor.mBB;
}

class UnbindNodes : public osg::NodeVisitor {
public:
  UnbindNodes() : osg::NodeVisitor(osg::NodeVisitor::TRAVERSE_ALL_CHILDREN) {
    setNodeMaskOverride(0xffffffff);
  }

  virtual void apply(osg::Node& node) {
    if(node.getUserData() != 0)
      mNodes.push_back(&node);
    traverse(node);
  }

  //
  // Walk through Groups regardless (event those restricted by the Group traverse
  // policy).
  //
  virtual void apply(osg::Group& node) {
    if(node.getUserData() != 0)
      mNodes.push_back(&node);
    for(unsigned int i = 0; i < node.getNumChildren(); i++) {
      apply(*node.getChild(i));
    }
  }

  void clearUserData() {
    for(std::vector<osg::ref_ptr<osg::Node> >::iterator i = mNodes.begin();
	i != mNodes.end();
	i++)
      (*i)->setUserData(0);
  }

  std::vector<osg::ref_ptr<osg::Node> > mNodes;
};


void RecursiveClearUserData(osg::Node* node) {
  //
  // Controllers bound to osg::Node introduce a circular dependency
  // that must be broken at some point. Since the goal of registering
  // the Controllers in osg::Nodes is solely for the purpose of 
  // the picking implemented in MAFSceneModel, get rid of them when 
  // MAFSceneModel is deallocated.
  //
  UnbindNodes unbind;
  node->accept(unbind);
  unbind.clearUserData();
}

class LeakNodes : public osg::NodeVisitor {
public:
  LeakNodes() : osg::NodeVisitor(osg::NodeVisitor::TRAVERSE_ALL_CHILDREN) {
    setNodeMaskOverride(0xffffffff);
    mIndent = 0;
  }

  virtual void apply(osg::Node& node) {
    //    g_debug("LeakNodes: |%*s%s|%s|0x%08x %d", mIndent, " ", node.className(), node.getName().c_str(), (int)&node, node.referenceCount());
    if(node.referenceCount() > 1)
      mNodes.push_back(&node);
    mIndent += 2;
    traverse(node);
    mIndent -= 2;
  }

  std::vector<osg::ref_ptr<osg::Node> > mNodes;
  int mIndent;
};

osg::NodeVisitor* RecursiveLeakCollect(osg::Node* node) {
  LeakNodes* leakNodes = new LeakNodes();
  node->accept(*leakNodes);
  return leakNodes;
}

void RecursiveLeakCheck(osg::NodeVisitor* visitor) {
  LeakNodes* leakNodes = dynamic_cast<LeakNodes*>(visitor);
  for(std::vector<osg::ref_ptr<osg::Node> >::iterator i = leakNodes->mNodes.begin();
      i != leakNodes->mNodes.end();
      i++) {
    if((*i)->referenceCount() > 1) {
      g_debug("RecursiveLeakCheck: reachable: %s|%s|0x%08x %d", (*i)->className(), (*i)->getName().c_str(), (int)i->get(), (*i)->referenceCount());
    }
  }
  delete leakNodes;
}

void MAFVisionController::BindToNode(osg::Node* node) {
  g_assert(node != 0);
  if(node && node->getUserData() != this) {
    node->setUserData(this);
    node->addDescription("MAFController");
  }
}

