/**
   Copyright (C) 2004 Cedric Pinson <cpinson@freesheep.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   ****************************************************************************
   * @file   profile_timer.cpp
   *
   * @brief   Profile system
   *
   *****************************************************************************
   *
   * @author  Cedric Pinson, Based from article on Gems Game programming 3
   *	    Original authors GregHjelstrom and Byon Garrabrant
   *
   * @date    Created 2002/11
   *
   * @version $Id: timer.cpp,v 1.4 2004/10/08 18:43:51 loic Exp $
   *
   ****************************************************************************/

#include "mafStdAfx.h"

#ifndef MAF_USE_VS_PCH

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <maf/maferror.h>
#include <maf/timer.h>

#include <iostream>
#include <cstdio>

#endif


#ifdef WIN32

#include <windows.h>
double GetRealTime()
{
  static double SCALE_COUNTER=-1;

  if (SCALE_COUNTER==-1.0) {
    LARGE_INTEGER pc;

    if(!QueryPerformanceFrequency((LARGE_INTEGER *)&pc))
	{
    //  g_log("No freqency !!!");
    }

    SCALE_COUNTER=1.0/((double)pc.QuadPart);

  }
  //unsigned int perf_cnt = pc.QuadPart;

  //double scale_time = 1.0 / perf_cnt;
  LARGE_INTEGER ct;
  double count;
  if(!QueryPerformanceCounter((LARGE_INTEGER *) &ct))
  {
  //  g_log("No timer !!!");
    return 1;
  }
  count=(double)ct.QuadPart;
  count*=SCALE_COUNTER;
  return count;
}

double GetRealTimeInMS() 
{
  double t=GetRealTime();
  double t2=t*1000.0;
  return t2;
}

#else 

#if defined(__i386__) || defined(__ia64__)

/** Time system from xxx*/
#include <sys/time.h>
#include <inttypes.h>

inline uint64_t TimerGetCycle() {
  uint64_t x;
  __asm__ volatile ( "RDTSC" : "=A" (x) ) ;
  return x;
}

inline uint64_t TimerGetHz()
{
  static struct timeval t1, t2;
  static struct timezone tz;
  uint64_t c1,c2;

  gettimeofday(&t1, &tz);
  c1 = TimerGetCycle();
  for (int i = 0; i < 2500000; i++);
  gettimeofday(&t2, &tz);
  c2 = TimerGetCycle();
  return (1000000 * (c2 - c1)) / ((t2.tv_usec - t1.tv_usec) + 1000000 * (t2.tv_sec - t1.tv_sec));
}


double GetRealTime() 
{
  static double SCALE_COUNTER=-1;
  if (SCALE_COUNTER==-1.0)
    SCALE_COUNTER=1.0/(double)TimerGetHz() ;
  
  return SCALE_COUNTER*(double)TimerGetCycle() ;
}

double GetRealTimeInMS() 
{
  double t=GetRealTime();
  double t2=t*1000.0;
//   printf ("time %f - %f\n",t,t2);
  return t2;
}

#else 

#include <SDL.h>

double GetRealTime()
{
  const double f=1.0/1000.0;
  return SDL_GetTicks()*f;
}

double GetRealTimeInMS()
{
  return SDL_GetTicks();
}

#endif

#endif
