/*
*
* Copyright (C) 2004 Mekensleep
*
*	Mekensleep
*	24 rue vieille du temple
*	75004 Paris
*       licensing@mekensleep.com
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*
* Authors:
*  Igor Kravtchenko <igor@obraz.net>
*
*/

#include "mafStdAfx.h"

#ifndef MAF_USE_VS_PCH
#include <glib.h>

#include <maf/mafexport.h>
#include <osg/CullFace>
#include <osg/TexGen>
#include <osg/PolygonOffset>
#include <osg/MatrixTransform>
#include <osg/Material>
#include <osg/TexEnvCombine>
#include <osgUtil/RenderToTextureStage>
#include <osgUtil/TransformCallback>
#include <osgUtil/CullVisitor>
#endif

//////////////////////////////////////////////////////////////////////////
////////////////////////// 
// SHADOW MAP TECHNIQUE //
//////////////////////////
//////////////////////////////////////////////////////////////////////////

const int depth_texture_height = 1024;
const int depth_texture_width  = 1024;

osg::ref_ptr<osg::RefMatrix> bias = new osg::RefMatrix(	0.5f, 0.0f, 0.0f, 0.0f,
													   0.0f, 0.5f, 0.0f, 0.0f,
													   0.0f, 0.0f, 0.5f, 0.0f,
													   0.5f, 0.5f, 0.5f, 1.0f);

class RenderToTextureCallback: public osg::NodeCallback
{
public:
	RenderToTextureCallback(osg::Node* subgraph, 
		osg::Texture2D* texture, 
		osg::MatrixTransform* light_transform,
		osg::TexGen* tex_gen):
	_subgraph(subgraph),
		_texture(texture),
		_local_stateset(new osg::StateSet),
		_viewport(new osg::Viewport),
		_light_projection(new osg::RefMatrix),
		_light_transform(light_transform),
		_tex_gen(tex_gen)
	{
		_local_stateset->setAttribute(_viewport.get());
		_local_stateset->setMode(GL_LIGHTING, osg::StateAttribute::OFF);

		osg::ref_ptr<osg::PolygonOffset> polygon_offset = new osg::PolygonOffset;
		polygon_offset->setFactor(1.1f);
		polygon_offset->setUnits(2.0f);
		_local_stateset->setAttribute(polygon_offset.get(), osg::StateAttribute::ON | osg::StateAttribute::OVERRIDE);
		_local_stateset->setMode(GL_POLYGON_OFFSET_FILL, osg::StateAttribute::ON | osg::StateAttribute::OVERRIDE);

		osg::ref_ptr<osg::CullFace> cull_face = new osg::CullFace;
		cull_face->setMode(osg::CullFace::FRONT);
		_local_stateset->setAttribute(cull_face.get(), osg::StateAttribute::ON | osg::StateAttribute::OVERRIDE);
		_local_stateset->setMode(GL_CULL_FACE, osg::StateAttribute::ON | osg::StateAttribute::OVERRIDE);

		_viewport->setViewport(0, 0, depth_texture_width, depth_texture_height);
/*
		float znear = 1.0f * _subgraph->getBound().radius();
		float zfar  = 3.0f * _subgraph->getBound().radius();
		float top   = 0.5f * _subgraph->getBound().radius();
		float right = 0.5f * _subgraph->getBound().radius();
		znear *= 0.8f;
		zfar *= 1.2f;
*/

//		float znear = 1.0f; //_subgraph->getBound().radius() * 0.5f;
//		float zfar  = 10000; //_subgraph->getBound().radius() * 50.0f;
//		float top   = 1; //_subgraph->getBound().radius();
//		float right = 1; //_subgraph->getBound().radius();
//		znear *= 0.8f;
//		zfar *= 1.2f;

//		_light_projection->makeFrustum(-right, right, -top, top, znear, zfar);
		_light_projection->set( osg::Matrixd::perspective(90, 1, 500.0f, 700.0f) );
	}

	virtual void operator()(osg::Node* node, osg::NodeVisitor* nv)
	{

		osgUtil::CullVisitor* cullVisitor = dynamic_cast<osgUtil::CullVisitor*>(nv);
		if (cullVisitor && _texture.valid() && _subgraph.valid())
		{            
			_request_render_to_depth_texture(*node, *cullVisitor);
		}

		// must traverse the subgraph            
		traverse(node,nv);
	}

	void _request_render_to_depth_texture(osg::Node& node, osgUtil::CullVisitor& cv);

	osg::ref_ptr<osg::Node>                     _subgraph;
	osg::ref_ptr<osg::Texture2D>                _texture;
	osg::ref_ptr<osg::StateSet>                 _local_stateset;
	osg::ref_ptr<osg::Viewport>                 _viewport;
	osg::ref_ptr<osg::RefMatrix>                     _light_projection;
	osg::ref_ptr<osg::MatrixTransform>               _light_transform;
	osg::ref_ptr<osg::TexGen>                        _tex_gen;
};

void RenderToTextureCallback::_request_render_to_depth_texture(osg::Node&, osgUtil::CullVisitor& cv)
{   
	// create the render to texture stage.
	osg::ref_ptr<osgUtil::RenderToTextureStage> rtts = new osgUtil::RenderToTextureStage;

	// set up lighting.
	// currently ignore lights in the scene graph itself..
	// will do later.
	osgUtil::RenderStage* previous_stage = cv.getCurrentRenderBin()->getStage();

	// set up the background color and clear mask.
	rtts->setClearMask(GL_DEPTH_BUFFER_BIT);
	rtts->setColorMask(new osg::ColorMask(false, false, false, false));

	// set up to charge the same RenderStageLighting is the parent previous stage.
	rtts->setRenderStageLighting(previous_stage->getRenderStageLighting());


	// record the render bin, to be restored after creation
	// of the render to text
	osgUtil::RenderBin* previousRenderBin = cv.getCurrentRenderBin();

	osgUtil::CullVisitor::ComputeNearFarMode saved_compute_near_far_mode = cv.getComputeNearFarMode();
	cv.setComputeNearFarMode(osgUtil::CullVisitor::DO_NOT_COMPUTE_NEAR_FAR);

	// set the current renderbin to be the newly created stage.
	cv.setCurrentRenderBin(rtts.get());

	osg::ref_ptr<osg::RefMatrix> light_view = new osg::RefMatrix;
//	light_view->makeLookAt(_light_transform->getMatrix().getTrans(), osg::Vec3(0, 0, 0), osg::Z_AXIS);
//	light_view->makeLookAt(_light_transform->getMatrix().getTrans(), osg::Vec3(0, 0, 0.01f), osg::Y_AXIS);
	light_view->set( osg::Matrix::inverse( _light_transform.get()->getMatrix() ) );
	osg::Matrix texture_matrix = (*light_view.get()) * (*_light_projection.get()) * (*bias.get());
	_tex_gen->setPlane(osg::TexGen::S, osg::Vec4(texture_matrix(0, 0), 
		texture_matrix(1, 0), 
		texture_matrix(2, 0), 
		texture_matrix(3, 0))); 
	_tex_gen->setPlane(osg::TexGen::T, osg::Vec4(texture_matrix(0, 1), 
		texture_matrix(1, 1), 
		texture_matrix(2, 1), 
		texture_matrix(3, 1))); 
	_tex_gen->setPlane(osg::TexGen::R, osg::Vec4(texture_matrix(0, 2), 
		texture_matrix(1, 2), 
		texture_matrix(2, 2), 
		texture_matrix(3, 2))); 
	_tex_gen->setPlane(osg::TexGen::Q, osg::Vec4(texture_matrix(0, 3), 
		texture_matrix(1, 3), 
		texture_matrix(2, 3), 
		texture_matrix(3, 3))); 

	cv.pushProjectionMatrix(_light_projection.get());
	cv.pushModelViewMatrix(light_view.get());
	cv.pushStateSet(_local_stateset.get());

	// traverse the subgraph
	_subgraph->accept(cv);

	cv.popStateSet();
	cv.popModelViewMatrix();
	cv.popProjectionMatrix();

	cv.setComputeNearFarMode(saved_compute_near_far_mode);

	// restore the previous renderbin.
	cv.setCurrentRenderBin(previousRenderBin);

	if (rtts->getRenderGraphList().size()==0 && rtts->getRenderBinList().size()==0)
	{
		// getting to this point means that all the subgraph has been
		// culled by small feature culling or is beyond LOD ranges.
		return;
	}

	rtts->setViewport(_viewport.get());

	// and the render to texture stage to the current stages
	// dependancy list.
	cv.getCurrentRenderBin()->getStage()->addToDependencyList(rtts.get());

	// if one exist attach texture to the RenderToTextureStage.
	rtts->setTexture(_texture.get());
}


//////////////////////////////////////////////////////////////////////////
////////////////////////////// 
// SHADOW TEXTURE TECHNIQUE //
//////////////////////////////
//////////////////////////////////////////////////////////////////////////


class CreateShadowTextureCullCallback : public osg::NodeCallback
{
public:

	CreateShadowTextureCullCallback(osg::Node* shadower,const osg::Vec3& position, const osg::Vec4& ambientLightColor, unsigned int textureUnit, const osg::BoundingSphere &bs):
	  _shadower(shadower),
		  _position(position),
		  _ambientLightColor(ambientLightColor),
		  _unit(textureUnit),
		  _shadowState(new osg::StateSet),
		  _shadowedState(new osg::StateSet),
		  _bs(bs)
	  {
		  _texture = new osg::Texture2D;
		  _texture->setFilter(osg::Texture2D::MIN_FILTER,osg::Texture2D::LINEAR);
		  _texture->setFilter(osg::Texture2D::MAG_FILTER,osg::Texture2D::LINEAR);
		  _texture->setWrap(osg::Texture2D::WRAP_S,osg::Texture2D::CLAMP_TO_BORDER);
		  _texture->setWrap(osg::Texture2D::WRAP_T,osg::Texture2D::CLAMP_TO_BORDER);
		  _texture->setBorderColor(osg::Vec4(1.0f,0.0f,0.0f,1.0f));
	  }

	  virtual void operator()(osg::Node* node, osg::NodeVisitor* nv)
	  {

		  osgUtil::CullVisitor* cullVisitor = dynamic_cast<osgUtil::CullVisitor*>(nv);
		  if (cullVisitor && (_texture.valid() && _shadower.valid()))
		  {            
			  doPreRender(*node,*cullVisitor);

		  }
		  else
		  {
			  // must traverse the shadower            
			  traverse(node,nv);
		  }
	  }

protected:

	void doPreRender(osg::Node& node, osgUtil::CullVisitor& cv);

	osg::ref_ptr<osg::Node>      _shadower;
	osg::ref_ptr<osg::Texture2D> _texture;
	osg::Vec3                    _position;
	osg::Vec4                    _ambientLightColor;
	unsigned int                 _unit;
	osg::ref_ptr<osg::StateSet>  _shadowState;
	osg::ref_ptr<osg::StateSet>  _shadowedState;
	osg::BoundingSphere			_bs; 
};

void CreateShadowTextureCullCallback::doPreRender(osg::Node& node, osgUtil::CullVisitor& cv)
{   

	const osg::BoundingSphere& bs = _shadower->getBound();
	if (!bs.valid())
	{
		osg::notify(osg::WARN) << "bb invalid"<<_shadower.get()<<std::endl;
		return;
	}


	// create the render to texture stage.
	osg::ref_ptr<osgUtil::RenderToTextureStage> rtts = new osgUtil::RenderToTextureStage;

	// set up lighting.
	// currently ignore lights in the scene graph itself..
	// will do later.
	osgUtil::RenderStage* previous_stage = cv.getCurrentRenderBin()->getStage();

	// set up the background color and clear mask.
	rtts->setClearColor(osg::Vec4(1.0f,0.0f,0.0f,1.0f));
	//rtts->setClearMask(previous_stage->getClearMask());

	// set up to charge the same RenderStageLighting is the parent previous stage.
	rtts->setRenderStageLighting(previous_stage->getRenderStageLighting());


	// record the render bin, to be restored after creation
	// of the render to text
	osgUtil::RenderBin* previousRenderBin = cv.getCurrentRenderBin();

	// set the current renderbin to be the newly created stage.
	cv.setCurrentRenderBin(rtts.get());


	float centerDistance = (_position-bs.center()).length();

	float znear = centerDistance-bs.radius();
	float zfar  = centerDistance+bs.radius();
	float zNearRatio = 0.001f;
	if (znear<zfar*zNearRatio) znear = zfar*zNearRatio;


	// 2:1 aspect ratio as per flag geomtry below.
	float top   = (bs.radius()/centerDistance)*znear;
	float right = top;

	// set up projection.
	osg::RefMatrix* projection = new osg::RefMatrix;
	projection->makeFrustum(-right,right,-top,top,znear,zfar);
	//projection->makeFrustum(-1, 1, -1, 1, 1, 1000);

	cv.pushProjectionMatrix(projection);

	osg::RefMatrix* matrix = new osg::RefMatrix;
//	matrix->makeLookAt(_position,bs.center(),osg::Vec3(0.0f,1.0f,0.0f));
//	osg::Vec3 vec = bs.center() - _position;
//	vec.normalize();
	//matrix->makeLookAt(_position, osg::Vec3(0, 0, 0.0f),osg::Vec3(0.0f,1.0f,0.0f));
	matrix->set( osg::Matrix::identity() );


	osg::Matrix MV = cv.getModelViewMatrix();

	// compute the matrix which takes a vertex from local coords into tex coords
	// will use this later to specify osg::TexGen..
	osg::Matrix MVPT = 
		*matrix * 
		*projection *
		osg::Matrix::translate(1.0,1.0,1.0) *
		osg::Matrix::scale(0.5f,0.5f,0.5f);

	cv.pushModelViewMatrix(matrix);

	// make the material black for a shadow.
	osg::Material* material = new osg::Material;
	material->setAmbient(osg::Material::FRONT_AND_BACK,osg::Vec4(0.0f,0.0f,0.0f,1.0f));
	material->setDiffuse(osg::Material::FRONT_AND_BACK,osg::Vec4(0.0f,0.0f,0.0f,1.0f));
	material->setEmission(osg::Material::FRONT_AND_BACK,_ambientLightColor);
	material->setShininess(osg::Material::FRONT_AND_BACK,0.0f);
	_shadowState->setAttribute(material,osg::StateAttribute::OVERRIDE);

	cv.pushStateSet(_shadowState.get());

	{

		// traverse the shadower
		_shadower->accept(cv);

	}

	cv.popStateSet();

	// restore the previous model view matrix.
	cv.popModelViewMatrix();


	// restore the previous model view matrix.
	cv.popProjectionMatrix();

	// restore the previous renderbin.
	cv.setCurrentRenderBin(previousRenderBin);

	if (rtts->getRenderGraphList().size()==0 && rtts->getRenderBinList().size()==0)
	{
		// getting to this point means that all the shadower has been
		// culled by small feature culling or is beyond LOD ranges.
		return;
	}



	int height = 512;
	int width  = 512;


	const osg::Viewport& viewport = *cv.getViewport();

	// offset the impostor viewport from the center of the main window
	// viewport as often the edges of the viewport might be obscured by
	// other windows, which can cause image/reading writing problems.
	int center_x = viewport.x()+viewport.width()/2;
	int center_y = viewport.y()+viewport.height()/2;

	osg::Viewport* new_viewport = new osg::Viewport;
	new_viewport->setViewport(center_x-width/2,center_y-height/2,width,height);
	rtts->setViewport(new_viewport);

	_shadowState->setAttribute(new_viewport);

	// and the render to texture stage to the current stages
	// dependancy list.
	cv.getCurrentRenderBin()->getStage()->addToDependencyList(rtts.get());

	// if one exist attach texture to the RenderToTextureStage.
	if (_texture.valid()) rtts->setTexture(_texture.get());


	// set up the stateset to decorate the shadower with the shadow texture
	// with the appropriate tex gen coords.
	osg::TexGen* texgen = new osg::TexGen;
	//texgen->setMatrix(MV);
	texgen->setMode(osg::TexGen::EYE_LINEAR);
#if 0
	texgen->setPlanesFromMatrix(MVPT);
#endif
	cv.getRenderStage()->addPositionedTextureAttribute(0,new osg::RefMatrix(MV),texgen);


	_shadowedState->setTextureAttributeAndModes(_unit,_texture.get(),osg::StateAttribute::ON);
	_shadowedState->setTextureAttribute(_unit,texgen);
	_shadowedState->setTextureMode(_unit,GL_TEXTURE_GEN_S,osg::StateAttribute::ON);
	_shadowedState->setTextureMode(_unit,GL_TEXTURE_GEN_T,osg::StateAttribute::ON);
	_shadowedState->setTextureMode(_unit,GL_TEXTURE_GEN_R,osg::StateAttribute::ON);
	_shadowedState->setTextureMode(_unit,GL_TEXTURE_GEN_Q,osg::StateAttribute::ON);

	cv.pushStateSet(_shadowedState.get());

	// must traverse the shadower            
	traverse(&node,&cv);

	cv.popStateSet();    

}




osg::Group* MAFAddShadowFromShadowerAndShadowed(	osg::Node *_shadower,
											osg::Node *_shadowed,
											const osg::Vec3 &_lightPosition,
											int _textureUnit)
{
	g_assert(_shadower);
	g_assert(_shadowed);

//	osg::LightSource* lightgroup = new osg::LightSource;
//	osg::Light *light = new osg::Light;
//	light->setPosition(osg::Vec4(_lightPosition,1.0f));
//	light->setLightNum(0);
//	lightgroup->setLight(light);

	osg::Vec4 ambientLightColor(0.4f, 0.4f, 0.4f, 1.0f);
	// add the shadowed with the callback to generate the shadow texture.

	const osg::BoundingSphere& bs = _shadowed->getBound();

	_shadowed->setCullCallback(new CreateShadowTextureCullCallback(_shadower, _lightPosition, ambientLightColor, _textureUnit, bs));
//	lightgroup->addChild(_shadower);
//	lightgroup->addChild(_shadowed);

//	osg::Geode* lightgeode = new osg::Geode;
//	lightgeode->getOrCreateStateSet()->setMode(GL_LIGHTING,osg::StateAttribute::OFF);
//	lightgeode->addDrawable(new osg::ShapeDrawable(new osg::Sphere(_lightPosition,10)));
//	lightgroup->addChild(lightgeode);

	return 0; //lightgroup;
}


void MAFAddShadowMapTo(	osg::Group *_scene,
						osg::Node *_shadowedScene,
						osg::MatrixTransform *_light_transform,
						int _textureUnit,
						float _shadowAmbient)
{
	osg::ref_ptr<osg::Texture2D> texture = new osg::Texture2D;
	texture->setInternalFormat(GL_DEPTH_COMPONENT);
	texture->setShadowComparison(true);
	texture->setShadowTextureMode(osg::Texture::LUMINANCE);
	texture->setShadowAmbient(_shadowAmbient);

	osg::ref_ptr<osg::TexGen> tex_gen = new osg::TexGen;
	tex_gen->setMode(osg::TexGen::EYE_LINEAR);
	_shadowedScene->getOrCreateStateSet()->setTextureAttributeAndModes(_textureUnit, texture.get(), osg::StateAttribute::ON);
	_shadowedScene->getOrCreateStateSet()->setTextureAttributeAndModes(_textureUnit, tex_gen.get(), osg::StateAttribute::ON);

	_scene->setCullCallback(new RenderToTextureCallback(_shadowedScene, texture.get(), _light_transform, tex_gen.get()));
}

MAF_EXPORT void MAFAddSoftShadowMapTo(	osg::Group *_scene,
										osg::Node *_shadowedScene,
										osg::MatrixTransform *_light_transform,
										osg::Texture *_shadow_reducer)
{
	osg::ref_ptr<osg::Texture2D> texture = new osg::Texture2D;
	texture->setInternalFormat(GL_DEPTH_COMPONENT);
	texture->setShadowComparison(true);
	texture->setShadowTextureMode(osg::Texture::LUMINANCE);

	osg::ref_ptr<osg::TexGen> tex_gen = new osg::TexGen;
	tex_gen->setMode(osg::TexGen::EYE_LINEAR);
	osg::StateSet *ss = _shadowedScene->getOrCreateStateSet();
	ss->setTextureAttributeAndModes(0, texture.get(), osg::StateAttribute::ON);
	ss->setTextureAttributeAndModes(0, tex_gen.get(), osg::StateAttribute::ON);

	osg::TexEnvCombine *combiner;

	combiner = new osg::TexEnvCombine();
	combiner->setCombine_RGB(GL_REPLACE);
	combiner->setSource0_RGB(GL_TEXTURE);
	combiner->setSource0_Alpha(GL_TEXTURE);
	combiner->setOperand0_RGB(GL_SRC_COLOR);
	combiner->setOperand0_Alpha(GL_SRC_ALPHA);
	ss->setTextureAttributeAndModes(0, combiner, osg::StateAttribute::ON);

	//state->setTextureAttributeAndModes(1, _shadow_reducer, osg::StateAttribute::ON);

	_scene->setCullCallback(new RenderToTextureCallback(_shadowedScene, texture.get(), _light_transform, tex_gen.get()));
}
