/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Vincent Caron <zerodeux@gnu.org>
 *
 */

#include "mafStdAfx.h"

#ifndef MAF_USE_VS_PCH

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifdef WIN32
#include "config_win32.h"
#endif

#include <glib.h>

#include <maf/maferror.h>
#include <maf/audio.h>
#include <maf/scene.h>

#include <osgAL/SoundManager>

#endif

void MAFAudioModel::CheckError(const std::string& message) {
  ALenum error = alGetError();
  if(error != AL_NO_ERROR) {
    g_critical("MAFAudioModel::CheckError: Audio <%s> alGetError() = %s", message.c_str(), alGetString(error) );
  }
}

MAFAudioModel::~MAFAudioModel()
{
  if (mSoundState.get())
    mSoundState->releaseSource();
}


void MAFAudioModel::SetPriority(int pri)
{
  mPriority=pri;
}

void MAFAudioModel::SetGain(float gain)
{
  mSoundState->setGain(gain);
  if (mSoundState->hasSource())
    mSoundState->apply();
}

void MAFAudioModel::SetReferenceDistance(float d)
{
  mSoundState->setReferenceDistance(d);
  if (mSoundState->hasSource())
    mSoundState->apply();
}

float MAFAudioModel::GetGain()
{
  return mSoundState->getGain();
}


void MAFAudioModel::SetAmbient(bool a)
{
  mSoundState->setAmbient(a);
  if (mSoundState->hasSource())
    mSoundState->apply();
}


const std::string& MAFAudioModel::GetName()
{
  return mName;
}

void MAFAudioModel::SetName(const std::string& name)
{
  mName=name;
  mSoundState->setName(name);
}

bool MAFAudioModel::PlayEvent(int priority)
{
  if (priority==-1)
    priority=GetPriority();
  return osgAL::SoundManager::instance()->pushSoundEvent(mSoundState.get(),priority);
}

void MAFAudioModel::SetPosition(const osg::Vec3& v)
{
  mSoundState->setPosition(v);
  if (mSoundState->hasSource())
    mSoundState->apply();
}

osg::Vec3 MAFAudioModel::GetPosition()
{
  return mSoundState->getPosition();
}



bool MAFAudioModel::GetRelative()
{
  return mSoundState->getRelative();
}



osg::Vec3 MAFAudioModel::GetDirection()
{
  return mSoundState->getDirection();
}

osg::Vec3 MAFAudioModel::GetVelocity()
{
  osg::Vec3 v;
  return mSoundState->getVelocity(v);
}


bool MAFAudioModel::GetPlaying()
{
  return mSoundState->isActive();
}


void MAFAudioModel::SetStatePlaying(bool b)
{
  mSoundState->setPlay(b);
}


void MAFAudioModel::SetPlaying(bool b)
{
  mSoundState->setPlay(b);
  if (0 && b && !mSoundState->hasSource())
    mSoundState->allocateSource(mPriority);

  if (mSoundState->hasSource())
    mSoundState->apply();
}

int MAFAudioModel::GetPriority()
{
  return mSoundState->getPriority();
}

void MAFAudioModel::SetStopMethod(openalpp::SourceState stop)
{
  mSoundState->setStopMethod(stop);
  if (mSoundState->hasSource())
    mSoundState->apply();
}

MAFAudioModel::MAFAudioModel():mData(0),mPriority(0)
{
  mSoundState=new osgAL::SoundState("audiostate_noname");
  mNode= new osgAL::SoundNode;
  mNode->setSoundState(mSoundState.get());
  SetPlaying(false);
}

void MAFAudioModel::SetRolloff(float d)
{
  mSoundState->setRolloffFactor(d);
  if (mSoundState->hasSource())
    mSoundState->apply();
}

float MAFAudioModel::GetRolloff()
{
  return mSoundState->getRolloffFactor();
}

float MAFAudioModel::GetReferenceDistance()
{
  return mSoundState->getReferenceDistance(0.f);
}


void MAFAudioModel::DumpState()
{
  osg::Vec3 pos=GetPosition();
  g_debug("Sound %s position %f %f %f",mName.c_str(),pos[0],pos[1],pos[2]);
  pos=GetDirection();
  g_debug("Sound %s direction %f %f %f",mName.c_str(),pos[0],pos[1],pos[2]);
  pos=GetVelocity();
  g_debug("Sound %s velocity  %f %f %f",mName.c_str(),pos[0],pos[1],pos[2]);
  g_debug("Sound %s gain %f",mName.c_str(),GetGain());
  g_debug("Sound %s relative %d",mName.c_str(),GetRelative());
  g_debug("Sound %s reference distance %f",mName.c_str(),GetReferenceDistance());
  g_debug("Sound %s rolloff %f",mName.c_str(),GetRolloff());

}

void MAFAudioModel::Init()
{
  // Let the soundstate use the sample we just created
  openalpp::SoundData* soundData = mData->GetSoundData();
  openalpp::Sample* sample = dynamic_cast<openalpp::Sample*>(soundData);
  if(sample) {
    mSoundState->setSample(sample);
  } else {
    openalpp::Stream* stream = dynamic_cast<openalpp::Stream*>(soundData);
    mSoundState->setStream(stream);
  }
  mSoundState->allocateSource(mPriority);
  //  mSoundState->apply();
  // osgAL::SoundManager::instance()->addSoundState(mSoundState.get());
}



/*
 * Controller
 */
void MAFAudioController::Init(void) {
  if(!GetModel())
    SetModel(new MAFAudioModel);
  if(!GetView())
    SetView(new MAFAudioView);
  
  MAFController::Init();
}

void MAFAudioController::BindToScene(MAFSceneController* scene) {
  g_assert(scene != 0);
  scene->GetModel()->mGroup->addChild(GetModel()->GetNode());
}

void MAFAudioController::AttachTo( osg::Group* group )
{
  g_assert(group != 0);
  group->addChild(GetModel()->GetNode());
}


void MAFAudioController::MoveTo( osg::Vec3 position ) {
  GetModel()->SetPosition(position);
}
void MAFAudioController::Play(void) {
  if(!GetModel()->GetPlaying()) {
     GetModel()->SetStopMethod(openalpp::Stopped);
     GetModel()->SetPlaying(true);
  }
}

void MAFAudioController::PlayEvent(void) 
{
  GetModel()->SetStatePlaying(true);
  if(!GetModel()->PlayEvent()) {
    g_debug("Can't play event %s",GetModel()->GetName().c_str());
  }
}


void MAFAudioController::Stop(void) {
  if(GetModel()->GetPlaying()) {
    GetModel()->SetPlaying(false);
  }
}
