/* -*-c++-*- OpenSceneGraph - Copyright (C) 1998-2003 Robert Osfield 
 *
 * This library is open source and may be redistributed and/or modified under  
 * the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or 
 * (at your option) any later version.  The full license is in LICENSE file
 * included with this distribution, and on the openscenegraph.org website.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * OpenSceneGraph Public License for more details.
*/

#include "mafStdAfx.h"

#ifndef MAF_USE_VS_PCH
#include <maf/MultipleAnimationPathCallback.h>
#include <iostream>

#include <osg/AnimationPath>
#include <osg/MatrixTransform>
#include <osg/PositionAttitudeTransform>
#endif

using namespace osg;

class MultipleAnimationPathCallbackVisitor : public osg::NodeVisitor
{
    public:

        MultipleAnimationPathCallbackVisitor(const AnimationPath::ControlPoint& cp, const osg::Vec3d& pivotPoint, bool useInverseMatrix):
            _cp(cp),
            _pivotPoint(pivotPoint),
            _useInverseMatrix(useInverseMatrix) {}

        virtual void apply(MatrixTransform& mt)
        {
            Matrix matrix;
            if (_useInverseMatrix)
                _cp.getInverse(matrix);
            else
                _cp.getMatrix(matrix);
                
            mt.setMatrix(osg::Matrix::translate(-_pivotPoint)*matrix);
        }
        
        virtual void apply(PositionAttitudeTransform& pat)
        {
            if (_useInverseMatrix)
            {
                Matrix matrix;
                _cp.getInverse(matrix);
                pat.setPosition(matrix.getTrans());
                pat.setAttitude(_cp._rotation.inverse());
                pat.setScale(osg::Vec3(1.0f/_cp._scale.x(),1.0f/_cp._scale.y(),1.0f/_cp._scale.z()));
                pat.setPivotPoint(_pivotPoint);
                
            }
            else
            {
                pat.setPosition(_cp._position);
                pat.setAttitude(_cp._rotation);
                pat.setScale(_cp._scale);
                pat.setPivotPoint(_pivotPoint);
            }
        }
        
        AnimationPath::ControlPoint _cp;
        osg::Vec3d _pivotPoint;
        bool _useInverseMatrix;      
};

void MultipleAnimationPathCallback::operator()(Node* node, NodeVisitor* nv)
{
  if (nv->getVisitorType()==NodeVisitor::UPDATE_VISITOR && nv->getFrameStamp()) {
    
    double time = nv->getFrameStamp()->getReferenceTime();
    _latestTime = time;

    if (!_pause) {
//       static int frameNumber=0;
//       frameNumber++;
      // Only update _firstTime the first time, when its value is still DBL_MAX
      if (_firstTime==DBL_MAX) _firstTime = time;
      update(*node);
    }
  }
    
  // must call any nested node callbacks and continue subgraph traversal.
  NodeCallback::traverse(node,nv);
}

double MultipleAnimationPathCallback::getAnimationTime() const
{
  if (_pause)
    return ((_pauseTime-_firstTime)-_timeOffset)*_timeMultiplier;

  return ((_latestTime-_firstTime)-_timeOffset)*_timeMultiplier;
}

void MultipleAnimationPathCallback::update(osg::Node& node)
{
    AnimationPath::ControlPoint cp;
	if (getAnimationPath()->getInterpolatedControlPoint(getAnimationTime(),cp))
    {
        MultipleAnimationPathCallbackVisitor apcv(cp,_pivotPoint,_useInverseMatrix);
        node.accept(apcv);
    }
}


void MultipleAnimationPathCallback::reset()
{
    _firstTime = _latestTime;
    _pauseTime = _latestTime;
}

void MultipleAnimationPathCallback::setPause(bool pause)
{
    if (_pause==pause)
    {
        return;
    }
    
    _pause = pause;
    if (_pause)
    {
        _pauseTime = _latestTime;
    }
    else
    {
        _firstTime += (_latestTime-_pauseTime);
    }
}

