/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *
 */
/* -*-c++-*- OpenSceneGraph - Copyright (C) 1998-2003 Robert Osfield 
 *
 * This library is open source and may be redistributed and/or modified under  
 * the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or 
 * (at your option) any later version.  The full license is in LICENSE file
 * included with this distribution, and on the openscenegraph.org website.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * OpenSceneGraph Public License for more details.
*/

#include <exg/exg.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#ifdef WIN32
#include <winsock.h>
#else
#include <netinet/in.h>
#endif

#include <cassert>
#include <iostream>




#include <exg/exg_object.h>

#include <exg/exg_point.h>
#include <exg/exg_vertex.h>
#include <exg/exg_polygon.h>
#include <exg/exg_mesh.h>
#include <exg/exg_file.h>
#include <exg/exg_material.h>
#include <exg/exg_visitor.h>
#include <fstream>

#include <typeinfo>
#include <memory>

namespace exg
{
  //
  // From python2.3-2.3.3/Objects/floatobject.c
  // See LICENSE.PYTHON for licensing information
  //

  static int double2buffer(double x, unsigned char *p, int littleEndian)
  {
    unsigned char sign;
    int e;
    double f;
    unsigned int fbits;
    int incr = 1;

    if (littleEndian) {
      p += 3;
      incr = -1;
    }

    if (x < 0) {
      sign = 1;
      x = -x;
    }
    else
      sign = 0;

    f = frexp(x, &e);

    /* Normalize f to be in the range [1.0, 2.0) */
    if (0.5 <= f && f < 1.0) {
      f *= 2.0;
      e--;
    }
    else if (f == 0.0)
      e = 0;
    else {
      return -1;
    }

    if (e >= 128)
      goto overflow;
    else if (e < -126) {
      /* Gradual underflow */
      f = ldexp(f, 126 + e);
      e = 0;
    }
    else if (!(e == 0 && f == 0.0)) {
      e += 127;
      f -= 1.0; /* Get rid of leading 1 */
    }

    f *= 8388608.0; /* 2**23 */
    fbits = (unsigned int)(f + 0.5); /* Round */
    assert(fbits <= 8388608);
    if (fbits >> 23) {
      /* The carry propagated out of a string of 23 1 bits. */
      fbits = 0;
      ++e;
      if (e >= 255)
	goto overflow;
    }

    /* First byte */
    *p = (sign << 7) | (e >> 1);
    p += incr;

    /* Second byte */
    *p = (char) (((e & 1) << 7) | (fbits >> 16));
    p += incr;

    /* Third byte */
    *p = (fbits >> 8) & 0xFF;
    p += incr;

    /* Fourth byte */
    *p = fbits & 0xFF;

    /* Done */
    return 0;

  overflow:
    return -1;
  }

  static double buffer2double(const unsigned char *p, int littleEndian) {
    unsigned char sign;
    int e;
    unsigned int f;
    double x;
    int incr = 1;

    if (littleEndian) {
      p += 3;
      incr = -1;
    }

    /* First byte */
    sign = (*p >> 7) & 1;
    e = (*p & 0x7F) << 1;
    p += incr;

    /* Second byte */
    e |= (*p >> 7) & 1;
    f = (*p & 0x7F) << 16;
    p += incr;

    /* Third byte */
    f |= *p << 8;
    p += incr;

    /* Fourth byte */
    f |= *p;

    x = (double)f / 8388608.0;

    /* XXX This sadly ignores Inf/NaN issues */
    if (e == 0)
      e = -126;
    else {
      x += 1.0;
      e -= 127;
    }
    x = ldexp(x, e);

    if (sign)
      x = -x;

    return x;
  }

  static std::auto_ptr<DeleteHandler> s_deleteHandler(0);

  Object::Factory* Object::mFactory = 0;

  Object::Factory& Object::GetFactory(void) {
    if(mFactory == 0) {
      mFactory = new Factory;
      (*mFactory)[Object::MAP_OBJECT_POINTER] = MapObjectPointer::Create;
      (*mFactory)[Object::VECTOR_INT] = VectorInt::Create;
      (*mFactory)[Object::VECTOR_FLOAT] = VectorFloat::Create;
      (*mFactory)[Object::VECTOR_STRING] = VectorString::Create;
      (*mFactory)[Object::VECTOR_OBJECT_POINTER] = VectorObjectPointer::Create;
      (*mFactory)[Object::POINT] = Point::Create;
      (*mFactory)[Object::VERTEX] = Vertex::Create;
      (*mFactory)[Object::POLYGON] = Polygon::Create;
      (*mFactory)[Object::MESH] = Mesh::Create;
      (*mFactory)[Object::FILE] = File::Create;
      (*mFactory)[Object::MATERIAL] = Material::Create;
    }
    return *mFactory;
  }

  void Object::setDeleteHandler(DeleteHandler* handler)
  {
  }

  DeleteHandler* Object::GetDeleteHandler()
  {
    return s_deleteHandler.get();
  }

  Object::~Object()
  {
    if (mRefCount>0)
      {
	//std::cerr <<"Warning: deleting still referenced object "<<this<<" of type '"<<typeid(this).name()<<"'"<<std::endl;
	//std::cerr <<"         the final reference count was "<<mRefCount<<", memory corruption possible."<<std::endl;
      }
  }

  void Save(const float& what, std::ostream& out, IOContext* context) {
//    std::ofstream debug("debug2.txt",std::ios_base::app);
//    debug << "DEBUG: " << what << std::endl;

    unsigned char tmp[4];
    double2buffer(what, tmp, 0); // bigEndian
    out.write((const char*)tmp, 4);
  }

  void Save(const int& what, std::ostream& out, IOContext* context) {
    int value = htonl(what);
    const char* tmp = (const char*)&value;
    out.write(tmp, sizeof(int));
  }

  void Save(const std::string& what, std::ostream& out, IOContext* context) {
    int length = what.size();
    Save(length, out, context);
    out << what;
  }

  void Save(const Object* what, std::ostream& out, IOContext* context) {
    bool context_allocated;
    if(context == 0) {
      context_allocated = true;
      context = new IOContext;
    } else {
      context_allocated = false;
    }
    bool exists;
    int serial;
    if(context->mObject2Serial.find(what) != context->mObject2Serial.end()) {
      exists = true;
      serial = context->mObject2Serial[what];
    } else {
      exists = false;
      serial = context->mSerial++;
    }

    Save(serial, out, context);

    if(!exists) {
      int type = Object::NONE;
      if(what)
        type = what->GetType();
      Save(type, out, context);
      context->mObject2Serial[what] = serial;
      if(type != Object::NONE)
        what->OSave(out, context);
    }
    if(context_allocated)
      delete context;
  }

  void Load(float& what, std::istream& in, IOContext* context) {
    char tmp[4];
    in.read(tmp, 4);
    what = buffer2double((unsigned char*)tmp, 0); // bigEndian
  }

  void Load(int& what, std::istream& in, IOContext* context) {
    int value;
    char* tmp = (char*)&value;
    in.read(tmp, sizeof(int));
    what = ntohl(value);
  }
  
  void Load(std::string& what, std::istream& in, IOContext* context) {
    int length;
    Load(length, in, context);
    char* tmp = (char*)malloc(length);
    in.read(tmp, length);
    what.assign(tmp, length);
    free(tmp);
  }

void Load(Pointer<Object>& what, std::istream& in, IOContext* context)
{
	bool context_allocated;

	if (context == 0) {
		context_allocated = true;
		context = new IOContext;
	}
	else {
      context_allocated = false;
    }

	int serial;
	Load(serial, in, context);

	if (context->mSerial2Object.find(serial) != context->mSerial2Object.end()) {
		what = context->mSerial2Object[serial].Get();
	}
	else {
		int type;
		Load(type, in, context);
		if (type != Object::NONE) {
			Object::Factory& factory = Object::GetFactory();
			assert(factory.find(type) != factory.end());
			what = (*(factory[type]))(); // instance a new object of type "type"
			assert(what.Get());
			assert(what->GetType() == type);
		}
		else {
			what = NULL;
		}

		context->mSerial2Object[serial] = what;
		if(what.Valid())
			what->OLoad(in, context);
    }

	if(context_allocated)
		delete context;
}

void Load(Pointer<Mesh>& what, std::istream& in, IOContext* context)
{
	Pointer<Object> w;
	exg::IOContext contextTmp;
	
	w = what.Get();
    if (!context)
      context = &contextTmp;
    Load(w, in, context);
    Repair(w.Get(), context);
    what = (Mesh*) w.Get();
}

  void Repair(Object* what, IOContext* context) {
    for(std::map<int, Pointer<Object> >::iterator i = context->mSerial2Object.begin();
	i != context->mSerial2Object.end();
	i++) {
      if(i->second == 0)
	continue;
      Object* object = i->second.Get();
      object->ORepair(context);
    }
  }


  VectorFloat* MapObjectPointer::AddPropertyVectorFloat(const std::string& name) 
  { 
    return AddProperty<VectorFloat>(name);
  }

  VectorString* MapObjectPointer::AddPropertyVectorString(const std::string& name) 
  { 
    return AddProperty<VectorString>(name);
  }

  VectorObjectPointer* MapObjectPointer::AddPropertyVectorObjectPointer(const std::string& name) 
  { 
    return AddProperty<VectorObjectPointer>(name);
  }

  MapObjectPointer* MapObjectPointer::AddPropertyMapObjectPointer(const std::string& name) 
  { 
    return AddProperty<MapObjectPointer>(name);
  }

  int Object::indent=0;

  void Object::Accept(Visitor& visitor)
  {
    if(visitor.CheckAndTagNode(*this)) {
      visitor.PushOntoNodePath(this);
      visitor.Apply(*this);
      visitor.PopFromNodePath();
    }
  }

  void Object::Indent(int nb,std::ostream& o) {
    for (int i=0;i<nb;i++)
      o << "\t";
  }

  std::ostream& MapObjectPointer::operator<<(std::ostream& o) 
  {
    for (iterator i=begin();i!=end();i++) {

      Indent(indent,o);
      o << i->first.c_str() << " ";

      if (!(i->second.Valid()))
        o << "(null) " << std::endl;
      else {
        if ( i->second->AsVectorObjectPointer() || 
             i->second->AsMapObjectPointer() ) {
          o << std::endl;
          indent++;
          i->second->operator<<(o);
          indent--;
          //o << std::endl;
        } else {
          i->second->operator<<(o);
          //o << std::endl;
        }
      }
    }
    return o;
  }
  
  std::ostream& VectorObjectPointer::operator<<(std::ostream& o) 
  {
    for (iterator i=begin();i!=end();i++) {
      Indent(indent,o);

      if (!i->Valid())
	o << "(null) " << std::endl;
      else {
	if ( (*i)->AsVectorObjectPointer() || 
	     (*i)->AsMapObjectPointer() ) {
	  o << std::endl;
	  indent++;
	  (*i)->operator<<(o);
	  indent--;
	} else {
	  (*i)->operator<<(o);
	}
      }
    }
    return o;
  }

  std::ostream& VectorFloat::operator<<(std::ostream& o) 
  {
    for (iterator i=begin();i!=end();i++) {
      o << (*i) << " ";
    }
    o << std::endl;
    return o;
  }
    

  std::ostream& VectorString::operator<<(std::ostream& o) 
  {
    for (iterator i=begin();i!=end();i++) {
      o << (*i).c_str() << " ";
    }
    o << std::endl;
    return o;
  }


};


