
#include <osg/Image>
#include <osg/Notify>
#include <osg/Geode>
#include <osg/GL>

#include <osgDB/Registry>
#include <osgDB/FileNameUtils>
#include <osgDB/FileUtils>

#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <sstream>

#include "hdrloader.h"

class ReaderWriterHDR : public osgDB::ReaderWriter
{
public:
	virtual const char* className() { return "HDR Image Reader"; }
	virtual bool acceptsExtension(const std::string &extension) { return osgDB::equalCaseInsensitive(extension, "hdr"); }

	virtual ReadResult readImage(const std::string &_file, const osgDB::ReaderWriter::Options *_opts)
	{
		if (!HDRLoader::isHDRFile(_file.c_str()))
			return ReadResult::FILE_NOT_HANDLED;

		float mul = 1.0f;
		bool bYFlip = false;
		if(_opts) {
			std::istringstream iss(_opts->getOptionString());
			std::string opt;
			while (iss >> opt) {
				if(opt == "RGBMUL") {
					iss >> mul;
				}
				else if(opt == "YFLIP") {
					bYFlip = true;
				}
			}
		}

		HDRLoaderResult res;
		bool ret = HDRLoader::load(_file.c_str(), res);
		if (!ret)
			return ReadResult::FILE_NOT_FOUND;

		int nbPixs = res.width * res.height;
		int nbElements = nbPixs * 3;
		unsigned char *rgb = new unsigned char[ nbElements ];
		unsigned char *tt = rgb;
		float *cols = res.cols;

		for (int i = 0; i < nbElements; i++) {
			float element = *cols++;
			element *= mul;
			if (element < 0) element = 0;
			else if (element > 1) element = 1;
			int intElement = (int) (element * 255.0f);
			*tt++ = intElement;
		}

		delete [] res.cols;

		int pixelFormat = GL_RGB;
		int dataType = GL_UNSIGNED_BYTE;

		osg::Image *img = new osg::Image;
		img->setFileName(_file.c_str());
		img->setImage(	res.width, res.height, 1,
						3,
						pixelFormat,
						dataType,
						(unsigned char*) rgb,
						osg::Image::USE_NEW_DELETE);

		return img;
	}
};

// now register with Registry to instantiate the above
// reader/writer.
osgDB::RegisterReaderWriterProxy<ReaderWriterHDR> g_readerWriter_HDR_Proxy;
