/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Vincent Caron <zerodeux@gnu.org>
 *  Cedric Pinson <cpinson@freesheep.org>
 *
 */

#ifndef maf_application_h 
#define maf_application_h

#ifndef MAF_USE_VS_PCH

#include <osg/ref_ptr>

#include <libxml/tree.h>

#include <string>
#include <map>
#include <list>

#include <maf/mafexport.h>

#endif

#include <osgDB/ReaderWriter>

typedef struct _object PyObject; // forward declaration

#include <maf/controller.h>
#include <maf/scene.h>
#include <maf/cursor.h>

class MAFPacketsModule;
class MAFPacket;
//class MAFController;
class MAFWindow;

class MAF_EXPORT MAFApplication
{
  osg::ref_ptr<MAFCursorController> mCursor;

  
public:
  typedef std::map<std::string,xmlDocPtr> Headers;

  /** @name Application life cycle */
  //@{
  MAFApplication();
  virtual ~MAFApplication();

  virtual void Init(void) = 0;
  virtual void SplashScreen(void) { }

  int  Run(void);
  void Quit(int code = 0);
  //@}

  MAFWindow* GetWindow(bool opengl);

  /** @name Event loop */
  //@{
  void AddController(MAFController* controller);
  void RemoveController(MAFController* controller);

  /*
   * This method should only be called in special cases
   * (picking etc ...), do not use in "normal" controller
   */
  SDL_Event*	GetLastEventIgnoreLocking() const;

  SDL_Event* GetLastEvent(MAFController* controller) const;
  bool	HasEvent() const;

  /* base locking event */
  bool	LockEvent(Uint8 type, MAFController* controller);
  void	UnlockEvent(Uint8 type, MAFController* controller);
  bool	IsLocked(Uint8 type) const;

  bool	LockMouse(MAFController* controller);
  void	UnlockMouse(MAFController* controller);
  bool	IsLockedMouse() const;

  bool	LockKeyboard(MAFController* controller);
  void	UnlockKeyboard(MAFController* controller);
  bool	IsLockedKeyboard() const;

  MAFController* GetFocus(void);
  void SetFocus(MAFController* controller);
  //@}

  void SetHeaders(const Headers& headers) { mHeaders = headers; }
  Headers& GetHeaders(void) { return mHeaders; }

  bool GetAmetista(void) { return mAmetista; }
  void SetAmetista(bool ametista) { mAmetista = ametista; }

  bool IsRunning(void) { return mRunning; } 

  void SetReactor(PyObject* reactor);
  void SetClient(PyObject* reactor);
  void PythonCall(const std::string& method, MAFPacket* packet = NULL) { PythonCall(mClient, method, packet); }
  void PythonCall(PyObject* instance, const std::string& method, MAFPacket* packet = NULL);
  virtual void PythonAccept(PyObject* pypacket) {}
  void CheckReactor(void);
  void SetPacketsModule(const std::string& module);
  MAFPacketsModule* GetPacketsModule(void);
  MAFSceneController* GetScene(void) { return mScene.get(); }
  void SetScene(MAFSceneController* scene) { mScene = scene; }
  virtual osg::Referenced* SearchAnimated(const std::string& name) { return 0; }

  std::list<std::string> HeaderGetList(const std::string& name, const std::string& path);
  void HeaderSet(const std::string& name, const std::string& path, const std::string& value);
  std::string HeaderGet(const std::string& name, const std::string& path);
  typedef std::map<std::string,std::string> Properties;
  Properties HeaderGetProperties(const std::string& name, const std::string& path);
  typedef std::list<std::map<std::string,std::string> > PropertiesList;
  PropertiesList HeaderGetPropertiesList(const std::string& name, const std::string& path);
  double GetDeltaFrame() { return mDeltaFrame;}

  osgDB::ReaderWriter::Options* GetOptions() { return mOptions.get(); }
  void SetOptions(osgDB::ReaderWriter::Options* options) { mOptions = options; }

  void SetCursor(MAFCursorController* cursor);
  MAFCursorController* GetCursor() { return mCursor.get();}

  /** @name User callbacks */
  //@{
  virtual void OnExit(int Code) = 0;
  //@}

  float	GetFPS() const
  {
    return mFPS;
  }

private:
  typedef std::list<osg::ref_ptr<MAFController> > ControllerList;
  typedef MAFController*	EventLockMap[SDL_NUMEVENTS];

  osg::ref_ptr<MAFSceneController> mScene;
  Headers	 mHeaders;
  bool           mRunning;
  int            mExitCode;
  ControllerList mControllers;
  ControllerList mControllersToAdd;
  ControllerList mControllersToRemove;
  bool		 mRunningControllers;
  MAFWindow*     mWindow;
  int            mWindowWidth, mWindowHeight;
  bool           mWindowFullScreen;
  int            mFrameRateLimit;
  SDL_Event      mEvent;
  SDL_Event*     mLastEvent;
  MAFController* mFocus;
  int		 mFd;
  PyObject*	 mClient;
  PyObject*	 mReactor;
  MAFPacketsModule*      mPacketsModule;
  double         mReactorLastCall;
  Uint32		 mFPSCount;
  double         mFPSLastUpdate;
  float          mFPS;
  bool		 mAmetista;
  double         mDeltaFrame;
  osg::ref_ptr<osgDB::ReaderWriter::Options> mOptions;

  int	mMouseX, mMouseY;

  EventLockMap	mEventLockMap;

  void RunControllers(void);
  void ParseDefaultOpts(void);
  void Uninit(void);
};

// to unable event locking debug put _DEBUG_LOCK_EVENT_ before
// including application.h in the appropriate source file.

#ifdef	_DEBUG_LOCK_EVENT_

#	ifdef	__GNUC__

#		define _DEBUG_LOCK_EVENT_FUNCTION_ __PRETTY_FUNCTION__

#	else	// !__GNUC__

#		define _DEBUG_LOCK_EVENT_FUNCTION_ __FUNCTION__

#	endif	// __GNUC__

#	define LOCK_MOUSE(app, ctrl) \
	(g_debug("%s (%s@%d) lock mouse by %p", \
	_DEBUG_LOCK_EVENT_FUNCTION_, __FILE__, __LINE__, ctrl), \
	app->LockMouse(ctrl))
#	define UNLOCK_MOUSE(app, ctrl) \
	(g_debug("%s (%s@%d) unlock mouse by %p", \
	_DEBUG_LOCK_EVENT_FUNCTION_, __FILE__, __LINE__, ctrl), \
	app->UnlockMouse(ctrl))
#	define LOCK_KEYBOARD(app, ctrl) \
	(g_debug("%s (%s@%d) lock keyboard by %p", \
	_DEBUG_LOCK_EVENT_FUNCTION_, __FILE__, __LINE__, ctrl), \
	app->LockKeyboard(ctrl))
#	define UNLOCK_KEYBOARD(app, ctrl) \
	(g_debug("%s (%s@%d) unlock keyboard by %p", \
	_DEBUG_LOCK_EVENT_FUNCTION_, __FILE__, __LINE__, ctrl), \
	app->UnlockKeyboard(ctrl))

#else	// !_DEBUG_LOCK_EVENT_

#	define LOCK_MOUSE(app, ctrl)	app->LockMouse(ctrl)
#	define UNLOCK_MOUSE(app, ctrl)	app->UnlockMouse(ctrl)
#	define LOCK_KEYBOARD(app, ctrl)	app->LockKeyboard(ctrl)
#	define UNLOCK_KEYBOARD(app, ctrl)	app->UnlockKeyboard(ctrl)

#endif	// _DEBUG_LOCK_EVENT_

#endif // maf_application_h
