/* -*-c++-*- OpenSceneGraph - Copyright (C) 1998-2003 Robert Osfield 
 *
 * This library is open source and may be redistributed and/or modified under  
 * the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or 
 * (at your option) any later version.  The full license is in LICENSE file
 * included with this distribution, and on the openscenegraph.org website.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * OpenSceneGraph Public License for more details.
 */

#ifndef exg_nodevisitor_connectivity_h
#define exg_nodevisitor_connectivity_h


#include <exg/exg_visitor.h>
#include <exg/exg_point.h>
#include <exg/exg_polygon.h>
#include <set>

namespace exg
{

  class ConnectivityEdge;


  class ConnectivityPoint
  {
   public:

    ConnectivityPoint(){}
    ConnectivityPoint(const ConnectivityPoint& e) {
      mPoint=e.mPoint;
      mEdges=e.mEdges;
    }

    Point* mPoint;
    std::vector<ConnectivityEdge*> mEdges;
    
  };


  /*
   * Reference a 2 point
   */
  class ConnectivityPolygon;
  class ConnectivityEdge
  {
   public:
    ConnectivityEdge(){}
    ConnectivityEdge(const ConnectivityEdge& e) {
      mPoints[0]=e.mPoints[0];
      mPoints[1]=e.mPoints[1];
      mPolygons=e.mPolygons;      
    }
    
    ConnectivityPoint* mPoints[2];
    std::vector<ConnectivityPolygon*> mPolygons;

    void SetPoints(ConnectivityPoint* p0,ConnectivityPoint* p1) {

      if (p0>p1) {
        ConnectivityPoint* tmp=p0;
        p0=p1;
        p1=tmp;
      }
      mPoints[0]=p0;
      mPoints[1]=p1;
    }

    ConnectivityPoint* GetPoint(int i) { return mPoints[i];}

  };




  struct EdgeCompare
  {
    bool operator()(ConnectivityEdge* s1, ConnectivityEdge* s2) const {
      for (int i=0;i<2;i++)
        if (s1->GetPoint(i)<s2->GetPoint(i))
          return 1;
        else if (s1->GetPoint(i)>s2->GetPoint(i))
          return 0;
      return 0;
    }
  };


  class ConnectivityPolygon
  {

   public:

    Polygon* mRealPolygon;
    std::vector<ConnectivityEdge*> mEdges;

  };


  class PointCompare
  {
   public:
    PointCompare(){}
    bool operator()(const ConnectivityPoint* s1,const ConnectivityPoint* s2) const {
      const float threshold = 1e-3;

      Vector3f p1=s1->mPoint->GetPosition();
      Vector3f p2=s2->mPoint->GetPosition();

      for (int i=0;i<3;i++)
        if (p2[i]-p1[i]>threshold)
          return 1;
        else return 0;
      return 0;
    }
  };



  /**
   *  Build edge connectivity to use later for other visitor
   */
  class VisitorBuildEdgeConnectivity : public Visitor
  {

   public:

    
    VisitorBuildEdgeConnectivity():Visitor(ONCE){
      mPoints.clear();
    }


    typedef std::set<ConnectivityEdge*,EdgeCompare> SetEdgeContainer;
    typedef std::set<ConnectivityPoint*,PointCompare> SetPointContainer;
    typedef std::map<Polygon*,ConnectivityPolygon*> MapPolygonContainer;
     SetPointContainer mPoints;
    SetEdgeContainer mEdges;
    MapPolygonContainer mPolygons;


    template <class T,class T2> T2* FindOrInsertElement(T& container,T2& element) {
      typename T::iterator i=container.find(&element);
      if (i!=container.end()) {
        return *i;
      }
      T2* e=new T2(element);
      if (!container.insert(e).second)
        assert(0 && "not possible debug");
      return e;
    }


    void Apply(Polygon& node) {
      
      ConnectivityPolygon* cp=new ConnectivityPolygon;
      cp->mRealPolygon=&node;
      mPolygons[&node]=cp;

      int nb=node.GetNbVertexes();
      ConnectivityEdge tmpEdge;
      ConnectivityPoint tmpPoint;
      ConnectivityEdge* resultEdge=0;
      ConnectivityPoint* resultPoint[2];

      for (int i=0;i<nb;i++) {

        tmpPoint.mPoint=node.GetVertex(i)->GetPoint();
        resultPoint[0]=FindOrInsertElement(mPoints,tmpPoint);
        tmpPoint.mPoint=node.GetVertex((i+1)%nb)->GetPoint();
        resultPoint[1]=FindOrInsertElement(mPoints,tmpPoint);

        tmpEdge.SetPoints(resultPoint[0],resultPoint[1]);
        resultEdge=FindOrInsertElement(mEdges,tmpEdge);

        resultPoint[0]->mEdges.push_back(resultEdge);
        resultPoint[1]->mEdges.push_back(resultEdge);

        cp->mEdges.push_back(resultEdge);
        resultEdge->mPolygons.push_back(cp);
      }
    }
    
  };


}
#endif 
