/**
    Copyright (C) 2004 Cedric Pinson <cpinson@freesheep.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 ****************************************************************************
 * @file   exg_vertex.h
 *
 * @brief   base for vertex exchange
 *
 *****************************************************************************
 *
 * @author  Cedric Pinson
 *
 * @date    Created 2001/04
 *
 * @version $Id: exg_vertex.h,v 1.18 2004/04/28 10:37:19 loic Exp $
 *
 ****************************************************************************/

#ifndef exg_vertex_h
#define exg_vertex_h

#include <exg/exg_basetypes.h>
#include "exg/exg_point.h"

namespace exg
{


  
  /**
   *  Class that describes a Vertex.
   *
   *  reservered keywords:
   *
   *  "point"    PointerObject() that point to a Point() @see Point
   *  "normal"   VectorFloat(3) Should exist if the field "normal" exist 
   *             in the material
   *
   *  unreserved keywords but could be here:
   *
   *  if there are entries in material["maps"] then we should have a channel
   *  in vertexes like this:
   *  "channel"  a VectorFloat(X) of data. Where channel is a name from 
   *             material["maps"][i]["name"] and X material["maps"][i]["element"]
   *
   */
  class EXG_EXPORT Vertex : public MapObjectPointer
  {
   protected:
    virtual ~Vertex() {}

   public:

    COMMON_FUNCTIONS(Vertex);

    /// Default constructor
    Vertex() {
      (*this)["point"]=0;
    }



    /** 
     *  Set the point3d to vertex
     *
     *  @param const Point* point to assign
     */
     void SetPoint(Point* point) {
      (*this)["point"]=point;
    }


    /**
     *  Return the point 3d of vertex
     *
     *  @return Pointer<Point> point
     */
     Point* GetPoint() {
      return (Point*)(*this)["point"].Get();
    }


    /**
     *  Return the point 3d of vertex
     *
     *  @return Pointer<Point> point
     */
     const Point* GetPoint() const {
      return (const Point*)(*find("point")).second.Get();
    }


    void Traverse(Visitor& visitor);

    virtual Vertex* AsVertex(void) { return this; }
    virtual const Vertex* AsVertex(void) const { return this; }
    virtual int GetType(void) const { return VERTEX; }
    inline static Object* Create() { return new Vertex; }

    std::ostream& operator<<(std::ostream& o);
  };


}
#endif
