/**
    Copyright (C) 2004 Cedric Pinson <cpinson@freesheep.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 ****************************************************************************
 * @file   exg_matrix4.h
 *
 * @brief   matrix 4x4
 *
 *****************************************************************************
 *
 * @author  psc80
 *
 * @date    Created 2001/08
 *
 * @version $Id: exg_matrix4.h,v 1.5 2004/06/15 18:51:35 cpinson Exp $
 *
 ****************************************************************************/

#ifndef EXG_MATRIX4_H
#define EXG_MATRIX4_H

#include "exg/exg_vector4.h"
#include "exg/exg_matrix3.h"
#include "exg/exg_vector3.h"

namespace exg {

  /**
   * Matrix 4x4 
   */
  template <class T> class Matrix4_T {

    /** Representation
     *
     *  v[0][0] v[0][1] v[0][2] v[0][3]
     *  v[1][0] v[1][1] v[1][2] v[1][3]
     *  v[2][0] v[2][1] v[2][2] v[2][3]
     *  v[3][0] v[3][1] v[3][2] v[3][3]
     *
     */


   protected:

    Vector4_T<T> v[4];


   public:

    /// Default constructor
    Matrix4_T<T>() {}


    /// Constructor with matrix
    template <class C> explicit Matrix4_T(const Matrix4_T<C>& _m) { 
      v[0]=(Vector4_T<T>) _m[0];
      v[1]=(Vector4_T<T>) _m[1];
      v[2]=(Vector4_T<T>) _m[2];
      v[3]=(Vector4_T<T>) _m[3]; }



    /// Constructor from four vectors
    inline Matrix4_T(const Vector4_T<T>& _v1, 
                     const Vector4_T<T>& _v2, 
                     const Vector4_T<T>& _v3, 
                     const Vector4_T<T>& _v4) {
      Init(_v1, _v2, _v3, _v4); }



    /// Build from a 3d matrix with a translation
    Matrix4_T(const Matrix3_T<T>& _m, 
              const Vector3_T<T>& _o=Vector3_T<T>(0,0,0)) { Init(_m, _o); }



    /// Constructor for euler angles
    Matrix4_T(const T _h, const T _p, const T _r) {
      SetRotationFromEuler(_h, _p, _r); }



    /// Constructor for a rotation with respect to a (normalized) axis
    Matrix4_T(T angle, const Vector3_T<T>& ax) {
      SetRotationFromAxis(angle, ax); }


    inline bool IsIdentity() const {
      Matrix4_T m=Identity();
      return !memcmp(&v,&m.v,sizeof(m));
    }


    /// Initialize a matrix from four vectors
    inline void Init(const Vector4_T<T>& _v1, 
                     const Vector4_T<T>& _v2, 
                     const Vector4_T<T>& _v3, 
                     const Vector4_T<T>& _v4) {
      v[0]=_v1; v[1]=_v2; v[2]=_v3; v[3]=_v4;  }



    /// Set the matrix to zero
    inline void Init() { v[0].Init(); v[1].Init(); v[2].Init();  v[3].Init(); }


    /// Build from a 3d matrix and a vector 3
    inline void Init(const Matrix3_T<T>& _m,
                     const Vector3_T<T>& _v=Vector3_T<T>(0,0,0)) { 
      v[0].Init(_m[0][0],_m[0][1],_m[0][2],_v[0]);
      v[1].Init(_m[1][0],_m[1][1],_m[1][2],_v[1]);
      v[2].Init(_m[2][0],_m[2][1],_m[2][2],_v[2]);
      v[3].Init(0,0,0,1);}





    /// Return matrix identity
    inline static Matrix4_T Identity() { 
      Matrix4_T a;
      a.SetIdentity();
      return a;
    }


    /// Return matrix translation
    inline static Matrix4_T Translation(const Vector3_T<T>& _v) { 
      Matrix4_T a;
      a.SetIdentity();
      a.SetTranslation(_v);
      return a;
    }


    /// Return matrix rotation
    inline static Matrix4_T Rotation(T angle, const Vector3_T<T>& ax) { 
      Matrix4_T a;
      a.SetRotationFromAxis(angle,ax);
      return a;
    }


    /// Return matrix rotation from euler
    inline static Matrix4_T Rotation(const T h, const T p, const T r) { 
      Matrix4_T a;
      a.SetRotationFromEuler(h,p,r);
      return a;
    }
    

    /// Set the matrix to identity
    inline void SetIdentity() {
      v[0].Init(1,0,0,0); 
      v[1].Init(0,1,0,0); 
      v[2].Init(0,0,1,0);
      v[3].Init(0,0,0,1);
    }




    /// Coordinates accessor operator
    inline Vector4_T<T>& operator [] (int i) { return v[i]; }

    /// Coordinates accessor operator
    inline const Vector4_T<T>& operator [] (int i) const { return v[i]; }


    /// Addtion of two matrixes
    inline Matrix4_T operator + (const Matrix4_T& _m) const {
      return Matrix4_T(v[0]+_m.v[0], v[1]+_m.v[1], v[2]+_m.v[2], v[3]+_m.v[3]); }


    /// Division by a scalar
    inline Matrix4_T operator / (const T a) const {
      return Matrix4_T(v[0]/a, v[1]/a, v[2]/a, v[3]/a); }


    /// Operator +=
    inline const Matrix4_T& operator += (const Matrix4_T& m) {
      v[0] += m.v[0]; v[1] += m.v[1]; v[2] += m.v[2]; v[3] += m.v[3]; return *this; }



    /// Mutliplication by a salar value
    inline Matrix4_T operator * (T a) const {
      return Matrix4_T(a*v[0], a*v[1], a*v[2], a*v[3]);
    }



    /// Mutliplication by a vector (vector to the right)
    inline Vector4_T<T> operator * (const Vector4_T<T>& iv) const {
      return Vector4_T<T>( v[0].Dot(iv), v[1].Dot(iv) , v[2].Dot(iv) , v[3].Dot(iv)) ;
    }


    /// Multiplication by a 3D vector extended to 4 and trucated at end
    inline Vector3_T<T> operator * (const Vector3_T<T>& iv) const {
      Vector4_T<T> t(iv,1.0);
      return Vector3_T<T>( v[0].Dot(t) , v[1].Dot(t) , v[2].Dot(t));
    }


    /// Mutliplication by a matrix
    inline Matrix4_T operator * (const Matrix4_T& m) const {
      Matrix4_T<T> col(m) ;
      col.Transpose() ;
    
      Matrix4_T<T> res;
      res[0].Init(v[0].Dot(col[0]), v[0].Dot(col[1]), v[0].Dot(col[2]), v[0].Dot(col[3])) ;
      res[1].Init(v[1].Dot(col[0]), v[1].Dot(col[1]), v[1].Dot(col[2]), v[1].Dot(col[3])) ;
      res[2].Init(v[2].Dot(col[0]), v[2].Dot(col[1]), v[2].Dot(col[2]), v[2].Dot(col[3])) ;
      res[3].Init(v[3].Dot(col[0]), v[3].Dot(col[1]), v[3].Dot(col[2]), v[3].Dot(col[3])) ;
      return res;
    }


    /// Transpose the matrix
    void Transpose() { 
      T t;
      t = v[0][1]; v[0][1] = v[1][0]; v[1][0] = t;
      t = v[0][2]; v[0][2] = v[2][0]; v[2][0] = t;
      t = v[2][1]; v[2][1] = v[1][2]; v[1][2] = t;
      t = v[3][1]; v[3][1] = v[1][3]; v[1][3] = t;
      t = v[3][2]; v[3][2] = v[2][3]; v[2][3] = t;
      t = v[3][0]; v[3][0] = v[0][3]; v[0][3] = t;}


    /// Build a rotation with respect to a (normalized !) axis
    void SetRotationFromAxis(T angle, const Vector3_T<T>& ax) { Init(Matrix3_T<T>(angle, ax)); }




    /// Build a rotation from euler angles
    void SetRotationFromEuler(const T h, const T p, const T r) {   Init(Matrix3_T<T>(h, p, r)); }



    /// Get a Matrix 3x3 for rotation
    void GetTransform3x3(Matrix3_T<T> &m) {
      m[0].Init(Vector3_T<T>(v[0][0],v[0][1],v[0][2]));
      m[1].Init(Vector3_T<T>(v[1][0],v[1][1],v[1][2]));
      m[2].Init(Vector3_T<T>(v[2][0],v[2][1],v[2][2]));
    }

    /// Set a Matrix 3x3 for rotation
    inline void SetTransform3x3(const Matrix3_T<T> &m) {
      v[0].Init(m[0][0],m[0][1],m[0][2],v[0][3]);
      v[1].Init(m[1][0],m[1][1],m[1][2],v[1][3]);
      v[2].Init(m[2][0],m[2][1],m[2][2],v[2][3]);
      v[3].Init(0,0,0,1);
    }


    /// Get Translation
    inline void GetTranslation(Vector3_T<T> &m) {
      m[0]=v[0][3];
      m[1]=v[1][3];
      m[2]=v[2][3];
    }

    /// Set the matrix to a 3d translation
    inline void SetTranslation(const Vector3_T<T>& _o) {
      v[0][3]=_o[0]; 
      v[1][3]=_o[1];
      v[2][3]=_o[2];
      v[3][3]=1;}


    /// Inverse transform
    inline void InverseTransform() {
      Vector3_T<T> a;
      Matrix3_T<T> b;
      GetTranslation(a);
      GetTransform3x3(b);
      b.SetTranspose();
      Init(b,-(b*a));}

  };



  /// Mutliplication by a salar value
  template <class T> inline Matrix4_T<T> operator * (T a, const Matrix4_T<T>& m) {
    return Matrix4_T<T>(a*m.v[0], a*m.v[1], a*m.v[2], a*m.v[3]);
  }

  typedef Matrix4_T<float> Matrix4f ;
  typedef Matrix4_T<double> Matrix4d ;
}

#endif
