from xmlutils import *
from elements import Iq
import dataforms as df
import tsocksv5 as socksv5
import utils
from twisted.internet import defer, reactor

import time, sha, os

ns_si='http://jabber.org/protocol/si'
ns_disco_info='http://jabber.org/protocol/disco#info'
ns_data='jabber:x:data'
ns_xmpp_params='urn:ietf:params:xml:ns:xmpp-stanzas'
ns_features="http://jabber.org/protocol/feature-neg"

feat_bytestream="http://jabber.org/protocol/bytestreams"
feat_url_data="http://jabber.org/protocol/url-data"
feat_ibb="http://jabber.org/protocol/ibb"

profile_filetransfer="http://jabber.org/protocol/si/profile/file-transfer"

# XXX WARNING the interfaces of these classes will likely change in the next 
# releases, since there still many undefined aspects.
# These stuctures aims at building a modular framework for both stream 
# initiation and file transfer methods that be dynamically plugged into the 
# client and used without further configuration. Since this is not completely
# achieved, expect several adjustements.

# TODO
# better error handling
# support for proxies
# in band bytestreams and direct urls

class SIIq(Iq):
    """ Wrapper for SI Iqs """    
    def __init__(self, iq=None, **kw):
        if iq:
            Iq.__init__(self, element=iq.getDOM())
            si=iq.getChildNS('si', ns_si)
        else:
            Iq.__init__(self, **kw)
            si=None

        if not si:
            si=ezel('si', xmlns=ns_si).dom()
            self._el.appendChild(si)
        self._si=si
        feature=get_child(si, 'feature')
        if feature: self._features=df.DataForm(element=get_child(feature,'x'))
        else: self._features=None
        
    def getMimeType(self):
        return self._si.getAttribute('mime-type') or 'binary/octect-stream'
    
    def setMimeType(self, value):
        return self._si.setAttribute('mime-type', value) 
    
    def getStreamId(self):
        return self._si.getAttribute('id')
        
    def setStreamId(self, value):
        return self._si.setAttribute('id', value)
    
    def getProfile(self):
        name=self._si.getAttribute('profile')
        #for c in self._si.childNodes:
        #    if c.namespaceURI == name:
        #        return c
        return profileBuilder(name, self._si)
    
    def setProfile(self, profile):
        si=self._si
        si.setAttribute('profile', profile.profile_name)
        c=get_child(si, profile.profile_name)
        if c: si.removeNode(c)
        si.appendChild(profile.dom())
        
    def getFeatureOffer(self):
        return self._features.data['stream-method'].getOptionValues()
    
    def getFeatureAnswer(self):
        return self._features.data['stream-method'].getValues()[0]
        
    def addFeature(self, value):
        if not self._features: 
            si=self._si
            feature=get_child(si, 'feature')
            if not feature:
                feature=ezel('feature', xmlns=ns_features).dom()
                si.appendChild(feature)
            self._features=df.DataForm(type='form')
            feature.appendChild(self._features.dom())
        self._features.data.addField(var='stream-method', type='list-single', options=[('',value)])
    
    def addResult(self, value):
        if not self._features: 
            si=self._si
            feature=get_child(si, 'feature')
            if not feature:
                feature=ezel('feature', xmlns=ns_features).dom()
                si.appendChild(feature)
            self._features=df.DataForm(type='submit')
            feature.appendChild(self._features.dom())
        self._features.data.addField(var='stream-method', values=[value])
        
    def removeFeature(self, value):
        raise NotImplemented # it should not be useful
   
    def makeResultFrom(self):
        offer=SIIq(Iq.makeResultFrom(self))
        offer.setStreamId(self.getStreamId())
        feature=ezel('feature', xmlns=ns_features).dom()
        self._si.appendChild(feature)
        self._features=df.DataForm(type='submit')
        feature.appendChild(self._features.dom())
        return offer

class StreamInitiation:
    """ Handling JEP 95: Stream Initiation """
    
    def __init__(self, xmppstream):#, profiles={}):
        """ Init with the supported stream methods """
        #self._profiles=profiles #store the supported profiles
        #self._offers={} #we need to store offers yes!
        self._xmppstream=xmppstream
        xmppstream.stream_initiation=self
        # hook the SI IQ
        xmppstream.setIqHandler(ns_si, 'si', self.gotSIOffer)
        # Register the feature for DISCO
        #if hasattr(xmppstream, 'registerFeature'):
        #the stream must support disco 
        xmppstream.registerFeature(ns_si, self)
            #for p,h in profiles:
            #    xmppstream.registerFeature(p)    
    
    def sendSIOffer(self, to, mime_type='', profile=None, features=[]):
        """ Send an offer of stream initiation; the SID is authomatically generated """
        # create the offer
        offer=SIIq(to=to, ttype='set')
        sid=sha.new(`time.time()`).hexdigest()[:10]
        offer.setStreamId(sid)
        if mime_type: offer.setMimeType(mime_type)
        if profile: offer.setProfile(profile)
        # add the features
        for m in features:
            offer.addFeature(m)
            
        d=self._xmppstream.sendIq(offer)
        d.addCallback(self.gotSIOfferResult, sid)
        d.addErrback(self.gotSIOfferError)
        return d
    
    def gotSIOfferResult(self, iq, sid):
        offer=SIIq(iq)
        #if offer.getStreamId()==sid: return offer
        #else: raise ValueError
        offer.setStreamId(sid) # set the stream id 
        return offer
        
    def gotSIOfferError(self, failure):
        return failure
    
    def gotSIOffer(self, iq):
        """ Got an offer of a stream initiation """
        if not iq.getType()=='set': return # only iq of type 'set'?
        
        offer=SIIq(iq=iq)
        #self._offers[offer.sid]=offer #XXX necessary?
        #try to find a handler
        p=offer.getProfile()
        if p:
            h=self._xmppstream.getFeatureHandler(p.profile_name)
            getattr(h, p.offer_handler_name)(offer)
        else:
            self.declineSIOffer(iq, 'bad profile',
                text='Offer declined, no suitable profile handler found')
        
    def acceptSIOffer(self, offer, feature, profile=None):
        """ """
        answer=offer.makeResultFrom()
        if profile: answer.setProfile()
        answer.addResult(feature)
        self._xmppstream.sendIq(answer)
        
    def declineSIOffer(self, offer, type, text=''):
        """ """
        e=self.makeSIError(type, text)
        answer=Iq(id=offer.getID(), to=offer.getFrom(), ttype='error')
        answer.appendChild(e)
        self._xmppstream.sendIq(answer)

    def makeSIError(self, type, text=''):
        """ utility method to make SI errors """
        error=ezel('error', type='cancel')
        if type=='no valid streams':
            error['code']='400'
            error.add('bad-request', xmlns=ns_xmpp_params)
            error.add('no-valid-streams', xmlns=ns_si)
        elif type=='bad profile':
            error['code']='400'
            error.add('bad-request', xmlns=ns_xmpp_params)
            error.add('bad-profile', xmlns=ns_si)
        elif type=='reject':
            error['code']='403'
            error.add('forbidden', xmlns=ns_xmpp_params)
        else: raise ValueError
        if text: error.add('text', xmlns=ns_xmpp_params).text(text)
        return error.dom()
        
    def sendStanza(self, stanza):
        self._xmppstream.sendStanza(stanza)

# File Transfer requires Stream Initiation and one or more file transfer
# methods 
class FileTransfer:
    
    profile_name=profile_filetransfer
    
    # XXX the method parameter should be removed, but discovered
    def __init__(self, xmppstream=None, methods=[feat_bytestream],
        downloaddir="download", port=8321):
        
        self._methods=methods # file transfer methods
        self.downloaddir=downloaddir
        if not xmppstream: xmppstream=self
        self._xmppstream=xmppstream
        
        # Add required extensions if necessary
        if not xmppstream.hasFeature(feat_bytestream):
            e=SOCKSv5ByteStream(xmppstream, port=port)
            setattr(xmppstream, 's5_bytestream', e)
        
        if not xmppstream.hasFeature(ns_si):
            e=StreamInitiation(xmppstream)
            setattr(xmppstream, 'stream_initiation', e)
            
        # register with disco
        self._xmppstream.registerFeature(profile_filetransfer, self)
        
        
    def selectFileTransferMethod(self, offer):
        features=offer.getFeatureOffer()
        for m in self._methods:
            if m in features: return m
    
    def gotFileOffer(self, offer):
        """ Received an offer of file transfer.
        Default behavior: if a valid method is found a file is accepted  """
        if hasattr(self._xmppstream, 'gotFileOffer'):
            self._xmppstream.gotFileOffer(offer) # XXX offer o parameters
        else: # default beahvior
            self.autoAccept(offer)
            
    def autoAccept(self, offer):
            m=self.selectFileTransferMethod(offer)
            if m:
                return self.acceptFileOffer(offer, m)
            else: #no method found
                return self.declineFileOffer(offer, 'no valid streams')
                
    def declineFileOffer(self, offer, type, reason=''):
        """ """
        si=self._xmppstream.getFeatureHandler(ns_si)
        si.declineSIOffer(offer, type, reason)
        d=defer.Deferred()
        d.errback('no valid stream found')
        
    def acceptFileOffer(self, offer, method, filename=None):
        """ accept an offer """
        stream=self._xmppstream
        handler=stream.getFeatureHandler(method)
        sid=offer.getStreamId()
        if not filename:
            filename=os.path.join(self.downloaddir,offer.getProfile().getName())
            
        d=handler.waitForFile(sid=sid, filename=filename)
        #d.addCallback(self.fileReceived, offer)
        #d.addErrback(self.fileReceptionError, offer)
        si=stream.getFeatureHandler(ns_si)
        si.acceptSIOffer(offer, method)
        return d
    
    # XXX these two callbacks are unuseful
    #def fileReceived(self, file, offer):
    #    """ File received """
    #    return file 
    # 
    #def fileReceptionError(self, failure, offer):
    #    """ Error while receiving a file """
    #    return failure

    def sendFile(self, to, path, mime_type='', makehash=0):
        """ start the file sending process.
        returns a deffered called when the file is sent """
        # get file stats and build the profile element
        stats=os.stat(path)
        folder,name=os.path.split(path)
        # XXX set time and hash
        profile=FileProfile(name=name, size=stats.st_size)
        
        # send the offer
        d_sent=defer.Deferred()
        si=self._xmppstream.getFeatureHandler(ns_si)
        d=si.sendSIOffer(to, mime_type, profile, self._methods)
        d.addCallback(self._gotFileOfferResult, path, d_sent)
        d.addErrback(self._gotFileOfferError, d_sent)
        return d_sent
        
    def _gotFileOfferResult(self, offer, path, d_sent):
        """ activate the stream """
        #get the selected feature
        feature=offer.getFeatureAnswer()
        handler=self._xmppstream.getFeatureHandler(feature)
        sid=offer.getStreamId()
        d=handler.sendFile(str(offer.getFrom()), path, sid=sid)
        d.chainDeferred(d_sent)
        
    def _gotFileOfferError(self, offer, d_sent):    
        """ """
        #XXX handle properly the exception
        d_sent.errback('error in offer')

class FileProfile:
    profile_name=profile_filetransfer
    offer_handler_name='gotFileOffer'
    element_name='file'
    
    def __init__(self, name='', size=0, date='', hash='', element=None):
        """ """
        if element: self._el=element
        else: self._el=ezel(self.element_name, xmlns=profile_filetransfer).dom()
        
        if name: self.setName(name)
        if size: self.setSize("%d"%(size,))
        if date: self.setDate(date)
        if hash: self.set(hash)
   
    def getName(self): return self._el.getAttribute('name')
    def setName(self, name): self._el.setAttribute('name', name)
    
    def getSize(self): return self._el.getAttribute('size')
    def setSize(self, size): self._el.setAttribute('size', size)

    def getDate(self): return self._el.getAttribute('date')
    def setDate(self, date): self._el.setAttribute('date', date)

    def getHash(self): return self._el.getAttribute('hash')
    def setHash(self, hash): self._el.setAttribute('hash', hash)
    
    def setRange(self, offset=-1, length=-1):
        el=self._el
        range=get_child(el, 'range')
        if offset==-1 and length==-1:
            el.removeChild(range)
            return
        if not range:
            range=ezel('range')
            el.appendChild(range.dom())
        else: range=ezel(range)
        if offset>=0: range['offset']="%d"%(offset,)
        if length>=0: range['length']="%d"%(length,)
    
    def getRange(self):
        el=self._el
        range=get_child(el, 'range')
        if not range: return (0,-1)
        offset=int(el.getAttribute('offset') or '0')
        length=int(el.getAttribute('length') or '-1')
        return (offset, length)
    
    def dom(self): return self._el

supported_profiles={ profile_filetransfer : FileProfile,
    }

def profileBuilder(name, element):
    p=supported_profiles.get(name, None)
    if not p: return None
    else: return p(element=get_childNS(element, p.profile_name, p.element_name))

class SOCKSv5ByteStream:
    
    def __init__(self, xmppstream, port=8321):
        self.port=port
        self._xmppstream=xmppstream
        #register our iq handler
        xmppstream.setIqHandler(feat_bytestream, 'query', self.gotStreamHost)
        #register with disco
        xmppstream.registerFeature(feat_bytestream, self)
        self._infiles={} #files we are waiting for
        self._outifiles={} #files we are sending
        #start the socksv5 server
        self._s5_server=socksv5.S5ServerFactory()
        reactor.listenTCP(port, self._s5_server)
        
    def waitForFile(self, sid, filename):
        """ sid: stream id
            jid: jid of the offerer
            filepath: file
            returns a callback fired when the file has been received
        """
        d=defer.Deferred()
        self._infiles[sid]=(d, filename)
        return d 
    
    def sendFile(self, to, filepath, sid):
        """ start sending the file through SOCKSv5"""
        stream=self._xmppstream
        d=self._s5_server.queueFile(filepath, sid, str(stream.getJID()), str(to))
        self.sendStreamHost(to, sid)
        return d
    
    def sendStreamHost(self, to, sid, jid=''):
        """ send streamhost data """
        stream=self._xmppstream
        iq=Iq(to=to, ttype='set')
        q=ezel(iq.setQueryNS(feat_bytestream))
        q['sid']=sid
        host=stream.transport.addr[0]
        q.add('streamhost', host=host, port=`self.port`, jid=jid or
            str(stream.getJID()))
        d=stream.sendIq(iq)
        d.addCallback(self.gotStreamHostUsed, sid)
    
    def gotStreamHost(self, iq):
        """ """
        #import pdb; pdb.set_trace()
        q=iq.getQueryNode()
        sid=q.getAttribute('sid')
        streamhosts=[]
        query=iq.getQueryNode()
        for c in get_children(query, 'streamhost'):
            streamhosts.append((c.getAttribute('host'), c.getAttribute('port'),
                c.getAttribute('jid')))
        if self._infiles.has_key(sid):
            #try the streamhosts in order
            self._downloadFile(iq, sid, streamhosts)
        else:
            # we do not expect this file
            self.sendStreamHostError('XXX', iq)
        
    def _downloadFile(self, iq, sid, streamhosts):
        """ """
        streamhost,port,initiating_jid=streamhosts.pop(0)
        deferred, filename = self._infiles[sid]
        target_jid=str(self._xmppstream.getJID())
        f=socksv5.S5ClientFactory(sid, initiating_jid=initiating_jid,
            target_jid=target_jid, ofile=open(filename,'w+'))
        f.d_connect_made.addCallback(self._connectMade, iq, sid, streamhost_jid=initiating_jid)
        f.d_connect_made.addErrback(self._connectError, sid, streamhosts)
        f.d_file_received.addCallback(self._gotFile, sid)
        f.d_file_received.addErrback(self._gotFileError, sid, streamhosts)
        reactor.connectTCP(streamhost, int(port), f)

    def _connectMade(self, address, iq, sid, streamhost_jid):
        """ """ 
        result=iq.makeResultFrom()
        q=ezel(result.setQueryNS(feat_bytestream))
        q.add('streamhost-used', jid=streamhost_jid)
        self._xmppstream.sendIq(result)
    
    def _connectError(self, failure, sid, streamhosts):
        """ """
        if debug:
            utils.debug(failure)
        
    def sendStreamHostError(self, error, iq):
        """ """
        pass #XXX not implemented yet
        
    def gotStreamHostUsed(self, iq, sid):
        """ ok, the bytestream is working """
        #XXX we should start the stream only now, but at present  it starts 
        # automatically once a correct connect is made
        
    def _gotFile(self, file, sid):
        d,path=self._infiles[sid]
        del self._infiles[sid]
        file.seek(0)
        d.callback(file)
        
    def _gotFileError(self, failure, sid, streamhosts):
        # try with next stream host if there are any
        if streamhosts: self._downloadFile(sid, streamhosts)
        else:
            d,path=self._infiles[sid]
            del self._infiles[sid]
            d.errback(failure)

