#
# Copyright (C) 2004 Mekensleep
#
# Mekensleep
# 24 rue vieille du temple
# 75004 Paris
#       licensing@mekensleep.com
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# Authors:
#  Loic Dachary <loic@gnu.org>
#  Henry Precheur <henry@precheur.org>
#
#
from twisted.internet.protocol import Protocol, Factory
from twisted.internet import reactor
from twisted.python import dispatch
from string import split, join, rstrip
from time import time

INTERFACE_READY = "//event/poker3d/pokerinterface/ready"
INTERFACE_GONE = "//event/poker3d/pokerinterface/gone"
INTERFACE_LOGIN = "//event/poker3d/pokerinterface/login"
INTERFACE_TABLES = "//event/poker3d/pokerinterface/tables"
INTERFACE_TABLES_REFRESH = "//event/poker3d/pokerinterface/tables_refresh"
INTERFACE_HANDS = "//event/poker3d/pokerinterface/hands"
INTERFACE_OUTFITS = "//event/poker3d/pokerinterface/outfits"
INTERFACE_YESNO = "//event/poker3d/pokerinterface/yesno"
INTERFACE_MESSAGE_BOX = "//event/poker3d/pokerinterface/message_box"
INTERFACE_USER_INFO = "//event/poker3d/pokerinterface/user_info"
INTERFACE_CHOOSER = "//event/poker3d/pokerinterface/chooser"
INTERFACE_POST_BLIND = "//event/poker3d/pokerinterface/post_blind"
INTERFACE_AUTO_BLIND = "//event/poker3d/pokerinterface/auto_blind"
INTERFACE_SIT_OUT = "//event/poker3d/pokerinterface/sit_out"
INTERFACE_SLIDER = "//event/poker3d/pokerinterface/slider"
INTERFACE_BUY_IN = "//event/poker3d/pokerinterface/buy_in"
INTERFACE_SHOW_OUTFITS = "//event/poker3d/pokerinterface/show_outfits"
INTERFACE_SHOW_CASHIER = "//event/poker3d/pokerinterface/show_cashier"
INTERFACE_SHOW_TABLE_LIST = "//event/poker3d/pokerinterface/show_table_list"
INTERFACE_CHAT_HISTORY = "//event/poker3d/pokerinterface/chat_history"
INTERFACE_CHAT_LINE = "//event/poker3d/pokerinterface/chat_line"
INTERFACE_FUTURE_ACTION = "//event/poker3d/pokerinterface/future_action"
INTERFACE_IN_POSITION = "//event/poker3d/pokerinterface/in_position"

class PokerInterfaceProtocol(Protocol, dispatch.EventDispatcher):
    def __init__(self):
        dispatch.EventDispatcher.__init__(self)
        self.initLobby()

    def initLobby(self):
        self.tables = None
        self.hands = None
        self.outfits = None
        self.keybindings = None

    def connectionMade(self):
        self.factory.publishEvent(INTERFACE_READY, self, self.factory)
        self.factory.clearCallbacks(INTERFACE_READY)

    def connectionLost(self, reason):
        self.factory.publishEvent(INTERFACE_GONE, self, self.factory)        
        self.factory.clearCallbacks(INTERFACE_GONE)
        
    def dataReceived(self, data):
        if self.verbose > 1:
            print "PokerInterfaceProtocol: dataReceived %s " % data
        data = split(rstrip(data, "\0"), "\0")
        while data:
            type = data[0]
            if type == "login":
                data = self.handleLogin(data[1:])
            elif type == "lobby":
                if self.tables:
                    data = self.handleTables(data[1:])
                elif self.hands:
                    data = self.handleHands(data[1:])
                elif self.keybindings:
                    data = self.handleKeyBinding(data[1:])
                else:
                    print "PokerInterfaceProtocol: unexpected lobby message"
                    data = data[1:]
            elif type == "yesno":
                data = self.handleYesNo(data[1:])
            elif type == "blind":
                data = self.handleBlind(data[1:])
            elif type == "sit_actions":
                data = self.handleSitActions(data[1:])
            elif type == "message_box":
                data = self.handleMessageBox(data[1:])
            elif type == "user_info":
                data = self.handleUserInfo(data[1:])
            elif type == "chooser":
                data = self.handleChooser(data[1:])
            elif type == "slider":
                data = self.handleSlider(data[1:])
            elif type == "buy_in":
                data = self.handleBuyIn(data[1:])
            elif type == "chat":
                data = self.handleChat(data[1:])
            elif type == "top_menu":
                data = self.handleTopMenu(data[1:])
            elif type == "future_action":
                data = self.handleFutureAction(data[1:])
            elif type == "in_position":
                data = self.handleInPosition(data[1:])
            elif type == "outfit":
                data = self.handleOutfit(data[1:])
            else:
                print "PokerInterfaceProtocol: unexpected type %s " % type
                data = data[1:]
            
    def handleLogin(self, data):
        (ok_or_cancel, name, password, remember) = data[:4]
        remember = remember == "1"
        if self.verbose > 1:
            print "PokerInterfaceProtocol: login %s, password %s, remember %s\n" % (name, password, remember)
        self.publishEvent(INTERFACE_LOGIN, ok_or_cancel, name, password, remember)
        self.clearCallbacks(INTERFACE_LOGIN)
        return data[4:]

    def requestLogin(self, name, password, remember):
        remember = remember and 1 or 0
        packet = "login\0%s\0%s\0%d\0" % (name, password, remember)
        if self.verbose > 1:
            print "PokerInterfaceProtocol:requestLogin %s" % packet
        self.transport.write(packet)

    def hideLogin(self):
        packet = "login\0hide\0"
        self.transport.write(packet)

    def showTables(self, tables):
        self.initLobby()
        self.tables = {}
        for table in tables:
            self.tables[str(table.id)] = table
        variants = {}
        for table in tables:
            if not variants.has_key(table.variant):
                variants[table.variant] = []
            variants[table.variant].append(table)
        packet = "lobby\0"
        packet += "table_list\0"
        packet += "Choose a poker table to join\0"
        packet += "%d\0" % len(variants) # number of tabs
        variant_names = variants.keys()
        variant_names.sort()
        for variant in variant_names:
            packet += "%d\0" % 11 # number of fields
            packet += "0\0001\0001\0000\0000\0000\0000\0000\0000\0000\0000\0" # field types
            packet += "id\0name\0structure\0seats\0avg. pot\0hands/h\0%flop\0playing\0observing\0waiting\0timeout\0" # headers
            packet += "%d\0" % len(variants[variant])
            for table in variants[variant]:
                packet += "%d\0" % table.id
                packet += "%s\0" % table.name
                packet += "%s\0" % table.betting_structure
                packet += "%d\0" % table.seats
                packet += "%d\0" % table.average_pot
                packet += "%d\0" % table.hands_per_hour
                packet += "%d\0" % table.percent_flop
                packet += "%d\0" % table.players
                packet += "%d\0" % table.observers
                packet += "%d\0" % table.waiting
                packet += "%d\0" % table.timeout
            packet += "%s\0" % variant
        if self.verbose > 1:
            print "PokerInterfaceProtocol:showTables %s" % packet
        self.transport.write(packet)

    def handleTables(self, data):
        (command, table_id) = data[:2]
        if command == "select" or command == "quit":
            tables = self.tables
            self.tables = None
            self.publishEvent(INTERFACE_TABLES, (command == "select" and tables[table_id]) or None)
        elif command == "refresh":
            if self.callbacks.has_key(INTERFACE_TABLES_REFRESH):
                self.publishEvent(INTERFACE_TABLES_REFRESH, "")
        self.clearCallbacks(INTERFACE_TABLES_REFRESH, INTERFACE_TABLES)
        return data[2:]

    def showOutfits(self, outfits):
        packet = "outfit\0"
        self.outfits_list=[]
        for type in outfits:
            outfitfortype=outfits[type]
            for outfit_name in outfitfortype:
                self.outfits_list.append(str(type)+" - "+str(outfit_name))

        packet += "%d\0" % len(self.outfits_list)# number of outfiname
        for i in self.outfits_list:
            packet += "%s\0" % (i)# outfit str

        if self.verbose > 1:
            print "PokerInterfaceProtocol:showOutfits %s" % packet
        self.transport.write(packet)

    def handleOutfit(self, data):
        (command, outfit_name) = data[:2]
        if command == "select" or command == "quit":
            self.outfits = None
            if command == "select":
                self.publishEvent(INTERFACE_OUTFITS,outfit_name)
        self.clearCallbacks(INTERFACE_OUTFITS)
        return data[2:]

#    def handleOutfit(self, data):
#        (action,id)=data[:2]
#        if action is "selected":
#            pass
            
#        if self.verbose > 1:
#            print "PokerInterfaceProtocol: outfit %s, id %s\n" % (action, str(id))
#        self.publishEvent(INTERFACE_OUTFITS, action, sexe,outfit)
#        self.clearCallbacks(INTERFACE_OUTFITS)
#        return data[4:]


    def showHands(self, hands):
        self.initLobby()
        self.hands = hands
        packet = "lobby\0"
        packet += "hands\0"
        packet += "Choose an hand to replay\0"
        packet += "1\0"# number of tabs
        packet += "1\0" # number of fields
        packet += "0\0" # field types
        packet += "Hand #\0" # headers
        packet += "%d\0" % len(hands)
        for hand in hands:
            packet += "%d\0" % hand
        packet += "Hands\0"
        if self.verbose > 1:
            print "PokerInterfaceProtocol:showHands %s" % packet
        self.transport.write(packet)

    def handleHands(self, data):
        (command, hand) = data[:2]
        if command == "select" or command == "quit":
            self.hands = None
            self.publishEvent(INTERFACE_HANDS, (command == "select" and int(hand)) or None)
        elif command == "refresh":
            pass # FIXME
        self.clearCallbacks(INTERFACE_HANDS)
        return data[2:]

    def buildLobbyInterfacePacket(self, packet):
        self.keybindings = True
        packetStr = "lobby\0"
        packetStr += "keybindings\0"
        packetStr += "%s\0" % packet["title"]
        packetStr += "%d\0" % len(packet["tabs"]) #number of tabs
        for tab in packet["tabs"]:
            packetStr += "%d\0" % len(tab["fields"]) #number of fields
            for field in tab["fields"]:
                packetStr += "%s\0" % field[0]
            for field in tab["fields"]:
                packetStr += "%s\0" % field[1]
            packetStr += "%d\0" % len(tab["values"]) #number of values
            for value in tab["values"]:
                for col in value:
                    packetStr += col + "\0"
            packetStr += "%s\0" % tab["title"]
        packet["str"] = packetStr;
        return packetStr

    def handleKeyBinding(self, data):
        self.keybindings = None
        return data[2:]
        
    def showKeyBinding(self, bindings):
        packet = {"title" : "Poker3d KeyBinding",
                  "tabs"  : ( { "title"  : "Keys",
                                "fields" : ( ("1", "Key"), ("1", "Action") ),
                                "values" : bindings }, #do not remove ,
                              ),
                  "str"   : ""
                  }        
        self.buildLobbyInterfacePacket(packet)
        self.transport.write(packet["str"]);
    
    def userInfo(self, *messages):
        packet = "user_info\000%d\0" % len(messages)
        packet += "\0".join(messages) + "\0"
        if self.verbose > 1:
            print "PokerInterfaceProtocol:userInfo %s : %s" % ( str(messages), packet )
        self.transport.write(packet)

    def handleUserInfo(self, data):
        if self.verbose > 1:
            print "PokerInterfaceProtocol:handleUserInfo"
        if self.callbacks.has_key(INTERFACE_USER_INFO):
            self.publishEvent(INTERFACE_USER_INFO)
            self.clearCallbacks(INTERFACE_USER_INFO)
        return data
        
    def chooser(self, title, alternatives):
        packet = "chooser\0%s\000%d\0" % ( title, len(alternatives) )
        packet += "\0".join(alternatives) + "\0"
        if self.verbose > 1:
            print "PokerInterfaceProtocol:chooser %s : %s" % ( str(alternatives), packet )
        self.transport.write(packet)

    def handleChooser(self, data):
        (alternative,) = data[:1]
        if self.verbose > 1:
            print "PokerInterfaceProtocol:chooser"
        if self.callbacks.has_key(INTERFACE_CHOOSER):
            self.publishEvent(INTERFACE_CHOOSER, alternative)
            self.clearCallbacks(INTERFACE_CHOOSER)
        return data[1:]
        
    def messageBox(self, message):
        if self.verbose > 1:
            print "PokerInterfaceProtocol:messageBox %s" % message
        self.transport.write("message_box\0%s\0" % message)

    def handleMessageBox(self, data):
        if self.verbose > 1:
            print "PokerInterfaceProtocol:handleMessageBox"
        if self.callbacks.has_key(INTERFACE_MESSAGE_BOX):
            self.publishEvent(INTERFACE_MESSAGE_BOX)
            self.clearCallbacks(INTERFACE_MESSAGE_BOX)
            
    def blindShow(self):
        if self.verbose > 1:
            print "PokerInterfaceProtocol:blind show"
        self.transport.write("blind\0show\0")

    def blindHide(self):
        if self.verbose > 1:
            print "PokerInterfaceProtocol:blind hide"
        self.transport.write("blind\0hide\0")

    def blindMessage(self, message, wait_blind):
        self.blindShow()
        packet = "blind\0blind message\0%s\0%s\0" % ( message, wait_blind )
        if self.verbose > 1:
            print "PokerInterfaceProtocol:blindMessage %s" % packet
        self.transport.write(packet)

    def handleBlind(self, data):
        if data[0] == "post":
            (tag, answer) = data[:2]
            self.publishEvent(INTERFACE_POST_BLIND, answer)
            self.clearCallbacks(INTERFACE_POST_BLIND)
            data = data[2:]
        else:
            raise Exception("bad packet received from blind")
        return data

    def sitActionsShow(self):
        if self.verbose > 1:
            print "PokerInterfaceProtocol:sitActions show"
        self.transport.write("sit_actions\0show\0")

    def sitActionsHide(self):
        if self.verbose > 1:
            print "PokerInterfaceProtocol:sitActions hide"
        self.transport.write("sit_actions\0hide\0")

    def sitActionsAuto(self, auto):
        if self.verbose > 1:
            print "PokerInterfaceProtocol:sitActions auto"
        self.transport.write("sit_actions\0auto\0%s\0" % auto)

    def sitActionsSitOut(self, status, message):
        if self.verbose > 1:
            print "PokerInterfaceProtocol:sitActions sit_out"
        self.transport.write("sit_actions\0sit_out\0%s\0%s\0" % ( status, message ) )

    def handleSitActions(self, data):
        if data[0] == "auto":
            answer = data[1]
            self.publishEvent(INTERFACE_AUTO_BLIND, answer == "yes")
        elif data[0] == "sit_out":
            answer = data[1]
            self.publishEvent(INTERFACE_SIT_OUT, answer == "yes")
        else:
            raise Exception("bad packet received from sit_actions")
        return data[2:]

    def yesnoBox(self, message):
        if self.verbose > 1:
            print "PokerInterfaceProtocol:yesnoBox %s" % message
        self.transport.write("yesno\0%s\0" % message)

    def handleYesNo(self, data):
        response = data[0]
        if response == "yes":
            result = True
        elif response == "no":
            result = False
        else:
            raise Exception("bad packet recieved from lobby")
        self.publishEvent(INTERFACE_YESNO, result)
        self.clearCallbacks(INTERFACE_YESNO)
        return data[1:]

    def requestSlider(self, bet, raiseMin, raiseMax, step):
        packet = "slider\0%d\0%d\0%d\0%d\0" % (bet, raiseMin, raiseMax, step)
        if self.verbose > 1:
            print "PokerInterfaceProtocol:requestSlider %s" % packet
        self.transport.write(packet)
        
    def handleSlider(self, data):
        value = data[0]
        self.publishEvent(INTERFACE_SLIDER, value)
        self.clearCallbacks(INTERFACE_SLIDER)
        return data[1:]

    def chatShow(self):
        self.transport.write("chat\0show\0")
        
    def chatHide(self):
        self.transport.write("chat\0hide\0")

    def chatHistory(self, message):
        self.transport.write("chat\0line\0%s\0" % message)
        
    def handleChat(self, data):
        if data[0] == "history":
            self.publishEvent(INTERFACE_CHAT_HISTORY, data[1])
        elif data[0] == "line":
            self.publishEvent(INTERFACE_CHAT_LINE, data[1])
        return data[2:]

    def buyInShow(self):
        self.transport.write("buy_in\0show\0")
        
    def buyInHide(self):
        self.transport.write("buy_in\0hide\0")
        
    def buyInParams(self, minimum, maximum, legend, max_label):
        packet = "buy_in\0params\0%d\0%d\0%s\0%s\0" % (minimum, maximum, legend, max_label)
        if self.verbose > 1:
            print "PokerInterfaceProtocol:requestBuyIn %s" % packet
        self.transport.write(packet)
        
    def handleBuyIn(self, data):
        value = data[0]
        self.publishEvent(INTERFACE_BUY_IN, value)
        self.clearCallbacks(INTERFACE_BUY_IN)
        return data[1:]

    def futureActionShow(self):
        self.transport.write("future_action\0show\0")
        
    def futureActionHide(self):
        self.transport.write("future_action\0hide\0")
        
    def futureActionParams(self, labels, selected):
        if selected:
            selected = str(labels.index(selected))
        else:
            selected = ""
        labels = labels + [''] * ( 8 - len(labels) )
        self.futureActionSerial = int(time())
        packet = "future_action\0set\000%s\000%s\000%s\000" % ( self.futureActionSerial, selected, "\000".join(labels) )
        if self.verbose > 1:
            print "PokerInterfaceProtocol:futureActionParams %s" % packet
        self.transport.write(packet)
        
    def handleFutureAction(self, data):
        serial = int(data[0])
        value = int(data[1])
        #
        # Avoid race condition (labels sent, user action published before
        # the labels are received).
        #
        if serial == self.futureActionSerial:
            self.publishEvent(INTERFACE_FUTURE_ACTION, value)
        return data[2:]

    def inPositionShow(self):
        self.transport.write("in_position\0show\0")
        
    def inPositionHide(self):
        self.transport.write("in_position\0hide\0")
        
    def inPositionParams(self, min_raise, max_raise, step, to_call, button1, button2, button3):
        packet = "in_position\0set\000%d\000%d\000%d\000%d\000%s\000%s\000%s\000" % ( min_raise, max_raise, step, to_call, button1, button2, button3 )
        if self.verbose > 1:
            print "PokerInterfaceProtocol:inPositionParams %s" % packet
        self.transport.write(packet)
        
    def handleInPosition(self, data):
        button = int(data[0])
        value = int(data[1])
        self.publishEvent(INTERFACE_IN_POSITION, button, value)
        return data[2:]

    def topMenuShow(self):
        self.transport.write("top_menu\0show\0")
        
    def topMenuHide(self):
        self.transport.write("top_menu\0hide\0")
        
    def topMenuParams(self, message):
        packet = "top_menu\0params\0%s\0" % (message)
        if self.verbose > 1:
            print "PokerInterfaceProtocol:topMenuParams %s" % packet
        self.transport.write(packet)
        
    def handleTopMenu(self, data):
        if data[0] == "table_list":
            self.publishEvent(INTERFACE_SHOW_TABLE_LIST, data[1])
        elif data[0] == "outfit":
            self.publishEvent(INTERFACE_SHOW_OUTFITS, data[1])
        elif data[0] == "cashier":
            self.publishEvent(INTERFACE_SHOW_CASHIER, data[1])
        return data[2:]

    def clearCallbacks(self, *events):
        for event in events:
            if self.callbacks.has_key(event):
                del self.callbacks[event]

class PokerInterfaceFactory(Factory, dispatch.EventDispatcher):

    protocol = PokerInterfaceProtocol
    
    def __init__(self, *args, **kwargs):
        dispatch.EventDispatcher.__init__(self)
        self.verbose = kwargs["verbose"]
        
    def buildProtocol(self, addr):
        protocol = Factory.buildProtocol(self, addr)
        protocol.verbose = self.verbose
        return protocol

    def clearCallbacks(self, *events):
        for event in events:
            if self.callbacks.has_key(event):
                del self.callbacks[event]
