#
# Copyright (C) 2004 Mekensleep
#
# Mekensleep
# 24 rue vieille du temple
# 75004 Paris
#       licensing@mekensleep.com
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# Authors:
#  Loic Dachary <loic@gnu.org>
#
#
import os
if os.name != "posix" :
    import win32api, win32pdhutil, win32con
    from underware import python_mywin32

import sys
import socket
from os.path import expanduser, exists
from string import split, join, replace
from time import sleep
from shutil import copy

import libxml2

from twisted.internet import reactor
from twisted.python import dispatch

from underware.config import ConfigError
from poker import pokerinterface
from poker import pokerjabber
from poker import jabberproxy

def expand(url, command, substitute):
    args = []
    for arg in split(command):
        for (original, destination) in substitute.iteritems():
            arg = replace(arg, original, destination)
        args.append(arg)
    if not exists(args[0]):
        raise ConfigError, "%s, as found in %s at line %s is not an existing file." % ( args[0], url, command )
    return args

def killProcName(procname_ori):
    print "killing " + procname_ori
    python_mywin32.killProcessByName(procname_ori)
    return
    procname=procname_ori.replace(".exe","")

    # Change suggested by Dan Knierim, who found that this performed a
    # "refresh", allowing us to kill processes created since this was run
    # for the first time.
    try:
        win32pdhutil.GetPerformanceAttributes('Process','ID Process',procname)
    except:
        pass

    pids = win32pdhutil.FindPerformanceAttributesByName(procname)
    try:
        pids.remove(win32api.GetCurrentProcessId())
    except ValueError:
        pass

    if len(pids)==0:
        result = "Can't find %s" % procname
    elif len(pids)>1:
        result = "Found too many %s's - pids=`%s`" % (procname,pids)
    else:
        handle = win32api.OpenProcess(win32con.PROCESS_TERMINATE, 0,pids[0])
        win32api.TerminateProcess(handle,0)
        win32api.CloseHandle(handle)
        result = ""

    return result

class PokerChild(dispatch.EventDispatcher):
    def __init__(self, config, settings):
        dispatch.EventDispatcher.__init__(self)
        self.config = config
        self.settings = settings
        self.verbose = settings.headerGetInt("/settings/@verbose")
        self.pid = None
        self.commandLine = None
        self.commandName = None
        self.ready = False

        if settings.headerGet("/settings/user/@path"):
            self.poker3drc = expanduser(settings.headerGet("/settings/user/@path"))
        else:
            self.poker3drc = '.'

    def kill(self):
        if not self.ready:
            return -1

        path = self.poker3drc + "/" + self.commandName + ".pid"
        if not self.pid:
            if exists(path):
                fd = file(path, "r")
                self.pid = int(fd.read(512))
                fd.close()
                if self.verbose:
                    print "found %s pid %d in %s" % ( self.commandName, self.pid, path )

        if self.pid:
            if os.name!="posix":
                try:
                    tokill=self.commandName
                    killProcName(tokill)
                except:
                    print "%s terminate()" % ( tokill )
            else:
                try:
                    os.kill(self.pid, 9)
            	    try:
                        os.waitpid(self.pid, 0)
                    except OSError:
                        print "cannot wait for %s process %d to die : %s" % ( self.commandName, self.pid, sys.exc_value )

                except:
                    print "%s kill(%d,9) : %s" % ( self.commandName, self.pid, sys.exc_value )

        if exists(path):
            os.remove(path)

        pid = self.pid
        self.pid = None
        return pid

    def savepid(self, what, pid):
        path = self.poker3drc + "/" + what + ".pid"
        fd = file(path, "w")
        fd.write(str(pid))
        fd.close()
        if self.verbose:
            print "%s pid %d saved in %s" % ( what, pid, path )
        return pid
    
    def spawn(self):
        if not self.ready:
            return False
        
        print "%s: %s" % (self.commandName, join(self.commandLine))
        import signal
        handler = signal.getsignal(signal.SIGINT)
        signal.signal(signal.SIGINT, signal.SIG_IGN)
        pid = os.spawnv(os.P_NOWAIT, self.commandLine[0], self.commandLine)
        signal.signal(signal.SIGINT, handler)
        self.pid = self.savepid(self.commandName, pid)
        return True

class PokerChildXwnc(PokerChild):

    def __init__(self, config, settings):
        PokerChild.__init__(self, config, settings)
        self.display = None
        self.ready = self.configure()
        self.kill()

    def configure(self):
        config = self.config
        settings = self.settings

        screen = self.settings.headerGetProperties("/settings/screen")[0]
#        self.display = config.headerGet("/sequence/metisse/@display")
        self.display = settings.headerGet("/settings/metisse/@display")
        replace = {}
        replace["%width%"] = screen["width"]
        replace["%height%"] = screen["height"]
        replace["%display%"] = self.display

        if not settings.headerGetList("/settings/metisse/xwnc"):
            print "no <xwnc> found in config file (ignored)"
            return False
#        self.commandLine = expand(self.config.url, config.headerGet("/sequence/metisse/xwnc"), replace)
        self.commandLine = expand(self.settings.url, settings.headerGet("/settings/metisse/xwnc"), replace)
        self.commandName = split(self.commandLine[0],"/").pop()
        if self.verbose:
            print "command name of xwnc : " + self.commandName
        return True

    def spawn(self):
        if not PokerChild.spawn(self):
            return False
        
        retry = 5
        if self.verbose:
            print "Wait for xwnc to respond (will try for %d seconds) " % retry
        for i in xrange(retry):
            try:
                fd = socket.socket()
                fd.connect(('127.0.0.1', 5900 + int(self.display)))
                fd.close()
                return True
                break
            except socket.error:
                print ".",
                sys.stdout.flush()
                sleep(1)
        return False

    def kill(self):
        PokerChild.kill(self)
        try:
            remove("/tmp/.X11-unix/X%d" % self.display)
        except:
            pass

CHILD_INTERFACE_READY = "//event/poker3d/pokerchildren/lobbyready"
CHILD_INTERFACE_GONE = "//event/poker3d/pokerchildren/lobbygone"

class PokerChildInterface(PokerChild):
    def __init__(self, config, settings):
        PokerChild.__init__(self, config, settings)
        self.verbose = settings.headerGetInt("/settings/@verbose")
#        self.port = config.headerGetInt("/sequence/metisse/lobby/@port")
        self.port = settings.headerGetInt("/settings/metisse/lobby/@port")
#        self.ready = ( config.headerGetList("/sequence/metisse/lobby") and
        self.ready = ( settings.headerGetList("/settings/metisse/lobby") and
                       self.configure() )
        self.kill()

    def configure(self):
        config = self.config
        settings = self.settings

#        display = config.headerGet("/sequence/metisse/@display")
        display = settings.headerGet("/settings/metisse/@display")
        replace = {}
        replace["%port%"] = str(self.port)
        replace["%display%"] = display
        replace["%verbose%"] = str(self.verbose)
        replace["%datadir%"] = settings.headerGet("/settings/data/@path")


        if not settings.headerGetList("/settings/metisse/lobby"):
            print "no <lobby> found in config file (ignored)"
            return False
        self.commandLine = expand(self.settings.url, settings.headerGet("/settings/metisse/lobby"), replace)
#        if not config.headerGetList("/sequence/metisse/lobby"):
#            print "no <lobby> found in config file (ignored)"
#            return False
#        self.commandLine = expand(self.config.url, config.headerGet("/sequence/metisse/lobby"), replace)
        self.commandName = split(self.commandLine[0],"/").pop()
        if self.verbose:
            print "command name of lobby : " + self.commandName
        return True

    def lobbyReady(self, lobby, lobbyFactory):
        self.publishEvent(CHILD_INTERFACE_READY, self, lobby, lobbyFactory)

    def lobbyGone(self, lobby, lobbyFactory):
        self.publishEvent(CHILD_INTERFACE_GONE, self, lobby, lobbyFactory)
        
    def spawn(self):
        lobby = pokerinterface.PokerInterfaceFactory(verbose = self.verbose)
        lobby.registerHandler(pokerinterface.INTERFACE_READY, self.lobbyReady)
        lobby.registerHandler(pokerinterface.INTERFACE_GONE, self.lobbyGone)
        reactor.listenTCP(self.port, lobby)
        return PokerChild.spawn(self)

CHILD_JABBER_READY = "//event/poker3d/pokerchildren/jabberready"
CHILD_JABBER_GONE = "//event/poker3d/pokerchildren/jabbergone"

class PokerChildGaim(PokerChild):
    
    def __init__(self, config, settings):
        PokerChild.__init__(self, config, settings)
        self.name = settings.headerGet("/settings/name")
        self.passwd = settings.headerGet("/settings/passwd")

        self.proxy = None
        self.ready = ( settings.headerGetList("/settings/jabber") and
                       self.configure() )
        self.kill()

    def setAuth(self):
        settings = self.settings
        self.name = settings.headerGet("/settings/name")
        self.passwd = settings.headerGet("/settings/passwd")
        self.ready = self.configure()
        
    def configure(self):
        self.gaimConfigure()
        
        config = self.config
        settings = self.settings

        screen = self.settings.headerGetProperties("/settings/screen")[0]
        display = settings.headerGet("/settings/metisse/@display")
        replace = {}
        replace["%width%"] = screen["width"]
        replace["%height%"] = screen["height"]
        replace["%display%"] = display
        replace["%name%"] = self.name
        replace["%connect_host%"] = settings.headerGet("/settings/jabber/connect/@host")
        replace["%gaimrc%"] = self.gaimrc
        replace["%datadir%"] = settings.headerGet("/settings/data/@path")

#        if not config.headerGetList("/sequence/metisse/chat"):
        if not settings.headerGetList("/settings/metisse/chat"):
            print "no <chat> found in config file (ignored)"
            return False
#        self.commandLine = expand(self.config.url, config.headerGet("/sequence/metisse/chat"), replace)
        self.commandLine = expand(self.settings.url, settings.headerGet("/settings/metisse/chat"), replace)
        self.commandName = split(self.commandLine[0],"/").pop()
        if self.verbose:
            print "command name of chat : " + self.commandName
        return True

    def spawn(self):
        settings = self.settings
        auth = pokerjabber.PokerJabberAuthOrCreate(
            host = settings.headerGet("/settings/jabber/connect/@host"),
            port = settings.headerGetInt("/settings/jabber/connect/@port"),
            name = self.name,
            password = self.passwd,
            verbose = self.verbose
            )
        auth.registerHandler(pokerjabber.JABBER_AUTHORCREATE_OK, self.jabberAuthOk)
        auth.registerHandler(pokerjabber.JABBER_AUTHORCREATE_FAILED, self.jabberAuthFailed)
        auth.jabberAuth()
        self.auth = auth
        return None
        
    def gaimConfigure(self):
        return ( self.gaimConfigureDirectory() and
                 self.gaimConfigureAccount() and
                 self.gaimConfigureWindow() )

    def gaimConfigureDirectory(self):
        config = self.config
        settings = self.settings
        self.gaimrc = self.poker3drc + "/gaim"
        if not exists(self.gaimrc):
            try:
                os.mkdir(self.gaimrc)
            except OSError:
                print sys.exc_value
                return False

        return True
        
    def gaimConfigureAccount(self):
        config = self.config
        settings = self.settings
        content = """
<?xml version='1.0' encoding='UTF-8' ?>

<accounts>
 <account>
  <protocol>prpl-jabber</protocol>
  <name>%name%@%connect_host%/Gaim</name>
  <password>%passwd%</password>
  <alias>%name%</alias>
  <settings>
   <setting name='use_tls' type='bool'>1</setting>
   <setting name='old_ssl' type='bool'>0</setting>
   <setting name='auth_plain_in_clear' type='bool'>0</setting>
   <setting name='connect_server' type='string'>%listen_host%</setting>
   <setting name='port' type='int'>%listen_port%</setting>
  </settings>
  <settings ui='gtk-gaim'>
   <setting name='auto-login' type='bool'>0</setting>
  </settings>
 </account>
</accounts>
"""
        content = replace(content, "%name%", self.name)
        content = replace(content, "%passwd%", self.passwd)
        content = replace(content, "%connect_host%", settings.headerGet("/settings/jabber/connect/@host"))
        content = replace(content, "%listen_host%", settings.headerGet("/settings/jabber/listen/@host"))
        content = replace(content, "%listen_port%", settings.headerGet("/settings/jabber/listen/@port"))
        path = self.gaimrc + "/accounts.xml"
        fd = file(path, "w")
        fd.write(content)
        fd.close()
        if self.verbose:
            print "wrote gaim configuration to %s" % path
        return True

    def gaimConfigureWindow(self):
        path = self.gaimrc + "/prefs.xml"
        if not exists(path):
            for dir in self.config.dirs:
                default = dir + "/prefs.xml"
                if exists(default):
                    copy(default, path)

        parameters = self.config.headerGetProperties("/sequence/chat")
        if not parameters:
            if self.verbose:
                print "%s has no /sequence/chat element, gaim configuration is not altered" % self.config.url
            return True
        
	doc = libxml2.parseFile(path)
        for (name, value) in parameters[0].iteritems():
            nodes = doc.xpathEval("//pref[@name='chat']/pref[@name='" + name + "']")
            if not nodes:
                print """%s /sequence/chat/@%s does not match any
//pref[@name='chat']/pref[@name='%s'] in %s (ignored)""" % ( self.settings.url, name, name, path )
            else:
                nodes[0].setProp("value", value)
        doc.saveFile(path)
        return True
    
    def jabberProxyReady(self, jabber):
        if self.verbose:
            print "Jabber session established (gaim <-> proxy <-> server)"
        self.publishEvent(CHILD_JABBER_READY, jabber)

    def jabberProxyFailed(self):
        settings = self.settings
        name = self.name
        jabber_host = settings.headerGet("/settings/jabber/connect/@host")
        jabber_port = settings.headerGetInt("/settings/jabber/connect/@port")
        gaimrc = expanduser(settings.headerGet("/settings/user/@path")) + "/gaim"
        print """
The chat client (gaim) failed to authenticate user %s
to the jabber server %s:%d using the jabber proxy built in
poker3d. When launched as part of poker3d, gaim uses the
configuration file at %s, generated from the poker3d
configuration files %s.
""" % ( name, jabber_host, jabber_port, gaimrc + "/accounts.xml", settings.url )
        print "Jabber disabled"

    def jabberAuthFailed(self):
        del self.auth
    
    def jabberAuthOk(self, jabber):
        #
        # The session that was used to test if the auth is valid won't
        # be used any more because we use a lighter proxy.
        #
        jabber.closeStream()
        del self.auth
        reactor.callLater(0, self.runProxyAndChild)

    def runProxyAndChild(self):
        if not self.proxy:
            settings = self.settings
            jabber_host = settings.headerGet("/settings/jabber/connect/@host")
            jabber_port = settings.headerGetInt("/settings/jabber/connect/@port")
            if self.verbose:
                print "jabberSpawnProxy: run proxy and chat client"
            factory = pokerjabber.PokerJabberProxyServerFactory(jabber_host,
                                                                jabber_port,
                                                                self.verbose)
            factory.registerHandler(jabberproxy.SESSION_START, self.jabberProxyReady)
            factory.registerHandler(jabberproxy.AUTH_FAILURE, self.jabberProxyFailed)
            reactor.listenTCP(settings.headerGetInt("/settings/jabber/listen/@port"),
                              factory,
                              settings.headerGetInt("/settings/jabber/listen/@backlog"),
                              settings.headerGet("/settings/jabber/listen/@host"))
            self.proxy = factory
        PokerChild.spawn(self)
        
class PokerChildren:

    def __init__(self, config, settings):
        self.config = config
        self.settings = settings
        self.verbose = settings.headerGetInt("/settings/@verbose")
        self.ready = False
        self.children = []

        self.xwnc = PokerChildXwnc(config, settings)
        if self.xwnc.ready:
            self.ready = self.xwnc.spawn()
        if self.verbose and self.ready:
            print "Xwnc ready"

    def spawn(self, child):
        child.spawn()
        self.children.append(child)
        
    def kill(self, child):
        child.kill()
        self.children.remove(child)
        
    def killall(self):
        status = True
        for child in self.children:
            status = child.kill() and status
        status = self.xwnc.kill() and status
        return status


