#
# Copyright (C) 2004 Mekensleep
#
# Mekensleep
# 24 rue vieille du temple
# 75004 Paris
#       licensing@mekensleep.com
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# Authors:
#  Loic Dachary <loic@gnu.org>
#
#
import sys
sys.path.insert(0, "..")
sys.path.insert(0, "../../python")

import os
if os.name != "posix" :
    import win32api, win32pdhutil, win32con
#    from underware.mywin32 import getProcessId

from string import split, lower
from os import makedirs
from os.path import expanduser, exists
import signal
from traceback import print_exc
import libxml2
from shutil import copy

from time import sleep


from twisted.internet import reactor
from twisted.internet import error

from underware.mafapplication import MAFApplication
from underware.client import UGAMEClientFactory
from underware.config import Config, ConfigError

from poker.pokerchildren import PokerChildren, PokerChildGaim, PokerChildInterface
from poker import pokerchildren
from poker import pokerinterface
from poker.pokerclient import PokerClientProtocol
from poker.pokerrenderer import PokerRenderer
from poker.pokergame import PokerGameClient
from poker import pokerjabber
from poker.pokerpackets import *

import poker.pokeranimation

def usage(message):
    print message
    print """
usage: poker3d [configuration file]

An example configuration file can be found in
/etc/poker3d/client/poker.client.xml or
~/.poker3d/poker.client.xml.

The configuration argument looks like a URL, as follows:

file:/etc/poker3d/client/poker.client.xml
file:~/.poker3d/poker.client.xml
file:~/myconfig.xml
"""
    sys.exit(1)


class PokerClientFactory(UGAMEClientFactory):
    "client factory"

    def __init__(self, *args, **kwargs):
        UGAMEClientFactory.__init__(self, *args, **kwargs)
        self.config = kwargs["config"]
        self.settings = kwargs["settings"]

        settings = self.settings
        config = self.config
        self.name = settings.headerGet("/settings/name")
        self.password = settings.headerGet("/settings/passwd")
        self.host = "unknown"
        self.port = 0
        self.remember = settings.headerGet("/settings/remember") == "yes"
        self.chat_config = config.headerGetProperties("/sequence/chat")
        if self.chat_config:
            self.chat_config = self.chat_config[0]
            for (key, value) in self.chat_config.iteritems():
                self.chat_config[key] = int(value)
        else:
            self.chat_config = {}
            
        chips_values = config.headerGetList("/sequence/chips/subvalue/@value")
	self.chips_values = [ int(x) for x in chips_values ]
        self.dirs = split(self.settings.headerGet("/settings/path"))
        self.verbose = self.settings.headerGetInt("/settings/@verbose")
        self.delays = self.settings.headerGetProperties("/settings/delays")
        if self.delays:
            self.delays = self.delays[0]
            for (key, value) in self.delays.iteritems():
                self.delays[key] = int(value)
        else:
            self.delays = {}
        self.delays_enable = self.settings.headerGet("/settings/@delays") == "true"
        self.protocol = PokerClientProtocol
        self.renderer = PokerRenderer(self)
        self.serial2player_info = {}
        self.interface = None
        self.jabber = None
        self.gaim = None
	self.games = {}
        self.file2name = {}
        self.initOutfits()
        self.initUnderware()
        self.initChildren()
        self.underware.SetAmetista(self.children.ready and 1 or 0)
        self.underware.SetClient(self)
        self.animations = poker.pokeranimation.create(self.underware, config, settings)

    def __del__(self):
        del self.games

    def quit(self, dummy = None):
        interface = self.interface
        if interface:
            if not interface.callbacks.has_key(pokerinterface.INTERFACE_YESNO):
                interface.yesnoBox("Do you really want to quit ?")
                interface.registerHandler(pokerinterface.INTERFACE_YESNO, self.confirmQuit)
        else:
            self.confirmQuit(True)

    def confirmQuit(self, response):
        if response:
            #
            # !!! The order MATTERS here !!! underware must be notified last
            # otherwise leak detection won't be happy. Inverting the two
            # is not fatal and the data will be freed eventually. However,
            # debugging is made much harder because leak detection can't
            # check as much as it could.
            #
            self.renderer.confirmQuit()
            packet = PacketQuit()
            self.underware.PythonAccept(packet)

    def reload(self):
        if not self.animations:
            return
        
        animations = self.animations
        reload(poker.pokeranimation)
        self.animations = poker.pokeranimation.create(self.underware, self.config, self.settings)
        self.animations.stealFrom(animations)
        print "Animation reload %d" % self.animations.stealCount()
        
    def translateFile2Name(self, file):
        if not self.file2name.has_key(file):
            config = Config(self.dirs)
            config.loadHeader("file:poker.%s.xml" % file)
            name = config.headerGet("/bet/description")
            if not name:
                name = config.headerGet("/poker/variant/@name")
                if not name:
                    print "*CRITICAL* can't find readable name for %s" % file
                    name = file
            self.file2name[file] = name
        return self.file2name[file]
        
    def initOutfits(self):
        settings = self.settings
        config = self.config

        datadir = settings.headerGet("/settings/data/@path")
        self.outfit = self.settings.headerGet("/settings/outfit/@name") or "default"
        self.outfits = {}
        types = split(config.headerGet("/sequence/outfit/@types"))
        for type in types:
            description = Config([''])
            description.loadHeader("file:" + datadir + "/player." + type + ".cal3d/cal3d.xml")
            self.outfits[type] = description.headerGetList("/cal3d/outfits/outfit/@name")
        if len(self.outfits) < 1:
            print "*CRITICAL* no outfit found"

    def getOutfits(self):
        return self.outfits

    def setOutfit(self, outfit):
        settings = self.settings
        settings.headerSet("/settings/outfit/@name", outfit)
        settings.saveHeader()
        self.outfit = outfit

    def getOutfit(self):
        return self.outfit
        
    def initChildren(self):
        settings = self.settings
        config = self.config
        
        self.children = PokerChildren(config, settings)
            
        if self.children.ready:
            interface = PokerChildInterface(config, settings)
            interface.registerHandler(pokerchildren.CHILD_INTERFACE_READY, self.setInterface)
            interface.registerHandler(pokerchildren.CHILD_INTERFACE_GONE, self.unsetInterface)
            self.children.spawn(interface)

    def initUnderware(self):
        settings = self.settings
        config = self.config

        self.underware = MAFApplication()
        self.underware.SetHeaders(settings.doc._o, config.doc._o)
        self.underware.SetReactor(reactor)
        self.underware.InitWindow()
        
    def runJabberClient(self):
        settings = self.settings
        config = self.config
        if not settings.headerGetList("/settings/jabber"):
            return
        if self.gaim:
            self.gaim.setAuth()
        else:
            gaim = PokerChildGaim(config, settings)
            gaim.registerHandler(pokerchildren.CHILD_JABBER_READY, self.setJabber)
            gaim.registerHandler(pokerchildren.CHILD_JABBER_GONE, self.unsetJabber)
            self.gaim = gaim
        self.children.spawn(self.gaim)

    def runBasicClient(self):
        if self.settings.headerGetList("/settings/jabber"):
            return
        if self.interface:
            self.interface.chatShow()

    def killJabberClient(self):
        if self.gaim:
            self.unsetJabber(self.jabber)
            self.children.kill(self.gaim)
        
    def setJabber(self, jabber):
        self.jabber = jabber
        self.setJabberCallbacks()

    def unsetJabber(self, jabber):
        self.unsetJabberCallbacks()
        self.jabber = None

    def setJabberCallbacks(self):
        jabber = self.jabber
        if self.renderer and jabber:
            jabber.registerHandler(pokerjabber.JABBER_MESSAGE, self.renderer.jabberMessageReceive)

    def unsetJabberCallbacks(self):
        jabber = self.jabber
        if self.renderer and jabber:
            del jabber.callbacks[pokerjabber.JABBER_MESSAGE]
            
    def setInterface(self, interfaceChild, interface, interfaceFactory):
        self.interface = interface
        self.showServers()
        self.renderer.interfaceReady(interface)

    def unsetInterface(self, interfaceChild, interface, interfaceFactory):
        self.interface = None
        self.children.kill(interfaceChild)
        if hasattr(self, "factory") and not self.factory.shutting_down:
            print "2D interface process is gone, can't continue without it."
            reactor.stop()

    def showServers(self):
        servers = split(self.settings.headerGet("/settings/servers"))
        if len(servers) > 1:
            interface = self.interface
            interface.chooser("Choose a poker3d server", servers)
            interface.registerHandler(pokerinterface.INTERFACE_CHOOSER, self.selectServer)
        else:
            self.selectServer(servers[0])
            
    def selectServer(self, server):
        (self.host, self.port) = split(server, ":")
        self.port = int(self.port)
        config = self.config
        reactor.connectTCP(self.host,
                           self.port,
                           self,
                           config.headerGetInt("/sequence/tcptimeout/@value"))
        

    def buildProtocol(self, addr):
        protocol = UGAMEClientFactory.buildProtocol(self, addr)
        self.renderer.setProtocol(protocol)
        self.animations.setProtocol(protocol)
        self.setJabber(self.jabber)
        return protocol

    def saveAuthToFile(self, name, password, remember):
        settings = self.settings
        self.name = name
        self.password = password
        self.remember = remember
        if remember:
            remember = "yes"
        else:
            remember = "no"
            name = "username"
            password = "password"
        settings.headerSet("/settings/remember", remember)
        settings.headerSet("/settings/name", name)
        settings.headerSet("/settings/passwd", password)
        settings.saveHeader()
        settings.headerSet("/settings/name", self.name)
        settings.headerSet("/settings/passwd", self.password)

    def getGame(self, game_id):
	if not self.games.has_key(game_id):
	    return False
	else:
	    return self.games[game_id]

    def getGameByNameNoCase(self, name):
        for (serial, game) in self.games.iteritems():
            if lower(game.name) == name:
                return game
        return None
    
    def getOrCreateGame(self, game_id):
	if not self.games.has_key(game_id):
	    game = PokerGameClient("file:poker.%s.xml", self.dirs)
            game.verbose = self.verbose
	    game.id = game_id
	    self.games[game_id] = game

        return self.games[game_id]

    def deleteGame(self, game_id):
	del self.games[game_id]

    def packet2game(self, packet):
        if hasattr(packet, "game_id") and self.games.has_key(packet.game_id):
            return self.games[packet.game_id]
        else:
            return False

    def clientConnectionFailed(self, connector, reason):
        print "connectionFailed: %s" % reason
        self.interface.messageBox("Unable to reach the poker3d\nserver at %s:%d" % ( self.host, self.port ))
        self.interface.registerHandler(pokerinterface.INTERFACE_MESSAGE_BOX, self.showServers)
        
    def clientConnectionLost(self, connector, reason):
        self.renderer.setProtocol(None)
        reconnect = True
        if hasattr(self, "reconnect"):
            reconnect = self.reconnect
            del self.reconnect

        if reconnect:
            message = "The poker3d server connection was closed"
            if not reason.check(error.ConnectionDone):
                message += " " + str(reason)
            print message
            if self.interface:
                self.interface.messageBox("Lost connection to poker3d\nserver at %s:%d" % ( self.host, self.port ))
                self.interface.registerHandler(pokerinterface.INTERFACE_MESSAGE_BOX, self.showServers)
    
class Main:
    "Poker gameplay"

    def __init__(self, configfile, settingsfile):
        self.settings = Config([''])
        self.settings.loadHeader(settingsfile)
        self.shutting_down = False
        if self.settings.header:
            rcdir = self.configureDirectory()
            self.dirs = split(self.settings.headerGet("/settings/path"))
            self.config = Config(self.dirs)
            self.config.loadHeader(configfile)
            self.verbose = self.settings.headerGetInt("/settings/@verbose")
            self.poker3d_factory = None
#            sleep(12) # we would like this feature on windows (in dev)

    def configOk(self):
        return self.settings.header and self.config.header
    
    def shutdown(self, signal, stack_frame):
        self.shutting_down = True
        self.poker3d_factory.underware.Stop()
        if self.verbose:
            print "received signal %s, exiting" % signal
            
    def configureDirectory(self):
        settings = self.settings
        if not settings.headerGet("/settings/user/@path"):
            print """
No <user path="user/settings/path" /> found in file %s.
Using current directory instead.
""" % settings.url
            return
        
        rcdir = expanduser(settings.headerGet("/settings/user/@path"))
        if not exists(rcdir):
            os.mkdir(rcdir)

    def run(self):
        settings = self.settings
        config = self.config

        signal.signal(signal.SIGINT, self.shutdown)
        if os.name == "posix":
            signal.signal(signal.SIGQUIT, self.shutdown)
        signal.signal(signal.SIGTERM, self.shutdown)

        name = settings.headerGet("/settings/name")
        passwd = settings.headerGet("/settings/passwd")
        poker3d_factory = PokerClientFactory(settings = settings,
                                             config = config)
        self.poker3d_factory = poker3d_factory

        try:
            poker3d_factory.underware.Run()
        except:
            print_exc()

        if hasattr(poker3d_factory, "children"):
            poker3d_factory.children.killall()
        poker3d_factory.underware.Stop()
        poker3d_factory.underware = None

settingsfile = len(sys.argv) > 1 and sys.argv[1] or [ "file:~/.poker3d/poker.client.xml", "file:/etc/poker3d/client/poker.client.xml" ]
configfile = len(sys.argv) > 2 and sys.argv[2] or "file:client.xml"

# copy client.xml in ~/.poker3d if he does not already exists
if os.name == "posix":
    user_dir = expanduser("~/.poker3d/")
    conf_file = user_dir + "poker.client.xml"
    if not exists(conf_file) and exists("/etc/poker3d/client/poker.client.xml"):
        if not exists(user_dir):
            makedirs(user_dir)
        copy("/etc/poker3d/client/poker.client.xml", expanduser("~/.poker3d/"))

client = Main(configfile, settingsfile)

if client.configOk():
    try:
        client.run()
    except:
        if client.verbose:
            print_exc()
        else:
            print sys.exc_value

reactor.stop()
reactor.runUntilCurrent();
del client
libxml2.cleanupParser()

#
# Emacs debug:
# cp /usr/lib/python2.3/pdb.py .
# M-x pdb
# ../../underware pdb.py poker3d.py file:conf/client/mekensleep.xml
#
