/**
    Copyright (C) 2004 Cedric Pinson <cpinson@freesheep.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 ****************************************************************************
 * @file   mth_hermite.h
 *
 * @brief   hermite curve
 *
 *****************************************************************************
 *
 * @author  cpinson
 *
 * @date    Created 2003/05
 *
 * @version $Id: mth_hermite.h,v 1.4 2004/05/17 17:11:13 proppy Exp $
 *
 ****************************************************************************/

#ifndef MTH_HERMITE_H
#define MTH_HERMITE_H

#include <list>
#include <assert.h>

namespace mth {

  template <class T> T Frac(const T& _x) { return _x-floorf(_x);}

  /**
   *  General hermite curve.
   *
   *  You need to have some operator to use this class:
   *  You must use a class that accept (T + T) , (T - T), (T * scalar) with T as class
   *
   *  @param T class to use for your data to interpolate
   *  @param P precision wanted (default float)
   */
  template <class T,class P=float> class Hermite_T
  {

   public:

    /**
     *  Key data of T
     */
    class Key_t
    {
      P mTime;
      T mData;
      P mTens;
      P mCont;
      P mBias;
     public:
      Key_t(){}
      Key_t(P _time,const T& _data,P _tens=0,P _cont=0,P _bias=0):mTime(_time),mData(_data),mTens(_tens),mCont(_cont),mBias(_bias) {}

      void Init(P _time,const T& _data,P _tens=0,P _cont=0,P _bias=0) {
        mTime=_time;mData=_data;mTens=_tens;mCont=_cont;mBias=_bias;
      }

      void Time(P _time) { mTime=_time;}
      inline P Time() const { return mTime;}

      void Data(const T& _data) { mData=_data;}
      inline const T& Data() const { return mData;}

      inline P Tens() const { return mTens;}
      void Tens(P _tens) { mTens=_tens;}

      inline P Bias() const { return mBias;}
      void Bias(P _bias) { mBias=_bias;}

      inline P Cont() const { return mCont;}
      void Cont(P _cont) { mCont=_cont;}

    };


   private:

    /// shortcut
    typedef std::list<Key_t*> KeyContainer_t;
    typedef typename KeyContainer_t::iterator KeyIt_t;
    typedef typename KeyContainer_t::const_iterator KeyConstIt_t;
    typedef typename KeyContainer_t::size_type Size_t;

    // data
    KeyContainer_t mKeys;
  

    inline P DeltaTime(Key_t* _cur,Key_t* _next) const { return _next->Time()-_cur->Time();}


    inline void BuildSi(Key_t* _prev,Key_t* _cur,Key_t* _next,T& _ts) const {

      assert(_prev && _cur && _next);

      const T tmp1=_cur->Data()-_prev->Data();
      const T tmp2=_next->Data()-_cur->Data();
    
      const P one=1;    //just to avoid big syntax
      const P two=2;    //just to avoid big syntax
      const P half=0.5; //just to avoid big syntax
      const P pars3= one - _cur->Tens();

      const P pars0= pars3 * ( one + _cur->Bias() )*( one - _cur->Cont()) * half;
      const P pars1= pars3 * ( one - _cur->Bias() )*( one + _cur->Cont()) * half;

      const P curDeltaI=DeltaTime(_cur,_next);
      const P prevDeltaI=DeltaTime(_prev,_cur);
    
      const P delta=two * curDeltaI/(prevDeltaI+curDeltaI);

      _ts=(tmp1*pars0+tmp2*pars1)*delta;
    }


    inline void BuildDi(Key_t* _prev,Key_t* _cur,Key_t* _next,T& _td) const {

      assert(_prev && _cur && _next);

      const T tmp1=_cur->Data()-_prev->Data();
      const T tmp2=_next->Data()-_cur->Data();

      const P one=1;    //just to avoid big syntax
      const P two=2;    //just to avoid big syntax
      const P half=0.5; //just to avoid big syntax
      const P pars3= one - _cur->Tens();

      const P pars0= pars3 * ( one + _cur->Bias())*( one + _cur->Cont()) * half;
      const P pars1= pars3 * ( one - _cur->Bias())*( one - _cur->Cont()) * half;

      const P curDeltaI=DeltaTime(_cur,_next);
      const P prevDeltaI=DeltaTime(_prev,_cur);
    
      const P delta= two *prevDeltaI/(prevDeltaI+curDeltaI);

      _td=(tmp1*pars0+tmp2*pars1)*delta;
    }

  
   public:


    /**
     *  Default constructor
     */
    Hermite_T() { mKeys.clear();}



    /**
     *  Clear the list of keyframes
     */
    void Clear() { mKeys.clear();}


   /**
     *  Insert a key frame with this time in the spline.
     *  warning you can't insert more than one time an element
     *
     *  @param _key to insert
     */
    void InsertKey(Key_t* _key) {
      assert(_key);

      if (mKeys.empty()) {
        mKeys.push_back(_key);
        return;
      }

      // cheking begin
      P limit_t0=mKeys.front()->Time();
      if (_key->Time()<limit_t0) {
        mKeys.push_front(_key);
        return;
      }

      // checking last
      P limit_t1=mKeys.back()->Time();
      if (_key->Time()>limit_t1) {
        mKeys.push_back(_key);
        return;
      }


      // give me the good interval p0---t------p1
      KeyIt_t it0=mKeys.begin();
      while ( _key->Time()<(*it0)->Time())
        it0++;
      mKeys.insert(++it0,_key);
    }




   /**
     *  Remove a key frame in the spline.
     *
     *  @param _key to remove
     */
    void RemoveKey(Key_t* _key) {
      assert(_key);

      // no keys return
      if (mKeys.empty())
        return;

      // cheking begin
      if (mKeys.front()==_key) {
        mKeys.erase(mKeys.begin());
        return;
      }
      
      // checking last
      if (mKeys.back()==_key) {
        mKeys.erase(mKeys.end());
        return;
      }

      mKeys.remove(_key);
    }



    /**
     *  FirstKey return the first key of list.
     *
     *  @return Key_t* first key
     */
    Key_t* FirstKey() {

      // no keys return
      if (mKeys.empty())
        return 0;

      return mKeys.front();
    }


    /**
     *  LastKey return the last key of list.
     *
     *  @return Key_t* last key
     */
    Key_t* LastKey() {

      // no keys return
      if (mKeys.empty())
        return 0;

      return mKeys.back();
    }



   /**
     *  GetData return a interpolated value from time given as parameter
     *  We use data type defined by keyframe
     *	
     *  @param _time step you want
     *  @param _result of interpolation
     *  @param _loop or not
     *  @return int 0 if ok else -1
     */
    inline int GetData(P _time,T& _result,bool _loop=1) const {

      Size_t nbkeys=mKeys.size();
      if (!nbkeys)
        return -1;


      if (nbkeys<2) {
        _result=mKeys.front()->Data();
        return 0;
      }

    
      // cheking range
      const P limit_t0=mKeys.front()->Time();
      const P limit_t1=mKeys.back()->Time();


      if (_loop) {
        // loop mode
        if (_time < limit_t0)
          _time=limit_t0;
        if (_time>limit_t1) {
          P dist=limit_t1-limit_t0;
          _time=_time/dist;
          _time=mth::Frac(_time)*dist;
        }

      } else {
        // stop mode
        if (_time < limit_t0)
          _time=limit_t0;

        if (_time>limit_t1)
          _time=limit_t1;
      }


      KeyConstIt_t itPrev,itCur,itNext,itNext2;
      itCur=mKeys.begin();

      itCur++; //start at the second key
      // give me the good interval p0---t------p1
      while ( _time>(*itCur)->Time())
        itCur++;
      
      itCur--;
    
      // check valid prev and next
      itPrev=itCur;
      if (itCur!=mKeys.begin())
        itPrev--;

      itNext=itCur;
      if (itCur!=--mKeys.end())
        itNext++;
    
    
      const P t=(_time-(*itCur)->Time())/DeltaTime(*itCur,*itNext); // rescale the time
      const P t2=t*t;
      const P t3=t2*t;

      const P c2t3= t3 * (P)2;
      const P c3t2= t2 * (P)3;

      // p=(2t3-3t2+1)*p0 + (t3-2t2+t)*tg0 + (-2t3+3t2)*p1 + (t3-t2)*tg1
      const P one=1;
      const P two=2;
      const P h1= c2t3 - c3t2 + one;           // 2t3-3t2+1 
      const P h2= t3 + t  - t2 * two;          // t3-2t2+t
      const P h3= -c2t3 + c3t2;                // -2t3+3t2 
      const P h4= t3 - t2;                     // t3-t2

      T tg0,tg1;
    
      BuildDi(*itPrev,*itCur,*itNext,tg0);


      itNext2=itNext;
      if (itNext!=--mKeys.end())
        itNext2++;

      BuildSi(*itCur,*itNext,*itNext2,tg1);
    
      _result=(*itCur)->Data()*h1+ tg0*h2 + (*itNext)->Data()*h3 + tg1*h4;
      return 0;
    }


  };


};

#endif
