/*
 * * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Johan Euphrosine <johan@mekensleep.com>
 *  Loic Dachary <loic@gnu.org>
 *
 */

#include "pokerStdAfx.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef WIN32
#include "config_win32.h"
#endif

#include <sstream>
#include <iomanip>
#include <map>

#include <osg/Shape>
#include <osg/ShapeDrawable>
#include <osg/Geode>
#include <osg/MatrixTransform>
#include <osg/Material>
#include <osg/NodeVisitor>
#include <osgUtil/IntersectVisitor>

#include <maf/utils.h>
#include <maf/packets.h>

#ifndef WIN32
#include <PokerSeat.h>
#include <PokerApplication.h>
#include <maf/interpolator.h>
#include <PokerError.h>
#else
#define snprintf _snprintf
#endif


//model
PokerSeatModel::PokerSeatModel() : mId(0), mFree(true), mArrowPAT(0)
{
}

PokerSeatModel::~PokerSeatModel()
{
  g_debug("PokerSeatModel::~PokerSeatModel()");
}

void PokerSeatModel::Init()
{
  UGAMEArtefactModel::Init();

  GetNode()->setName("PokerSeat");
  mArrowPAT = new osg::PositionAttitudeTransform;
  mScaleInterpolator.Init(osg::Vec3(1.0f, 1.0f, 1.0f), osg::Vec3(2.0f, 2.0f, 2.0f));
  mTimer.Init(0.5f);
}


class DisableVisitor : public osg::NodeVisitor {
public:
  DisableVisitor() : osg::NodeVisitor(osg::NodeVisitor::TRAVERSE_ALL_CHILDREN) {
  }
  virtual void apply(osg::Geode& geode) {
    geode.setNodeMask(0);
  }
};

class EnableVisitor : public osg::NodeVisitor {
public:
  EnableVisitor() : osg::NodeVisitor(osg::NodeVisitor::TRAVERSE_ALL_CHILDREN) {
  }
  virtual void apply(osg::Geode& geode) {
    geode.setNodeMask(0xffffffff);
  }
};

struct CloneAnchorNodePath
{
  static osg::Node *doit(osg::Node *anchor, osg::Node *child)
  {
    osg::MatrixTransform *t = new osg::MatrixTransform;
    osg::NodePath path;
    osg::Node* node = anchor;
    while(node) {
      path.push_back(node);
      if(node->getNumParents() >= 1) {
	if(node->getNumParents() > 1)
	  g_critical("more than one parent");
	node = node->getParent(0);
      } else {
	node = 0;
      }
    }
    std::reverse(path.begin(), path.end());  
    t->setMatrix(osg::computeLocalToWorld(path));
    t->addChild(child);
    return t;
  }
};

struct BoxCreate
{
  static osg::Node *doit()
  {
    osg::Geode* d=new osg::Geode;
    osg::Box* dbox=new osg::Box(osg::Vec3(0,0,0),10.f);
    osg::ShapeDrawable* bd=new osg::ShapeDrawable(dbox);
    d->addDrawable(bd);
    return d;
  }
};


class KillAllGeode : public osg::NodeVisitor {
public:
  KillAllGeode() : osg::NodeVisitor(osg::NodeVisitor::TRAVERSE_ALL_CHILDREN) {
  }

  virtual void apply(osg::Geode& geode) {
    geode.setNodeMask(0x0);
  }
};

//controller
PokerSeatController::PokerSeatController()
{
  SetModel(new PokerSeatModel());
}

PokerSeatController::~PokerSeatController()
{
  Anchor(0);

  GetModel()->mArrowPAT = 0;
  GetModel()->mSeatGeode=0;

  RecursiveClearUserData(GetModel()->GetNode());

  g_debug("PokerSeatController::~PokerSeatController");
  osg::NodeVisitor* leakNodes = RecursiveLeakCollect(GetModel()->GetNode());
  RecursiveLeakCheck(leakNodes);
}

void PokerSeatController::Init(int id, PokerApplication *game)
{
  UGAMEArtefactController::Init();  
  
  PokerSeatModel *model = GetModel();
  model->mId = id;  


  //
  // Set the artefact : Artefact()
  //
  {
    const std::string &url = game->HeaderGet("sequence", "/sequence/seat/@url");
    MAFOSGData *data = static_cast<MAFOSGData *>(game->mDatas->GetVision(url)->Clone());
    osg::Group* group = data->GetGroup();
    KillAllGeode visitor;
    group->accept(visitor);
    model->SetArtefact(group);
    delete data;
  }

  //
  // Grab a seat in the scene : Anchor()
  //
  {

    const std::string &anchorName = game->HeaderGet("sequence", "/sequence/seat/@anchor");
    char oss[64];
    snprintf(oss, 64, anchorName.c_str(), id + 1);
    MAFAnchor *anchor = game->mSetData->GetAnchor(oss);
    g_assert(anchor != 0);
    Anchor(anchor);
  }

  //
  // Figure out where the arrow should be placed in the artefact
  //
  osg::PositionAttitudeTransform *arrowPAT;
  {
    const std::string &url = game->HeaderGet("sequence", "/sequence/seat/arrow/@anchor");
    osg::Group *anchor = dynamic_cast<osg::Group*>(GetNode(model->GetArtefact(), url));
    g_assert(anchor != 0);
    arrowPAT = model->mArrowPAT.get();
    arrowPAT->setName("ArrowPAT");
    anchor->addChild(arrowPAT);
  }

  //
  // This controller reacts to events occuring on the arrow
  //
  BindToNode(arrowPAT);

  //
  // Add the arrow object in the artefact
  //
  {
    const std::string &url = game->HeaderGet("sequence", "/sequence/seat/arrow/@url");
    MAFOSGData *data = static_cast<MAFOSGData *>(game->mDatas->GetVision(url)->Clone());
    g_assert(data != 0);
    osg::Node *node = data->GetGroup();
    g_assert(node != 0);
    arrowPAT->addChild(node);
    delete data;
  }

  //
  // Add the chair in the artefact
  //
  {
    const std::string &url = game->HeaderGet("sequence", "/sequence/seat/chair/@url");
    const std::string &anchor = game->HeaderGet("sequence", "/sequence/seat/chair/@anchor");
    MAFOSGData *data = static_cast<MAFOSGData *>(game->mDatas->GetVision(url)->Clone());
    g_assert(data != 0);
    osg::Group* group = dynamic_cast<osg::Group*>(GetNode(GetModel()->GetArtefact(), anchor));
    g_assert(group != 0);
    group->addChild(data->GetGroup());
    delete data;
    //
    // Keep this pointer to set transparency
    //
    GetModel()->mSeatGeode = dynamic_cast<osg::Geode*>(GetGeode(group));
    g_assert(GetModel()->mSeatGeode.valid());
  }
  
  Disable();
  //chairNode->getOrCreateStateSet()->setRenderingHint(osg::StateSet::TRANSPARENT_BIN);
}

void PokerSeatController::Disable()
{
  SetSelectable(false);
  GetModel()->GetArtefact()->setNodeMask(0);
}

void PokerSeatController::Enable()
{
  SetSelectable(true);
  GetModel()->GetArtefact()->setNodeMask(0xffffffff);
}

bool PokerSeatController::Update(MAFApplication *game)
{
  UGAMEArtefactController::Update(game);
  
  if (game->HasEvent())
    return true;

  bool focused = (game->GetFocus() == this);
  float delta = GetDeltaFrame()/1000.f;
  delta = (focused ? delta : -delta);
  GetModel()->mTimer.AddTime(delta);
  osg::Vec3 scale;
  GetModel()->mTimer.Get(scale, GetModel()->mScaleInterpolator);
  GetModel()->mArrowPAT->setScale(scale);

  if (GetSelected()) {
    GetModel()->SetSelected(false);
    GetModel()->mFree = !GetModel()->mFree;
    return false;
  }

  return true;
}

// manager

PokerSeatManager::PokerSeatManager() : mMainPlayerIn(false), mSeatsCount(0), mCurrentTableSeats(0), mSeats(0)
{
}

PokerSeatManager::~PokerSeatManager()
{
  g_debug("PokerSeatManager::~PokerSeatManager");
  for (unsigned int i = 0; i < mSeats.size(); i++)
  {
    PokerSeatController &seat = *(mSeats[i]);
    RecursiveClearUserData(seat.GetModel()->GetNode());
    mSeats[i] = 0;
  }
  //RecursiveClearUserData(mDoor->GetModel()->mDoorCollNode.get());
  return;
}

void PokerSeatManager::Init(PokerApplication *game)
{
  MAFController::Init();
  const int defaultSeatCount = 10;
  mSeatsCount = defaultSeatCount;
  mCurrentTableSeats.resize(defaultSeatCount);
  mSeats.resize(defaultSeatCount);
  for (int seatId = 0; seatId < defaultSeatCount; seatId++)
    {
      mCurrentTableSeats[seatId] = false;
      PokerSeatController *seat = new PokerSeatController();
      seat->Init(seatId, game);
      mSeats[seatId] = seat;
    }
}

void PokerSeatManager::SetSeats(const std::vector<int> &currentTableSeats)
{
  mMainPlayerIn = false;
  for (unsigned int i = 0; i < mCurrentTableSeats.size(); i++)
    {
      if (mCurrentTableSeats[i])
	mSeats[i]->Disable();
      mCurrentTableSeats[i] = false;
    }
  for (unsigned int i = 0; i < currentTableSeats.size(); i++)
    {
      mCurrentTableSeats[currentTableSeats[i]] = true;
    }
  for (unsigned int seatId = 0; seatId < mSeatsCount; seatId++)
    {
      if (mCurrentTableSeats[seatId])
	{
	  PokerSeatController &seat = *(mSeats[seatId]);
	  seat.Enable();
	}
    }
}

void PokerSeatManager::DisableAllSeats()
{
  for (unsigned int i = 0; i < mSeatsCount; i++)
    {
	if (mCurrentTableSeats[i])
	  mSeats[i]->Disable();
    }
}

void PokerSeatManager::MainPlayerArrive(const std::vector<guint> &seat2serial)
{
	mMainPlayerIn = true;
	g_assert(mSeatsCount <= seat2serial.size()); 
	for (unsigned int i = 0; i < mSeatsCount; i++)
	{
		if (!seat2serial[i])
			if (mCurrentTableSeats[i])
				mSeats[i]->Disable();
	}
}

void PokerSeatManager::MainPlayerLeave(const std::vector<guint> &seat2serial)
{
	mMainPlayerIn = false;
	g_assert(mSeatsCount <= seat2serial.size()); 
	for (unsigned int i = 0; i < mSeatsCount; i++)
	{
		if (!seat2serial[i])
			if (mCurrentTableSeats[i])
				mSeats[i]->Enable();
	}
}

void PokerSeatManager::PlayerLeave(unsigned int seat)
{
	if (!mMainPlayerIn)
		if (mCurrentTableSeats[seat])
			mSeats[seat]->Enable();
}

void PokerSeatManager::PlayerSeated(unsigned int seat)
{
	if (!mMainPlayerIn)
		if (mCurrentTableSeats[seat])
		  {
			mSeats[seat]->Disable();
		  }
}

bool PokerSeatManager::Update(MAFApplication *game)
{
  unsigned int seatsCount = mSeatsCount;

  if (!mMainPlayerIn)
    {
      for (unsigned int seatId = 0; seatId < seatsCount; seatId++)
	{
		if (mCurrentTableSeats[seatId])
		{
			PokerSeatController &seat = *(mSeats[seatId]);
			if (!seat.DoUpdate(game))
				{
				PokerModel *poker = static_cast<PokerApplication *>(game)->GetPoker()->GetModel();
				osg::ref_ptr<MAFPacket> packet;
				packet = game->GetPacketsModule()->Create("PacketPokerSeat");
				packet->SetMember("serial", poker->mMe);
				packet->SetMember("game_id", poker->mGameSerial);	  
				packet->SetMember("seat", seat.GetModel()->mId);
				game->PythonCall("getSeat", packet.get());
				}
		}
	}
	}
  return true;
}
