/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Johan Euphrosine <johan@mekensleep.com>
 *  Igor Kravtchenko <igor@obraz.net>
 *
 */

#include "pokerStdAfx.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef WIN32
#include "config_win32.h"
#endif

#ifndef WIN32
#include "PokerPlayerCamera.h"
#include "PokerPlayer.h"
#include "Poker.h"
#include <osg/ref_ptr>

#endif // WIN32

#include <osg/Material>
#include <osg/BlendFunc>

//#define CONVX(a) ((a)/800.0f)
//#define CONVY(a) ((a)/600.0f)

#define CONVX(a) a
#define CONVY(a) a

/*
static osg::Vec2f g_HUDPos_Dst[] = {
 osg::Vec2f( CONVX(0),		CONVY(293) ),
 osg::Vec2f( CONVX(12),		CONVY(218) ),
 osg::Vec2f( CONVX(28),		CONVY(145) ),
 osg::Vec2f( CONVX(81),		CONVY(74) ),

 osg::Vec2f( CONVX(303),	CONVY(5) ) ,

 osg::Vec2f( CONVX(568),	CONVY(74) ),
 osg::Vec2f( CONVX(618),	CONVY(145) ),
 osg::Vec2f( CONVX(645),	CONVY(218) ),
 osg::Vec2f( CONVX(648),	CONVY(293) ) };

 static osg::Vec2f g_HUDPos_Src[] = {
 osg::Vec2f( CONVX(-152-20),	CONVY(293) ),
 osg::Vec2f( CONVX(-152-20),	CONVY(218) ),
 osg::Vec2f( CONVX(-152-20),	CONVY(145) ),
 osg::Vec2f( CONVX(-152-20),	CONVY(74) ),

 osg::Vec2f( CONVX(303),		CONVY(-65) ),

 osg::Vec2f( CONVX(800+20),		CONVY(74) ),
 osg::Vec2f( CONVX(800+20),		CONVY(145) ),
 osg::Vec2f( CONVX(800+20),		CONVY(218) ),
 osg::Vec2f( CONVX(800+20),		CONVY(293) ) };
 */


static osg::Vec2f g_HUDPos_Dst[] = {
osg::Vec2f( CONVX(0),		CONVY(283) ),
osg::Vec2f( CONVX(0),		CONVY(218) ),
osg::Vec2f( CONVX(0),		CONVY(155) ),

osg::Vec2f( CONVX(124),		CONVY(10) ),
osg::Vec2f( CONVX(324),		CONVY(10) ) ,
osg::Vec2f( CONVX(524),		CONVY(10) ),

osg::Vec2f( CONVX(648),		CONVY(155) ),
osg::Vec2f( CONVX(648),		CONVY(218) ),
osg::Vec2f( CONVX(648),		CONVY(283) ) };

static osg::Vec2f g_HUDPos_Src[] = {
osg::Vec2f( CONVX(-152-20),	CONVY(283) ),
osg::Vec2f( CONVX(-152-20),	CONVY(218) ),
osg::Vec2f( CONVX(-152-20),	CONVY(155) ),

osg::Vec2f( CONVX(124),		CONVY(-65) ),
osg::Vec2f( CONVX(324),		CONVY(-65) ),
osg::Vec2f( CONVX(524),		CONVY(-65) ),

osg::Vec2f( CONVX(800+20),	CONVY(155) ),
osg::Vec2f( CONVX(800+20),	CONVY(218) ),
osg::Vec2f( CONVX(800+20),	CONVY(283) ) };


PokerPlayerCamera::PokerPlayerCamera(PokerCameraController* camera, std::map<std::string,std::string>& params) {
	int i;

	// Camera parameters
  std::string param = params["timeToReach"];
  mCameraTimeToReach = atof(param.c_str());
  param = params["timeToRewind"];
  mCameraTimeToRewind = atof(param.c_str());
  param = params["timeToRelease"];
  mCameraTimeToRelease = atof(param.c_str());
  param = params["timeShoulder"];
  mCameraTimeShoulder = atof(param.c_str());
  param = params["factorToReach"];
  mCameraFactorToReach = atof(param.c_str());
  param = params["factorToRewind"];
  mCameraFactorToRewind = atof(param.c_str());

  param = params["minTimeToLookCards"];
  mCameraMinTimeToLookCards = atof(param.c_str());
  param = params["velocityToLookCards"];
  mCameraVelocityToLookCards = atof(param.c_str());

  mCameraButtonDelayToActivateDirectMode = atof(params["buttonDelayToActivateDirectMode"].c_str());
  mCameraGameModeDelay = atof(params["gameModeDelay"].c_str());
  mCameraGameModeMouseThreshold = atof(params["gameModeMouseThreshold"].c_str());
  mCameraGameModeDelayLookCards = atof(params["gameModeDelayLookCards"].c_str());

  mCamera = camera;
  mCamera->SetMode(PokerCameraModel::CAMERA_FREE_MODE);
  mCameraTimeout = 0;
      
  mCameraLookCardTimeout = 0;
  mCameraLookCardState = 0;
  mLookingCardAnimationTimeout = 0;
  mCameraButtonPressedDuration = 0;
  mCameraMouseMotion = false;
  mDisplayedAndFocused = false;
  mCameraButtonPressed = false;
  mCameraButtonClickUp = false;
  mCameraButtonClickDown = false;
  mCameraButtonClickFocused = false;
  mInteractorSelected = false;
  mMouseDelta[0] = 0.0f;
  mMouseDelta[1] = 0.0f;
  timeSinceFirstPersonView_ += 0;

	PokerApplication *app = camera->mGame;
	std::string	path = app->HeaderGet("settings", "/settings/data/@path");

	osg::Group *set_root = app->mSetData->GetGroup();

	sens_ = 1;

	{
		osg::Image *imgLeftRight = MAFLoadImage(path + "/" + "hudcards_leftright.tga", app->GetOptions());
		osg::Image *imgMiddle = MAFLoadImage(path + "/" + "hudcards_middle.tga", app->GetOptions());

		osg::Group *root = new osg::Group();
		set_root->addChild(root);

		osg::StateSet *ss = root->getOrCreateStateSet();
		ss->setRenderBinDetails(12, "RenderBin");
		ss->setMode(GL_DEPTH_TEST,osg::StateAttribute::OFF);
		ss->setMode(GL_LIGHTING,osg::StateAttribute::OFF);

		osg::MatrixTransform* modelview_abs = new osg::MatrixTransform;
		modelview_abs->setReferenceFrame(osg::Transform::ABSOLUTE_RF);
		modelview_abs->setMatrix(osg::Matrix::identity());
		root->addChild(modelview_abs);

		osg::Projection *projection = new osg::Projection;
		modelview_abs->addChild(projection);

		osg::Geode *geode_left = new osg::Geode();
		osg::Geode *geode_right = new osg::Geode();
		osg::Geode *geode_middle = new osg::Geode();
		osg::Geometry *geom_left = new osg::Geometry();
		osg::Geometry *geom_right = new osg::Geometry();
		osg::Geometry *geom_middle = new osg::Geometry();

		geode_left->addDrawable(geom_left);
		geode_right->addDrawable(geom_right);
		geode_middle->addDrawable(geom_middle);
		geode_left->setCullingActive(false);
		geode_right->setCullingActive(false);
		geode_middle->setCullingActive(false);

		//osgText::Font *font = MAFLoadFont(path + "/Verdanab.ttf");
		osgText::Font *font = MAFLoadFont(path + "/VeraBd.ttf", app->GetOptions());

		for (i = 0; i < 9; i++) {
			osg::MatrixTransform *mt = new osg::MatrixTransform;
			mt_[i] = mt;
			mt->setNodeMask(0);
			projection->addChild(mt);

			if (i < 3)
				mt->addChild(geode_left);
			else if (i < 6)
				mt->addChild(geode_middle);
			else
				mt->addChild(geode_right);

			osg::MatrixTransform *font_matrix;
			UGAMEBasicText *text;

			font_matrix = new osg::MatrixTransform();
			if (i < 3)
				font_matrix->setMatrix( osg::Matrix::scale(0.003f, 0.003f, 0) * osg::Matrix::translate(3/400.0f, -7/300.0f, 0) );
			else if (i < 6)
				font_matrix->setMatrix( osg::Matrix::scale(0.003f, 0.003f, 0) * osg::Matrix::translate((152/2)/400.0f, -7/300.0f, 0) );
			else
				font_matrix->setMatrix( osg::Matrix::scale(0.003f, 0.003f, 0) * osg::Matrix::translate((152-3)/400.0f, -7/300.0f, 0) );
			text = new UGAMEBasicText("name", font);
			text->getText()->setCharacterSize(9);
			if (i < 3)
				text->getText()->setAlignment( osgText::Text::LEFT_TOP );
			else if (i < 6)
				text->getText()->setAlignment( osgText::Text::CENTER_TOP );
			else
				text->getText()->setAlignment( osgText::Text::RIGHT_TOP );
			mt->addChild(font_matrix);
			textName_[i] = text;
			font_matrix->addChild(text);

			font_matrix = new osg::MatrixTransform();
			if (i < 3)
				font_matrix->setMatrix( osg::Matrix::scale(0.003f, 0.003f, 0) * osg::Matrix::translate(3/400.0f, -25/300.0f, 0) );
			else if (i < 6)
				font_matrix->setMatrix( osg::Matrix::scale(0.003f, 0.003f, 0) * osg::Matrix::translate((76)/400.0f, -25/300.0f, 0) );
			else
				font_matrix->setMatrix( osg::Matrix::scale(0.003f, 0.003f, 0) * osg::Matrix::translate((152-3)/400.0f, -25/300.0f, 0) );

			text = new UGAMEBasicText("last action", font);
			text->getText()->setCharacterSize(9);
			if (i < 3)
				text->getText()->setAlignment( osgText::Text::LEFT_TOP );
			else if (i < 6)
				text->getText()->setAlignment( osgText::Text::CENTER_TOP );
			else
				text->getText()->setAlignment( osgText::Text::RIGHT_TOP );
			mt->addChild(font_matrix);
			textLastAction_[i] = text;
			font_matrix->addChild(text);
		}

		{
			osg::StateSet *state = geom_left->getOrCreateStateSet();
			osg::ref_ptr<osg::Material> material = new osg::Material;
			state->setAttributeAndModes(material.get(), osg::StateAttribute::ON);
			state->setMode(GL_LIGHTING, osg::StateAttribute::OFF);
			state->setMode(GL_CULL_FACE, osg::StateAttribute::OFF);
			material->setDiffuse(osg::Material::FRONT_AND_BACK, osg::Vec4f(1, 1, 1, 1) );

			osg::Vec3Array *array = new osg::Vec3Array();
			array->resize(4);
			geom_left->setVertexArray(array);

			float l = 152 / 400.0f;
			float h = -40 / 300.0f;
			array[0][0] = osg::Vec3f(0, 0, 0);
			array[0][1] = osg::Vec3f(l, 0, 0);
			array[0][2] = osg::Vec3f(l, h, 0);
			array[0][3] = osg::Vec3f(0, h, 0);

			osg::Vec2Array *uv = new osg::Vec2Array();
			uv->push_back( osg::Vec2f(0, 1) );
			uv->push_back( osg::Vec2f(1, 1) );
			uv->push_back( osg::Vec2f(1, 0) );
			uv->push_back( osg::Vec2f(0, 0) );
			geom_left->setTexCoordArray(0, uv);

			GLushort *index = new GLushort[6];
			index[0] = 0;
			index[1] = 1;
			index[2] = 2;
			index[3] = 0;
			index[4] = 2;
			index[5] = 3;

			geom_left->addPrimitiveSet(new osg::DrawElementsUShort(osg::PrimitiveSet::TRIANGLES, 6, index));
			geom_left->setUseDisplayList(false);
			geom_left->setUseVertexBufferObjects(false);

			osg::Texture2D *texture = new osg::Texture2D;
			texture->setImage( imgLeftRight );
			state->setTextureAttributeAndModes(0, texture, osg::StateAttribute::ON);

			osg::BlendFunc *bf = new osg::BlendFunc;
			bf->setFunction(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
			state->setAttributeAndModes(bf, osg::StateAttribute::ON);
		}

		{
			osg::StateSet *state = geom_right->getOrCreateStateSet();
			osg::ref_ptr<osg::Material> material = new osg::Material;
			state->setAttributeAndModes(material.get(), osg::StateAttribute::ON);
			state->setMode(GL_LIGHTING, osg::StateAttribute::OFF);
			state->setMode(GL_CULL_FACE, osg::StateAttribute::OFF);
			material->setDiffuse(osg::Material::FRONT_AND_BACK, osg::Vec4f(1, 1, 1, 1) );

			osg::Vec3Array *array = new osg::Vec3Array();
			array->resize(4);
			geom_right->setVertexArray(array);

			float l = 152 / 400.0f;
			float h = -40 / 300.0f;
			array[0][0] = osg::Vec3f(0, 0, 0);
			array[0][1] = osg::Vec3f(l, 0, 0);
			array[0][2] = osg::Vec3f(l, h, 0);
			array[0][3] = osg::Vec3f(0, h, 0);

			osg::Vec2Array *uv = new osg::Vec2Array();
			uv->push_back( osg::Vec2f(1, 1) );
			uv->push_back( osg::Vec2f(0, 1) );
			uv->push_back( osg::Vec2f(0, 0) );
			uv->push_back( osg::Vec2f(1, 0) );
			geom_right->setTexCoordArray(0, uv);

			GLushort *index = new GLushort[6];
			index[0] = 0;
			index[1] = 1;
			index[2] = 2;
			index[3] = 0;
			index[4] = 2;
			index[5] = 3;

			geom_right->addPrimitiveSet(new osg::DrawElementsUShort(osg::PrimitiveSet::TRIANGLES, 6, index));
			geom_right->setUseDisplayList(false);
			geom_right->setUseVertexBufferObjects(false);

			osg::Texture2D *texture = new osg::Texture2D;
			texture->setImage( imgLeftRight );
			state->setTextureAttributeAndModes(0, texture, osg::StateAttribute::ON);

			osg::BlendFunc *bf = new osg::BlendFunc;
			bf->setFunction(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
			state->setAttributeAndModes(bf, osg::StateAttribute::ON);
		}

		{
			osg::StateSet *state = geom_middle->getOrCreateStateSet();
			osg::ref_ptr<osg::Material> material = new osg::Material;
			state->setAttributeAndModes(material.get(), osg::StateAttribute::ON);
			state->setMode(GL_LIGHTING, osg::StateAttribute::OFF);
			state->setMode(GL_CULL_FACE, osg::StateAttribute::OFF);
			material->setDiffuse(osg::Material::FRONT_AND_BACK, osg::Vec4f(1, 1, 1, 1) );

			osg::Vec3Array *array = new osg::Vec3Array();
			array->resize(4);
			geom_middle->setVertexArray(array);

			float l = 152 / 400.0f;
			float h = -40 / 300.0f;
			array[0][0] = osg::Vec3f(0, 0, 0);
			array[0][1] = osg::Vec3f(l, 0, 0);
			array[0][2] = osg::Vec3f(l, h, 0);
			array[0][3] = osg::Vec3f(0, h, 0);

			osg::Vec2Array *uv = new osg::Vec2Array();
			uv->push_back( osg::Vec2f(0, 1) );
			uv->push_back( osg::Vec2f(1, 1) );
			uv->push_back( osg::Vec2f(1, 0) );
			uv->push_back( osg::Vec2f(0, 0) );
			geom_middle->setTexCoordArray(0, uv);

			GLushort *index = new GLushort[6];
			index[0] = 0;
			index[1] = 1;
			index[2] = 2;
			index[3] = 0;
			index[4] = 2;
			index[5] = 3;

			geom_middle->addPrimitiveSet(new osg::DrawElementsUShort(osg::PrimitiveSet::TRIANGLES, 6, index));
			geom_middle->setUseDisplayList(false);
			geom_middle->setUseVertexBufferObjects(false);

			osg::Texture2D *texture = new osg::Texture2D;
			texture->setImage( imgMiddle );
			state->setTextureAttributeAndModes(0, texture, osg::StateAttribute::ON);

			osg::BlendFunc *bf = new osg::BlendFunc;
			bf->setFunction(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
			state->setAttributeAndModes(bf, osg::StateAttribute::ON);
		}

	}
}

PokerPlayerCamera::UpdateResult PokerPlayerCamera::Update(SDL_Event* event, float delta, bool focused)
{
	int i;

  mDeltaS = delta/1000.0;

  {
	PokerApplication *app = GetCameraController()->mGame;
	PokerModel *model = app->GetPoker()->GetModel();

	guint me_id = model->mMe;
	PokerPlayer *me = model->mSerial2Player[me_id].get();
	int me_index = me->GetSeatId();

	int nb = model->mSeat2Serial.size();

	for (i = 0; i < 9; i++) {
		mt_[i]->setNodeMask(0);
	}

	for (i = 0; i < nb; i++) {
		unsigned int serial = model->mSeat2Serial[i];
		if (serial && serial != me_id) {
			PokerPlayer *player = model->mSerial2Player[serial].get();
			int index = player->GetSeatId();
			int r_index;
			if (index > me_index)
				r_index = index - me_index;
			else
				r_index = 10 + index - me_index;
			r_index--;

			mt_[r_index]->setNodeMask(0xffffffff);
			textName_[r_index]->getText()->setText( player->GetName() );

			const std::string &lastAction = player->GetLastActionString();
			int lastBet = player->GetLastBet();

			char str[200];
			if (lastAction != "Call" && lastAction != "Fold")
				sprintf(str, "%s %d$", lastAction.c_str(), lastBet );
			else
				strcpy(str, lastAction.c_str());

			textLastAction_[r_index]->getText()->setText( str );
		}
	}
  }

  updatePos(mDeltaS * 5);

  int lookCardState = mCameraLookCardState;

  mCameraButtonClickUp = false;
  mCameraButtonClickDown = false;
  mCameraMouseMotion = false;
 

  mDisplayedAndFocused = focused;

  if (mCameraButtonPressed)
    mCameraButtonPressedDuration+=mDeltaS;

  if (event)
    {
      switch(event->type)
	{
	case SDL_MOUSEBUTTONDOWN:
	  {
	    if  (event->button.button != SDL_BUTTON_LEFT)
	      break;
	    mMouseDelta[0] = 0;
	    mMouseDelta[1] = 0;
	    
	    mCameraButtonPressed = true;
	    mCameraButtonPressedDuration = 0;
	    mCameraButtonClickDown = true;
	    mCameraButtonClickFocused = mDisplayedAndFocused;
	  }
	  break;
	case SDL_MOUSEBUTTONUP:
	  {
	    if  (event->button.button != SDL_BUTTON_LEFT)
	      break;
	    mMouseDeltaRel[0] = 0;
	    mMouseDeltaRel[1] = 0;
	    mCameraButtonPressed = false;
	    mCameraButtonClickUp = true;
	  }
	  break;
	case SDL_MOUSEMOTION:
	  {
	    mCameraMouseMotion = true;
	    mMouseDelta[0] += event->motion.xrel;
	    mMouseDelta[1] += event->motion.yrel;
	    mMouseDeltaRel[0] = event->motion.xrel;	      
	    mMouseDeltaRel[1] = event->motion.yrel;
	  }
	  break;
	default:
	  break;
	}
    }    
    
  switch(mCamera->GetMode())
    {
    case PokerCameraModel::CAMERA_FREE_MODE:
	  timeSinceFirstPersonView_ = 0;
      ExecuteFreeMode();
      break;
    case PokerCameraModel::CAMERA_ENTER_MODE:
	  timeSinceFirstPersonView_ = 0;
      ExecuteEnterMode();
      break;
    case PokerCameraModel::CAMERA_GAME_MODE: // player has clicked once
	  timeSinceFirstPersonView_ += mDeltaS;
	  if ( timeSinceFirstPersonView_ > 2)
		sens_ = 0;
      ExecuteGameMode();
      break;
	case PokerCameraModel::CAMERA_DIRECT_MODE: // player has clicked and stay clicked
	  timeSinceFirstPersonView_ += mDeltaS;
	  if ( timeSinceFirstPersonView_ > 2)
 		sens_ = 0;
      ExecuteDirectMode();
      break;
    case PokerCameraModel::CAMERA_LEAVE_MODE:
	  timeSinceFirstPersonView_ = 0;
	  sens_ = 1;
      ExecuteLeaveMode();
      break;
    }

  if(mCameraLookCardState != lookCardState) {
    if(lookCardState == 0)
      return START_LOOK_CARDS;
    else
      return END_LOOK_CARDS;
  } else {
    return NONE;
  }
}

void PokerPlayerCamera::MoveCameraToCamLookModel()
{  
  std::map<std::string,MAFCameraModel>::iterator it=mCameras.find("CamLook");
  g_assert(it!=mCameras.end());
  
  MAFCameraModel& camera=(*it).second;
  const osg::Vec3 &position = camera.GetPosition();
  const osg::Vec3 &target = camera.GetTarget();
  
  MoveCamera(position, target);

  if (!mCameraLookCardState)
    {
      mCameraLookCardState=1;
    }
  else
    {
      g_error("PokerPlayerCamera::MoveCameraToCamLookModel: unexpected mCameraLookCardState == 1");
    }
}

void PokerPlayerCamera::MoveCamera(const osg::Vec3 &position, const osg::Vec3 &target)
{
  float timeout;
  float length = (position - mCamera->GetModel()->GetPosition()).length();
  
  if (mCameraTimeout > 0)
    {
      timeout = mCameraTotalTimeout*1000.f - mCameraTimeout*1000.f;
    }
  else
    {
      timeout = (length) / mCameraVelocityToLookCards;
    }
  
  if (timeout < mCameraMinTimeToLookCards)
    {
      timeout = mCameraMinTimeToLookCards;
    }
  
  g_assert(timeout >= 0);

  mCameraTotalTimeout = mCameraTimeout = timeout/1000.f;
  const float fovTo = 84.5f;
  mCamera->MoveTo(position, target, fovTo, timeout);
}


void PokerPlayerCamera::MoveCameraToPreviousModel()
{
  float timeout;
  float length = (mCamera->GetModel()->GetPosition() - mCamera->GetModel()->mCamPrevPosition).length();
  
  if (mCameraTimeout > 0)
    timeout = mCameraTotalTimeout*1000.f - mCameraTimeout*1000.f;
  else
    timeout = (length) / mCameraVelocityToLookCards;
  
  if (timeout < mCameraMinTimeToLookCards)
    timeout = mCameraMinTimeToLookCards;
  
  g_assert(timeout >= 0);
   
  mCameraTotalTimeout = mCameraTimeout = timeout/1000.f;
  
  mCamera->MoveToPrevious(timeout);
  
  mCameraLookCardState = 0;
}

bool PokerPlayerCamera::CameraEvaluateModeTransition()
{
  const float mCameraButtonDelayToActivateDirectMode=0.2;

  bool canActivateDirectMode = (mCameraButtonPressedDuration>mCameraButtonDelayToActivateDirectMode) && mCameraButtonPressed && mCameraButtonClickFocused;
  bool canActivateGameMode = mCameraButtonClickUp && mCameraButtonClickFocused;
  
  if (canActivateDirectMode)
    mCamera->SetMode(PokerCameraModel::CAMERA_DIRECT_MODE);
  else if (canActivateGameMode)
    mCamera->SetMode(PokerCameraModel::CAMERA_ENTER_MODE);
  return (mCamera->ModeChanged());
}

void PokerPlayerCamera::BeginFreeMode()
{
  mCamera->ConsumeMode();
  mCameraTimeout = 0.0f;
  mCameraTotalTimeout = 0.0f;
  mCameraLookCardState = 0;
  mLookingCardAnimationTimeout = 0.0f;
  mMouseDelta[0] = 0.0f;
  mMouseDelta[1] = 0.0f;
  mCamera->SetRespondToEvents(true);
}

void PokerPlayerCamera::ExecuteFreeMode()
{
  if (mCamera->ModeChanged())
    BeginFreeMode();
  
  CameraEvaluateModeTransition();
  
  if (mCamera->ModeChanged())
    EndFreeMode();
}

void PokerPlayerCamera::EndFreeMode()
{
  //mGame->mPreviousCamera=*(mCamera->GetModel());
  MoveCameraToCamLookModel();
}

void PokerPlayerCamera::BeginEnterMode()
{
  mCamera->ConsumeMode();
}

void PokerPlayerCamera::ExecuteEnterMode()
{
  if (mCamera->ModeChanged())
    BeginEnterMode();

  mCamera->SetMode(PokerCameraModel::CAMERA_GAME_MODE);

  if (mCameraTimeout > 0)
    mCameraTimeout-=mDeltaS;

  if (mLookingCardAnimationTimeout > 0)
    mLookingCardAnimationTimeout-=mDeltaS;
  
  if (mCameraTimeout<=0 && mLookingCardAnimationTimeout <= 0)
      mCamera->SetMode(PokerCameraModel::CAMERA_GAME_MODE);
  
  //if (!mDisplayedAndFocused && mCameraButtonClickUp)
  // mCamera->SetMode(PokerCameraModel::CAMERA_LEAVE_MODE);

  if (mCamera->ModeChanged())
    EndEnterMode();
}
void PokerPlayerCamera::EndEnterMode()
{

}

void PokerPlayerCamera::BeginGameMode()
{
  mCamera->ConsumeMode();

  mMouseDeltaOverall[0] = 0.0f;
  mMouseDeltaOverall[1] = 0.0f;

  mCameraTimeout=mCameraGameModeDelay;

  mLookingCardAnimationTimeout = 0;

  float constraint[4] = {-40.0f, 40.0f, -20.0f, 30.0f};
  mCamera->SetConstraint(constraint);
}

void PokerPlayerCamera::ExecuteGameMode()
{
  if (mCamera->ModeChanged())
    BeginGameMode();
  
  if (mCameraTimeout > 0)
    mCameraTimeout-=mDeltaS;
  if (mLookingCardAnimationTimeout > 0)
    mLookingCardAnimationTimeout-=mDeltaS;

  bool clickOnCard = mDisplayedAndFocused && mCameraButtonClickUp && mLookingCardAnimationTimeout <= 0;
  if (clickOnCard)
    {
      if (!mCameraLookCardState)
	{
	  mCameraLookCardState = 1;
	  mCameraTimeout=mCameraGameModeDelay;
	}
      else
	{
	  mCameraLookCardState=0;
	  mCameraTimeout = mCameraGameModeDelay-mCameraGameModeDelayLookCards;
	}
    }
  if (mCameraMouseMotion)
    {
      mCameraTimeout=mCameraGameModeDelay;
    }
  
  bool lookCardTimeouted = mCameraLookCardState 
    && (mCameraTimeout<(mCameraGameModeDelay-mCameraGameModeDelayLookCards));
  if (lookCardTimeouted)
    {
      mCameraLookCardState=0;
    }
  float absXDelta = mMouseDelta[0] > 0 ? mMouseDelta[0] : -mMouseDelta[0];
  float absYDelta = mMouseDelta[1] > 0 ? mMouseDelta[1] : -mMouseDelta[1];
  float downLimit = mCameraGameModeMouseThreshold/100.f;

  bool deltaIsMin = absXDelta<downLimit && absYDelta<downLimit;
  bool leaveMode = (mCameraTimeout<0)
    || (mInteractorSelected && mCameraButtonClickUp) 
    || (!mDisplayedAndFocused && mCameraButtonClickUp && deltaIsMin);
  
  if (leaveMode)
    mCamera->SetMode(PokerCameraModel::CAMERA_LEAVE_MODE);

  if (mCamera->ModeChanged())
    EndGameMode();
}

void PokerPlayerCamera::EndGameMode()
{
  mCamera->UnsetConstraint();
  mCameraTimeout = 0.0f;
}

void PokerPlayerCamera::BeginDirectMode()
{
  mCamera->ConsumeMode();
  mCamera->SetRespondToEvents(false);
}
void PokerPlayerCamera::ExecuteDirectMode()
{
  if (mCamera->ModeChanged())
    BeginDirectMode();
  if (mCameraTimeout > 0)
    mCameraTimeout-=mDeltaS;

  if (!mCameraButtonPressed)
    mCamera->SetMode(PokerCameraModel::CAMERA_LEAVE_MODE);

 
  if (mCamera->ModeChanged())
    EndDirectMode();
}

void PokerPlayerCamera::EndDirectMode()
{
  mCamera->SetRespondToEvents(true);
}

void PokerPlayerCamera::BeginLeaveMode()
{
  mCamera->ConsumeMode();
  MoveCameraToPreviousModel();
}

void PokerPlayerCamera::ExecuteLeaveMode()
{
  if (mCamera->ModeChanged())
    BeginLeaveMode();

  if (mCameraTimeout)
    mCameraTimeout-=mDeltaS;

  if (mLookingCardAnimationTimeout > 0)
    mLookingCardAnimationTimeout-=mDeltaS;
    
  //bool canMoveToLookModel = (mLookingCardAnimationTimeout <= 0) && (CameraEvaluateModeTransition());
  //if (canMoveToLookModel)
  //  MoveCameraToCamLookModel();
  
  if (mCameraTimeout<=0 && mLookingCardAnimationTimeout <= 0)
    mCamera->SetMode(PokerCameraModel::CAMERA_FREE_MODE);

    
  if (mCamera->ModeChanged())
    EndLeaveMode();
}

void PokerPlayerCamera::EndLeaveMode()
{
}

void PokerPlayerCamera::updatePos(float _elapsed)
{
	int i;

	for (i = 0; i < 9; i++) {
		float srcX, srcY, destX, destY;
		srcX = g_HUDPos_Src[i].x();
		srcY = g_HUDPos_Src[i].y();
		destX = g_HUDPos_Dst[i].x();
		destY = g_HUDPos_Dst[i].y();

		float fac = factor_[i];
		hudPos_[i].x() = srcX + (destX - srcX) * fac;
		hudPos_[i].y() = srcY + (destY - srcY) * fac;

		if (sens_ == 0)
			fac += (1 - fac) * _elapsed;
		else
			fac -= fac * _elapsed;

		if (fac < 0) fac = 0;
		else if (fac > 1) fac = 1;

		factor_[i] = fac;
	}

	for (i = 0; i < 9; i++) {
		float x0 = hudPos_[i].x();
		float y0 = hudPos_[i].y();

		float rX0 = (x0 / 800) * 2 - 1;
		float rY0 = (y0 / 600) * 2 - 1;

		osg::Matrix mat = osg::Matrix::translate(rX0, -rY0, 0);
		mt_[i]->setMatrix(mat);
	}
	
}

