/*
*
* Copyright (C) 2004 Mekensleep
*
*	Mekensleep
*	24 rue vieille du temple
*	75004 Paris
*       licensing@mekensleep.com
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*
* Authors:
*  Igor Kravtchenko <igor@obraz.net>
*
*/

#ifndef _pokercardprojector_h
#define _pokercardprojector_h

#include <map>
#include <vector>

#include <maf/model.h>
#include <maf/view.h>
#include <maf/controller.h>

#include <PokerCard.h>
#include <PokerLabel.h>
#include <osg/Transform>
#include <osg/Timer>
#include <osg/AutoTransform>
#include <osg/ShapeDrawable>


class PokerApplication;

#define BOARD_BUFFER_ANGLE 50

namespace PokerCardProjectorHelper
{
  template <class T> class PokerSampleValue
    {
      T mSamples[BOARD_BUFFER_ANGLE];
      int mPreviousSample;
      float mDelay,mSampleRate;
      float mInternalTimer;
      
    public:
      
      PokerSampleValue() : mPreviousSample(0), mDelay(0), mSampleRate(0) , mInternalTimer(0) {}
      
      void SetDelay(float t) { mDelay=t;}
      void SetSampleRate(float t) { mSampleRate=t;}
      
      void GetSample(T& result) const {
	float v=mInternalTimer-mDelay;
	if (v<0)
	  v=0;
	
	int res=(int)floor(v/mSampleRate);
	float frac=v/mSampleRate-res;
	res=res%BOARD_BUFFER_ANGLE;
	result=mSamples[res];
	result+=(mSamples[(res+1)%BOARD_BUFFER_ANGLE]-result)*frac;
      }
      
      void SetSample(T& sample) {
	int res=(int)floorf(mInternalTimer/mSampleRate);
	//     printf("mInternalTimer %f - mSampleRate %f\n",mInternalTimer,mSampleRate);
	int start=(int)ceilf(mPreviousSample);
	if (start>res)
	  res=start;
	for (int i=start;i<=res;i++)
	  mSamples[i%BOARD_BUFFER_ANGLE]=sample;
	mPreviousSample=res;
      }

      void Update(float dt) { mInternalTimer+=dt;}
    };

  template <> class PokerSampleValue<osg::Quat>
    {
      osg::Quat mSamples[BOARD_BUFFER_ANGLE];
      int mPreviousSample;
      float mDelay,mSampleRate;
      float mInternalTimer;

    public:
      PokerSampleValue() : mPreviousSample(0), mDelay(0), mSampleRate(0) , mInternalTimer(0) {}

      void SetDelay(float t) { mDelay=t;}
      void SetSampleRate(float t) { mSampleRate=t;}

      void GetSample(osg::Quat& result) const {
	float v=mInternalTimer-mDelay;
	if (v<0)
	  v=0;

	int res=(int)floor(v/mSampleRate);
	float frac=v/mSampleRate-res;
	res=res%BOARD_BUFFER_ANGLE;
	//     result=mSamples[res];
	//     result+=(mSamples[(res+1)%BOARD_BUFFER_ANGLE]-result)*frac;
	g_debug("get %d",res);
	result.slerp(frac,mSamples[res],mSamples[(res+1)%BOARD_BUFFER_ANGLE]);
      }
      void SetSample(osg::Quat& sample) {
	int res=(int)floorf(mInternalTimer/mSampleRate);
	int start=(int)ceilf(mPreviousSample);
	if (start>res)
	  res=start;
	g_debug("set start %d end %d",start,res);
	for (int i=start;i<=res;i++)
	  mSamples[i%BOARD_BUFFER_ANGLE]=sample;
	mPreviousSample=res;
      }

      void Update(float dt) { mInternalTimer+=dt;}

    };
}

class PokerCardProjector : public MAFController
{
	osg::ref_ptr<osg::MatrixTransform> mBoardCenter;
	osg::ref_ptr<osg::MatrixTransform> mTextAnchor;
	osg::Matrix mInitialMatrix;
	std::vector<osg::ref_ptr<osg::MatrixTransform> > mTransformCards;
	std::vector<osg::Matrix> mMatrixOriginal;
	std::vector<osg::ref_ptr<osg::MatrixTransform> > mTransformTransitions;
	std::vector<osg::ref_ptr<osg::MatrixTransform> > mTransformTransitionTranslations;
	std::vector<osg::ref_ptr<osg::MatrixTransform> > mTransformTransitionCards;

	PokerCardProjectorHelper::PokerSampleValue<osg::Vec3> mSamples;

	float mParamAnimationDuration;
	float mTransitionTime;
	osg::Vec3 mTargetPos[5];


	bool mStartToDisplayHandValue;
	float mOffsetCard;
	float mParamYoffsetForText;
	float mScale;

	osg::Vec4 mParamShadeColorForCardUsed;

	//   BezierInterpolator<float> mInterpolator;
	//   void LoadKeys(std::vector<osg::Vec2> &keys, MAFXmlData *data, const std::string &name);
	//   template<typename T> void LoadSpline(T &interpolator, MAFXmlData *data, const std::string &name) {
	//     std::vector<osg::Vec2> keys;
	//     LoadKeys(keys, data, name);
	//     interpolator.InitKeys(keys);
	//   }

public:

	PokerCardControllerVector mCards;
	PokerApplication* mGame;
	PokerLabelController* mLabel;

	osg::AutoTransform* mRoot;

	PokerCardProjector(PokerApplication* game);
	virtual ~PokerCardProjector();

	void SetHandValue(const std::string& str) { if(str != "") mLabel->GetView()->SetText(str);}
	void SetTextPosition(const osg::Vec3& pos) { mLabel->GetView()->SetCenter(pos); }
	void SetTextColor(const osg::Vec4& a) { mLabel->GetView()->SetForegroundColor(a);}
	bool StopToDisplayShowDown();
	void SetCards(const std::vector<int>& cards);
	void FoldCards(void);
	bool StartToDisplayShowDown();
	virtual bool Update(MAFApplication* application);


	osg::ref_ptr<osg::AutoTransform> mAutoHandValueSpace;
	osg::ref_ptr<osg::MatrixTransform> mHandValueSpace[5];
	osg::ref_ptr<osg::MatrixTransform> mHandValueScaleSpace[5];
	osg::ref_ptr<osg::MatrixTransform> mTranslateForScale[5];
	osg::ref_ptr<osg::MatrixTransform> mCardPosition[5];
	osg::ref_ptr<osg::MatrixTransform> mTransformTranslations[5];
	osg::ref_ptr<osg::MatrixTransform> mHandValueTranslations[5];
	osg::ref_ptr<osg::MatrixTransform> mHandValueTransform[5];

	bool mChangeSpace;
	bool mChangeSpaceFinal;
	bool mActivateFadein;
	float mTransitionTranslationTime;
	float mParamTransitionTranslationDuration;
	float mParamFadeInDuration;
	float mFadeInTime;

	osg::Vec3 mTargetPositionInHandSpace[5];
	osg::Vec3 mSourcePositionInHandSpace[5];

	void SetupCardInNewSpace(PokerCameraModel* camera);

	void ActionFromMainToTransition();
	void ActionFromHandvalueToMain();
	void ActionFromTransitionToHandvalue();

	void ShadeUsedCardForBestHand(const std::vector<int>& cards);
	void SetShadeOfCardsToDefault();

};
#endif
