/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Cedric Pinson <cpinson@freesheep.org>
 *  Loic Dachary <loic@gnu.org>
 *  Johan Euphrosine <johan@mekensleep.com>
 *
 */

#include "pokerStdAfx.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef WIN32
#include "config_win32.h"
#endif

#include <maf/window.h>

#include <osg/Matrix>
#include <osg/Quat>

#include <ugame/debug.h>

#ifndef WIN32
#	include "PokerApplication.h"
#	include "PokerCamera.h"
#	include "PokerError.h"
#endif // WIN32

#define NbKeyFrames 16

#define POKER_CAMERA_POSITION 0
#define POKER_CAMERA_TARGET 1

#define POKER_CAMERA_ANIMATION_COUNT 2

struct CamHelper
{
  static inline osg::Vec3 QuatGetUp(const osg::Quat &quat)
  {
    osg::Matrix mat;
    quat.get(mat);
    return osg::Vec3(MatrixGetUp(mat));
  }
  static inline osg::Vec3 QuatGetAt(const osg::Quat &quat)
  {
    osg::Matrix mat;
    quat.get(mat);
    return osg::Vec3(MatrixGetAt(mat));
  }
  static inline osg::Vec3 QuatGetRight(const osg::Quat &quat)
  {
    osg::Matrix mat;
    quat.get(mat);
    return osg::Vec3(MatrixGetRight(mat));
  }  
  static inline osg::Vec3 MatrixGetUp(const osg::Matrix &mat)
  {
    return osg::Vec3(mat(0, 1), mat(1, 1), mat(2, 1));
  }
  static inline osg::Vec3 MatrixGetAt(const osg::Matrix &mat)
  {
    return osg::Vec3(mat(0, 2), mat(1, 2), mat(2, 2));
  }
  static inline osg::Vec3 MatrixGetRight(const osg::Matrix &mat)
  {
    return osg::Vec3(mat(0, 0), mat(1, 0), mat(2, 0));
  } 
  static inline osg::Vec3 QuatRotate(const osg::Vec3 &in, const osg::Quat &quat)
  {
    osg::Quat V(osg::Vec4(in, 1.0f));
    osg::Quat rotatedPos;
    rotatedPos *= quat.conj();
    rotatedPos *= V;
    rotatedPos *= quat;
    return osg::Vec3(rotatedPos.asVec3());
  }
  static inline void QuatNormalize(osg::Quat &quat)
  {
    osg::Vec4 N = quat.asVec4();
    N.normalize();
    quat.set(N);
  }

  static inline osg::Quat QuatLookAt(const osg::Vec3 &position, const osg::Vec3 &target, const osg::Vec3 &up)
  {
    osg::Quat quat;
    quat.set(osg::Matrix::lookAt(position, target, up));
    return quat;
  }

  static inline osg::Quat QuatSlerp(double t, const osg::Quat& from, const osg::Quat& to, double scale = 1.0)
  {
    //    const double epsilon = 0.00001;
    double omega, cosomega, sinomega, scale_from, scale_to ;
    
    osg::Quat quatTo(to);
    // this is a dot product
    
    cosomega = from.asVec4() * to.asVec4();
    
    double phase = (scale - 1.0)*t;


    if ( cosomega <0.0 )
    { 
        cosomega = -cosomega; 
        quatTo = -to;
    }

    //    if( (1.0 - cosomega) > epsilon )
    //{
        omega= acos(cosomega) ;  // 0 <= omega <= Pi (see man acos)
        sinomega = sin(omega) ;  // this sinomega should always be +ve so
        // could try sinomega=sqrt(1-cosomega*cosomega) to avoid a sin()?
        scale_from = sin((1.0-t)*omega-phase)/sinomega ;
        scale_to = sin(t*omega+phase)/sinomega ;
	//}
  //else
  // {
        /* --------------------------------------------------
           The ends of the vectors are very close
           we can use simple linear interpolation - no need
           to worry about the "spherical" interpolation
           -------------------------------------------------- */
  //scale_from = 1.0 - t ;
  //     scale_to = t ;
  //  }
	
	return osg::Quat((from*scale_from) + (quatTo*scale_to));
	
	// so that we get a Vec4
  }
};


void PokerCameraModel::PointFunctorSin::process(int index, osg::Vec3& where, float& when) const {
  when = mTimeStep * index;
  float percent = (.5f * ( sin(- osg::PI_2 + ( float(index) / mPointCount) * osg::PI) + 1) );
  osg::Vec3 delta = mSegment * percent;
  if(mInverse == true)
    delta = mSegment - delta;
  where = mFrom + mSegment * percent;
}

//
// Where x is the percentage of the total time of the move
// (float(index)/mPointCount), 
// y is the percentage of the distance between the source
// point and the target, and each curve is for mFactor = 2 or 5 or 10
// 
//     1 ++------------+-------------+-------------+-------------+------------**
//       +             +             +             +             + x**2 *******$
//       |                                                         x**5 ###**##|
//       |                                                        x**10 $**$$#$|
//   0.8 ++                                                            **   # $+
//       |                                                           ***   # $ |
//       |                                                         **     #  $ |
//       |                                                       **      #  $  |
//   0.6 ++                                                    **       ## $  ++
//       |                                                  ***        #   $   |
//       |                                                **          ##  $    |
//       |                                             ***          ##   $     |
//   0.4 ++                                         ***            ##    $    ++
//       |                                       ***              #     $      |
//       |                                    ****             ###     $       |
//       |                                ****               ###      $$       |
//   0.2 ++                            ****               ####      $$        ++
//       |                        *****                 ###       $$$          |
//       |                   ******                #####       $$$$            |
//       +          *********        +     ########+      $$$$$$ +             +
//     0 ************######################$$$$$$$$$$$$$$$-------+------------++
//       0            0.2           0.4           0.6           0.8            1
//

void PokerCameraModel::PointFunctorPow::process(int index, osg::Vec3& where, float& when) const {
  when = mTimeStep * index;
  float percent = 0.f;
  if(index > 0)
    percent = powf(float(index) / mPointCount, mFactor);
    
  osg::Vec3 delta = mSegment * percent;
  if(mInverse == true)
    delta = mSegment - delta;
  where = mFrom + mSegment * percent;
}

void PokerCameraModel::PointFunctorHermite::process(int index, osg::Vec3& where, float &when) const
{
  when = mTimeStep * index;
  float percent = 0.f;
  if (index > 0)
    {
      float time = (float)index / (float)mPointCount;
      //float time = when / mDuration;
      mCurve.GetData(time,percent,false);
    }
  osg::Vec3 delta = mSegment * percent;
  where = mFrom + delta;
}


PokerCameraModel::PokerCameraModel() :
  mKeys(POKER_CAMERA_ANIMATION_COUNT),
  mSpline(POKER_CAMERA_ANIMATION_COUNT),
  mIsMoving(POKER_CAMERA_ANIMATION_COUNT),
  mRotateX(0.f),
  mRotateY(0.f),
  mNeedReadjust(false),
  mReadjusting(false),
  mReadjustTimeout(0.0f),
  mReadjustScale(1.0f),
  mMode(CAMERA_FREE_MODE),
  mModeChanged(false),
  mAnglePixelRatio(20.f)
{
  for(int i = 0; i < POKER_CAMERA_ANIMATION_COUNT; i++) {
    mKeys[i] = new KeyFrame[2*NbKeyFrames];
    mIsMoving[i] = false;
  }
}

PokerCameraModel::~PokerCameraModel()
{
  for(int i = 0; i < POKER_CAMERA_ANIMATION_COUNT; i++)
    delete [] mKeys[i];
}

void PokerCameraModel::StartInterpolation(float timeout)
{
  mTimer.Init(timeout);
  SetIsMoving(true);
}

void PokerCameraModel::SetupTargetInterpolator(const osg::Vec3 &target)
{  
  const osg::Vec3 &targetFrom = GetTarget();
  const osg::Vec3 &targetTo = target;
  mTargetBInterpolator.Init(targetFrom, targetTo);
}

void PokerCameraModel::SetupLengthInterpolator(const osg::Vec3 &position, const osg::Vec3 &target)
{
  const osg::Vec3 &positionFrom = GetPosition();
  const osg::Vec3 &positionTo = position;
  const osg::Vec3 &targetFrom = GetTarget();
  const osg::Vec3 &targetTo = target;
  float lengthFrom = (targetFrom - positionFrom).length();
  float lengthTo = (targetTo - positionTo).length();

  mLengthBInterpolator.Init(lengthFrom, lengthTo);
  mCamNextPosition = positionTo;
  mCamPrevPosition = positionFrom;
  mCamNextTarget = targetTo;
  mCamPrevTarget = targetFrom;
}

void PokerCameraModel::SetupFovInterpolator(float fov)
{
  mFovBInterpolator.Init(GetFov(), fov);
  mCamNextFov = fov;
  mCamPrevFov = GetFov();
}

void PokerCameraModel::SetupRotationInterpolator(const osg::Quat &attitude)
{
  const osg::Quat &attitudeFrom = mCamAttitude;
  const osg::Quat &attitudeTo = attitude;

  mRotationBInterpolator.Init(attitudeFrom, attitudeTo);

  mCamNextAttitude = attitudeTo;
  if (mReadjusting)
    {
      mReadjusting = false;
      mReadjustBInterpolator.Get(mCamPrevAttitude, 1.0f, 1.0f);
    }
  else    
    {
      mCamPrevAttitude = attitudeFrom;
    }
}


void PokerCameraModel::SetupReadjustInterpolator(const osg::Quat &attitude)
{
  const osg::Quat &attitudeFrom = mCamAttitude;
  const osg::Quat &attitudeTo = attitude;
  mReadjustBInterpolator.Init(attitudeFrom, attitudeTo);
}


void PokerCameraModel::SetupInterpolators(const osg::Vec3 &position, const osg::Vec3 &target, const osg::Vec3 &up, float fovTo, float timeout)
{
  osg::Quat attitude;
  attitude.set(osg::Matrix::lookAt(position, target, up));
  SetupInterpolators(position, target, attitude, fovTo, timeout);
}

void PokerCameraModel::SetupInterpolators(const osg::Vec3 &position, const osg::Vec3 &target, const osg::Quat& attitude, float fovTo, float timeout)
{
  const osg::Vec3 &positionFrom = GetPosition();
  const osg::Vec3 &positionTo = position;
  const osg::Vec3 &targetFrom = GetTarget();
  const osg::Vec3 &targetTo = target;
  float lengthFrom = (targetFrom - positionFrom).length();
  float lengthTo = (targetTo - positionTo).length();

  const osg::Quat &attitudeFrom = mCamAttitude;
  const osg::Quat &attitudeTo = attitude;

  mTargetBInterpolator.Init(targetFrom, targetTo);
  mLengthBInterpolator.Init(lengthFrom, lengthTo);
  if (mReadjusting)
    mReadjustBInterpolator.Init(attitudeFrom, attitudeTo);
  else
    mRotationBInterpolator.Init(attitudeFrom, attitudeTo);
  mFovBInterpolator.Init(GetFov(), fovTo);
  
  
  mCamPrevAttitude = attitudeFrom;
  if (mReadjusting)
    mRotationBInterpolator.Get(mCamPrevAttitude, 1.0f);
  mCamPrevPosition = positionFrom;
  mCamPrevTarget = targetFrom;  
  mCamPrevFov = GetFov();

  mTimer.Init(timeout);
  SetIsMoving(true);
}

void PokerCameraModel::Readjust()
{
  osg::Matrix mat;
  mCamAttitude.get(mat);
  osg::Vec3 eye, target, up;
  mat.getLookAt(eye, target, up);
  float angle = osg::PI * 1.0f;
  osg::Quat q(angle, GetTarget()-GetPosition());
  //up = CamHelper::QuatRotate(up, q);
  mReadjusting = true;
  //mReadjustScale = 1.5f;

  SetupReadjustInterpolator(CamHelper::QuatLookAt(GetPosition(), GetTarget(), -up));
  float timeout = 1250.0f;
  StartInterpolation(timeout);
}

bool PokerCameraModel::GetIsMoving(void) {
  for(int i = 0; i < POKER_CAMERA_ANIMATION_COUNT; i++) {
    if(mIsMoving[i] == true)
      return true;
  }
  return false;
}

void PokerCameraModel::SetIsMoving(bool isMoving) {
  for(int i = 0; i < POKER_CAMERA_ANIMATION_COUNT; i++)
    mIsMoving[i] = isMoving;
}

void PokerCameraModel::Update(float delta) {

  if (mTimer.Finished())
    {
      SetIsMoving(false);
      if (mReadjusting)
	  mReadjusting = false;
      return;
    }

  //else
  mTimer.AddTime(delta);
  
  osg::Quat current;

  if (mReadjusting)
    {
      mTimer.Get(current, mReadjustBInterpolator, mReadjustScale);
      osg::Matrix mat;
      current.get(mat);
      osg::Vec3 eye, center, up;
      mat.getLookAt(eye, center, up);
      SetUp(up);
      mCamAttitude = current;
      return;
    }
  
  mTimer.Get(current, mRotationBInterpolator);
  
  osg::Vec3 target;  
  mTimer.Get(target, mTargetBInterpolator);
  float length;
  mTimer.Get(length, mLengthBInterpolator);
  float fov;
  mTimer.Get(fov, mFovBInterpolator);
  
  osg::Matrix mat;
  current.get(mat);
  osg::Vec3 eye, center, up;
  mat.getLookAt(eye, center, up);
  osg::Vec3 at;
  at = (center - eye);
  at.normalize();
  
  // set LookAt
  SetPosition(target - at * length);
  SetTarget(target);
  SetUp(up);
  SetFov(fov);
  
  mCamAttitude = current;
  return;
}

void PokerCameraModel::LoadKeys(std::vector<osg::Vec2> &keys, MAFXmlData *data, const std::string &name)
{
  if (data != NULL)
  {
    const std::list<std::string> &xResultList = data->GetList("/splines/" + name + "/list/entry/@xvalue");
    const std::list<std::string> &yResultList = data->GetList("/splines/" + name + "/list/entry/@yvalue");
    
    g_assert(xResultList.size() == yResultList.size());
    typedef std::list<std::string>::const_iterator It;
    It xbegin = xResultList.begin();
    It xend = xResultList.end();
    
    It ybegin = yResultList.begin();
    It yend = yResultList.end();
    
    while(xbegin != xend) {
      keys.push_back(osg::Vec2(atof((*xbegin).c_str()), atof((*ybegin).c_str())));
      xbegin++;
      ybegin++;
   	}
  }
}

// controller

PokerCameraController::PokerCameraController(PokerApplication* game) : mRespondToEvents(true), mGame(game) , mCurrent(1) , mApplicationStealFocus(false) {
  SetModel(new PokerCameraModel());
  UnsetConstraint();
  Init();
}

void PokerCameraController::Init(void) {
  MAFCameraController::Init();


  osgUtil::SceneView* scene = mGame->GetScene()->GetModel()->mScene.get();
  osg::Matrix mat = scene->getViewMatrix();
  GetModel()->mCamAttitude.set(mat);
  GetModel()->mCamPosition = GetModel()->GetPosition();
  GetModel()->mFirstTime = true; // HACK
  
  const std::string &url = mGame->HeaderGet("sequence", "/sequence/cameras/cameraspline/@url");
  const std::string &rotate = mGame->HeaderGet("sequence", "/sequence/cameras/cameraspline/@rotate");
  const std::string &target = mGame->HeaderGet("sequence", "/sequence/cameras/cameraspline/@target");
  const std::string &length = mGame->HeaderGet("sequence", "/sequence/cameras/cameraspline/@length");
  const std::string &readjust = mGame->HeaderGet("sequence", "/sequence/cameras/cameraspline/@readjust");

  MAFXmlData *data = mGame->mDatas->GetXml(url);
  GetModel()->LoadSpline(GetModel()->mRotationBInterpolator, data, rotate);
  GetModel()->LoadSpline(GetModel()->mReadjustBInterpolator, data, readjust);
  float value = GetModel()->mReadjustBInterpolator.GetT(1.0f);
  GetModel()->mReadjustScale = 1.0f / value;

  GetModel()->LoadSpline(GetModel()->mTargetBInterpolator, data, target);
  GetModel()->LoadSpline(GetModel()->mLengthBInterpolator, data, length);
  GetModel()->LoadSpline(GetModel()->mFovBInterpolator, data, length);

  const std::string &anglePixelRatio = mGame->HeaderGet("sequence", "/sequence/cameras/@anglePixelRatio");
  if(anglePixelRatio != "")
    GetModel()->SetAnglePixelRatio(atof(anglePixelRatio.c_str()));
}

void PokerCameraController::Rotate(float dx, float dy)
{
  switch(GetMode())
    {
    case PokerCameraModel::CAMERA_FREE_MODE:
      RotateFreeMode(dx, dy);
      break;
    case PokerCameraModel::CAMERA_GAME_MODE:
      RotateGameMode(dx, dy);
      break;
    default:
      break;
    }
}

void PokerCameraController::RotateFreeMode(float dx, float dy)
{
  const osg::Vec3 &positionFrom = GetModel()->GetPosition();
  const osg::Vec3 &targetFrom = GetModel()->GetTarget();

  osg::Vec3 atFrom = positionFrom - targetFrom;
  atFrom.normalize();

  osg::Vec3 up = osg::Vec3(0.0f, 1.0f, 0.0f);

  float lengthFrom = (positionFrom - targetFrom).length();
  float factor = 1.0f;
  float angleFrom = acosf(atFrom * up);


  const osg::Quat &attitudeFrom = GetModel()->mCamAttitude;
  osg::Matrix matAttitudeFrom;
  attitudeFrom.get(matAttitudeFrom);
  osg::Vec3 eyeFrom, centerFrom, upFrom;
  matAttitudeFrom.getLookAt(eyeFrom, centerFrom, upFrom);
  

  float signFrom = (upFrom * up >= 0 ? 1.0f : -1.0f);
  angleFrom = angleFrom * signFrom;

  float anglePixelRatio = GetModel()->GetAnglePixelRatio();
  float ratioX = mGame->GetWindow(true)->GetWidth() / anglePixelRatio;
  float ratioY = mGame->GetWindow(true)->GetHeight() / anglePixelRatio;
  float drotateX = (osg::PI_4 * dx) / ratioX;
  float drotateY = (osg::PI_4 * dy * factor) / ratioY;
  
  if ((angleFrom > 0.0f) && (dy < 0) && ((angleFrom - drotateY) >= osg::PI_2))
    {
      float delta = -(osg::PI_2 - angleFrom);
      drotateY  =  (delta > 0.0f) ? delta : 0.0f;
    }
  if ((angleFrom < 0.0f) && (dy > 0) && ((angleFrom - drotateY) <= -osg::PI_2))
    {
      float delta = -(-osg::PI_2 - angleFrom);
      drotateY  =  (delta > 0.0f) ? delta : 0.0f;
    }
  
  osg::Quat attitudeTo = attitudeFrom;

  osg::Quat qX(drotateX, up);
  attitudeTo = qX * attitudeTo;

  osg::Vec3 right = CamHelper::QuatGetRight(attitudeTo);
  osg::Quat qY(drotateY, right);
  attitudeTo = qY * attitudeTo;
  
  CamHelper::QuatNormalize(attitudeTo);
 
  osg::Matrix matAttitudeTo;
  attitudeTo.get(matAttitudeTo);
  osg::Vec3 eyeTo, centerTo, upTo;
  matAttitudeTo.getLookAt(eyeTo, centerTo, upTo);
  osg::Vec3 atTo = eyeTo - centerTo;
  atTo.normalize();
 
  const osg::Vec3 &targetTo = targetFrom;
  float lengthTo = lengthFrom;

  osg::Vec3 positionTo = targetTo + atTo * lengthTo;


  float angleTo = acosf(atTo * up);
  float signTo = (upTo * up >= 0 ? 1.0f : -1.0f);
  angleTo = angleTo * signTo;
  if (angleTo < -osg::PI_2/4.0f)
    {
      GetModel()->mNeedReadjust = true;
      GetModel()->mReadjustTimeout = 200.0f;
    }
  else
    GetModel()->mNeedReadjust = false;

  GetModel()->mCamAttitude = attitudeTo;
  GetModel()->SetPosition(positionTo);
  GetModel()->SetUp(upTo);
}

void PokerCameraController::RotateGameMode(float dx, float dy)
{
  ApplyConstraint(dx, dy);
  osg::Vec3 at = GetModel()->GetPosition() - GetModel()->GetTarget();
  at.normalize();
  osg::Vec3 uptmp(0,1,0);

  osg::Vec3 up_down_axis = at ^ uptmp;
  osg::Matrix rotate(osg::Matrix::rotate((osg::PI_4 / 200.f) * -dx, uptmp[0], uptmp[1],uptmp[2]) * 
		     osg::Matrix::rotate((osg::PI_4 / 200.f) * dy, up_down_axis));
  GetModel()->SetPosition(GetModel()->GetPosition() * rotate);
  GetModel()->SetUp(uptmp);      
}

bool PokerCameraController::Update(MAFApplication* application) {  
  if (application->GetFocus() == mGame->GetInterface())    
    {
      SetRespondToEvents(false);
      mApplicationStealFocus = true;
    }
  else if (mApplicationStealFocus)
    {
      SetRespondToEvents(true);
      mApplicationStealFocus = false;
    }
      
  SDL_Event*	event = application->GetLastEvent(this);
  //init
  if (GetModel()->mFirstTime)
    {
	const osg::Vec3 &eye = GetModel()->GetPosition();
	const osg::Vec3 &center = GetModel()->GetTarget();
	osg::Vec3 up(0.0f,1.0f,0.0f);
 	osg::Matrix mat;
	mat.makeLookAt(eye, center, up);
	GetModel()->mCamAttitude.set(mat);
 	GetModel()->mFirstTime = false;
    }
  if(GetModel()->GetIsMoving()) {
    GetModel()->Update(GetDelta());
  } else if(GetRespondToEvents() && event) {
    int zoom = 0;
    GetModel()->mReadjustTimeout = 200.0f;
    
    if(event->type == SDL_MOUSEMOTION && (SDL_GetMouseState(NULL, NULL) & SDL_BUTTON(1))) {
      
      float dx = event->motion.xrel * mCurrent;
      float dy = event->motion.yrel;
      Rotate(dx, dy);
    } else if(event->type == SDL_MOUSEBUTTONDOWN) {
      mCurrent = (event->motion.y < (application->GetWindow(true)->GetHeight()/2)) ? -1 : 1;
      switch (event->button.button) {
      case SDL_BUTTON_WHEELUP:
	zoom = -1;
	break;
      case SDL_BUTTON_WHEELDOWN:
	zoom = 1;
	break;
      default:
	break;
      }
    } else if (event->type == SDL_KEYDOWN) {
      switch(event->key.keysym.sym) {
      case SDLK_UP:
	zoom = -1;
	break;
      case SDLK_DOWN:
	zoom = 1;
	break;
      default:
	break;
      }
    }
    if(zoom != 0) {
      osg::Vec3 offset = GetModel()->GetPosition() - GetModel()->GetTarget();
      offset.normalize();
      offset *= zoom * 10.f;
      
      const osg::Vec3 &position = GetModel()->GetPosition();
      const osg::Vec3 &target = GetModel()->GetTarget();
      osg::Vec3 delta = position - target;
      float length = delta.length();
      if (zoom < 0 && length < 10.0f)
	offset = osg::Vec3(0.0f, 0.0f, 0.0f);

      GetModel()->SetPosition(GetModel()->GetPosition() + offset);
    }
  } else if (GetModel()->mNeedReadjust)
    {
      // camera idle
      float timeout = GetModel()->mReadjustTimeout;
      
      if (timeout > 0.0f)
	timeout -= GetDelta();
      if (timeout <= 0.0f)
	{
	  timeout = 0.0f;
	  if (GetMode() == PokerCameraModel::CAMERA_FREE_MODE)
	    {
	      GetModel()->Readjust();
	      GetModel()->mNeedReadjust = false;
	    }
	}
      GetModel()->mReadjustTimeout = timeout;
    }
  return true;
}

void PokerCameraController::MoveTo(const osg::Vec3& destination, const PokerCameraModel::PointFunctor& pointGenerator) {
  g_assert(false);
}

void PokerCameraController::MoveTo(const MAFCameraModel& target, 
				   const PokerCameraModel::PointFunctor& positionPointGenerator)
{
  g_assert(false);
}

void PokerCameraController::MoveTo(const MAFCameraModel& target, 
	    const PokerCameraModel::PointFunctor& positionPointGenerator,
	    const PokerCameraModel::PointFunctor& targetPointGenerator)
{
  g_assert(false);
}

void PokerCameraController::MoveTo(const osg::Vec3 &position, const osg::Vec3 &target, float fov, float timeout)
{
  GetModel()->SetupTargetInterpolator(target);

  GetModel()->SetupLengthInterpolator(position, target);

  GetModel()->SetupFovInterpolator(fov);

  osg::Vec3 up = osg::Vec3(0.0f, 1.0f, 0.0f);
  GetModel()->SetupRotationInterpolator(CamHelper::QuatLookAt(position, target, up));

  GetModel()->StartInterpolation(timeout);
}

void PokerCameraController::MoveToPrevious(float timeout)
{
  const osg::Vec3 &target = GetModel()->mCamPrevTarget;
  const osg::Vec3 &position = GetModel()->mCamPrevPosition;
  float fov = GetModel()->mCamPrevFov;
  const osg::Quat &attitude = GetModel()->mCamPrevAttitude;

  GetModel()->SetupTargetInterpolator(target);

  GetModel()->SetupLengthInterpolator(position, target);

  GetModel()->SetupFovInterpolator(fov);

  GetModel()->SetupRotationInterpolator(attitude);

  GetModel()->StartInterpolation(timeout);
}

void PokerCameraController::ApplyConstraint(float &dx, float &dy)
{
  // Move Constraint in MODEL
  dx /= 5.0f;
  dy /= 5.0f;
  float deltaX = mDeltaCurrent[0] + dx;
  float deltaY = -(mDeltaCurrent[1] + dy);

  float dxMax;
  float dyMax;
  if (deltaX > 0)
    dxMax = mDeltaConstraint[1];
  else
    dxMax = mDeltaConstraint[0];
  if (deltaY > 0)
    dyMax = mDeltaConstraint[3];
  else
    dyMax = mDeltaConstraint[2];
      
  float dxN = (dxMax != 0.0f ? MIN(deltaX/dxMax, 1.0f) : 1.0f);
  float dyN = (dyMax != 0.0f ? MIN(deltaY/dyMax, 1.0f) : 1.0f);
  dx *= (1.0f - dxN);
  dy *= (1.0f - dyN);

  mDeltaCurrent[0] += dx;
  mDeltaCurrent[1] += dy;
}

void PokerCameraController::SetConstraint(float (&constraint)[4])
{
  mConstraintSet = true;
  mDeltaConstraint[0] = constraint[0];
  mDeltaConstraint[1] = constraint[1];
  mDeltaConstraint[2] = constraint[2];
  mDeltaConstraint[3] = constraint[3];
  mDeltaCurrent[0] = 0.0f;  
  mDeltaCurrent[1] = 0.0f;
}

void PokerCameraController::UnsetConstraint()
{
  mConstraintSet = false;
  mDeltaConstraint[0] = 0.0f;
  mDeltaConstraint[1] = 0.0f;
  mDeltaConstraint[2] = 0.0f;
  mDeltaConstraint[3] = 0.0f;
  mDeltaCurrent[0] = 0.0f;  
  mDeltaCurrent[1] = 0.0f;
}

void PokerCameraController::SetMode(PokerCameraModel::Mode mode)
{
  GetModel()->SetMode(mode);
}
PokerCameraModel::Mode PokerCameraController::GetMode() const
{
  return GetModel()->GetMode();
}
void PokerCameraController::ConsumeMode()
{
  GetModel()->SetModeChanged(false);
}
bool PokerCameraController::ModeChanged() const
{
  return GetModel()->GetModeChanged();
}
