/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Johan Euphrosine <johan@mekensleep.com>
 *  Cedric Pinson <cpinson@freesheep.org>
 *  Igor Kravtchenko <igor@obraz.net>
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#ifdef WIN32
#include "config_win32.h"
#include <cstdio>
#endif

#include "pokerStdAfx.h"

#include <iostream>

#include <cal3d/scheduler.h>

#include <osg/Material>
#include <osg/Depth>
#include <osg/Texture2D>
#include <osg/BlendFunc>

#include <osgCal/SubMeshSoftware>
#include <osgCal/SubMeshHardware>

#include <maf/application.h>
#include <maf/utils.h>
#include <maf/hit.h>
#include <maf/MultipleAnimationPathCallback.h>

#include <PokerBody.h>
#include <PokerError.h>
#include <PokerCard.h>
#include <PokerApplication.h>
#include <PokerFoldAnimation.h>

#ifdef WIN32
# define snprintf _snprintf
#endif
// Noise

// return the frac part of a real
inline double Frac(double _t){return _t-floor(_t);}


double NoiseElement::Noise(double _t,float _amp,float _freq) 
{
  //    double amplitude = 1.0;
  double result = 0.0;
  double freq=1.0;
  double x=_t;
  double persistance;
  for (int i=0; i<_freq; i++) {
    freq=(1<<i);
    persistance=pow(_amp,i);
    result += mNoise.noise(x*freq) * persistance;
  }
  return result;
}

double NoiseElement::NoiseFunction(double _t) 
{
  double t =Frac(_t);
  const double loop=1.0;
  const int freq=2;
  const float amp=0.5;
  double noise=(loop-t)*Noise(t,amp,freq)+ t*Noise(t-loop,amp,freq);
  noise/=loop;
  return noise;
}


CalCoreBone* NoiseElement::GetCoreBone(int boneId) {
  CalCoreModel* coreModel = mModel->getCoreModel();
  g_assert(coreModel != 0);
  g_assert(coreModel->getCoreSkeleton() != 0);
  CalCoreBone* bone = coreModel->getCoreSkeleton()->getCoreBone(boneId);
  g_assert(bone != 0);
  return bone;
}

void NoiseElement::CreateCoreAnimation(const std::string& path, std::list<std::string>& bones) 
{
  CalCoreModel* coreModel = mModel->getCoreModel();
  //
  // Create animation from template containing enough tracks for each bone
  //
  mCoreAnimationId = coreModel->loadCoreAnimation(mDatadir + "/" + path);
  if(mCoreAnimationId < 0)
    g_error("NoiseElement::CreateCoreAnimation: could not load %s", path.c_str());
  mCoreAnimation = coreModel->getCoreAnimation(mCoreAnimationId);
  g_assert(mCoreAnimation != 0);

  //
  // Check that there exists exactly one track for each bone
  //
  if(bones.size() != mCoreAnimation->getListCoreTrack().size())
    g_error("NoiseElement::CreateCoreAnimation: %s has contains %d tracks but expected exactly %d track", path.c_str(), bones.size(), mCoreAnimation->getListCoreTrack().size());

  std::list<std::string>::iterator bone = bones.begin();
  for(std::list<CalCoreTrack *>::iterator coreTrack = mCoreAnimation->getListCoreTrack().begin();
      coreTrack != mCoreAnimation->getListCoreTrack().end();
      coreTrack++, bone++) {
    int boneId = coreModel->getCoreSkeleton()->getCoreBoneId(*bone);
    if(boneId < 0)
      g_error("NoiseElement::CreateCoreAnimation: in %s, boneId of %s not found", path.c_str(), bone->c_str());
    (*coreTrack)->setCoreBoneId(boneId);
  }
}

NoiseElement::NoiseElement(CalModel* model, const std::string&  datadir) : mCoreAnimation(0), mCoreAnimationId(-1), mKeepGoing(true), mModel(model), mDatadir(datadir)
{ 
}

class NoiseEyes : public NoiseElement
{
 public:

  NoiseEyes(CalModel* model, const std::string& datadir) : NoiseElement(model, datadir) {
    std::list<std::string> bones;
    bones.push_back("boneEyeL");
    bones.push_back("boneEyeR");
    CreateCoreAnimation("noiseeyes.xaf", bones);
  }

  
  void process(CalModel* model, CalAnimationAlt* animation) {
    if(!mKeepGoing)
      return;

    CalScheduler* scheduler = (CalScheduler*)(model->getAbstractMixer());

    const double maxAngle=25*3.14/180.0;

    std::list<CalCoreTrack*>& tracks = mCoreAnimation->getListCoreTrack();
    int size = tracks.front()->getCoreKeyframeCount();
    double res = NoiseFunction(time(0)/20.0);

    osg::Quat quat;
    if (res>maxAngle)
      res=maxAngle;
    if (res<-maxAngle)
      res=-maxAngle;
    quat.makeRotate(res, osg::Vec3(0,1,0));
//     quat.makeRotate(res, osg::Vec3(0,0,1)); // on demo version axis to rotate eyes are pivoted
    CalQuaternion q2(quat[0],quat[1],quat[2],quat[3]);
    for (int i=0;i<size/2;i++) {
      for(std::list<CalCoreTrack*>::iterator track = tracks.begin();
	  track != tracks.end();
	  track++) {
	CalCoreBone* bone = GetCoreBone((*track)->getCoreBoneId());
        (*track)->getCoreKeyframe(i)->setTranslation(bone->getTranslation());
        (*track)->getCoreKeyframe(i)->setRotation(q2);
      }
    }
    
    scheduler->run(CalScheduler::FOREGROUND,
		   mCoreAnimationId,
		   CalScheduler::ONCE)
      ->setStopCallback(this);
  }
};

class NoiseNose : public NoiseElement
{
 public:

  NoiseNose(CalModel* model, const std::string& datadir) : NoiseElement(model, datadir){
    std::list<std::string> bones;
    bones.push_back("boneNoseL");
    bones.push_back("boneNoseR");
    CreateCoreAnimation("noisenose.xaf", bones);
  }
  
  virtual void process(CalModel* model, CalAnimationAlt* animation) {
    if(!mKeepGoing)
      return;

    CalScheduler* scheduler = (CalScheduler*)(model->getAbstractMixer());
    std::list<CalCoreTrack*>& tracks = mCoreAnimation->getListCoreTrack();
    CalCoreTrack* track_left = tracks.front();
    CalCoreBone* bone_left = GetCoreBone(track_left->getCoreBoneId());
    CalCoreTrack* track_right = tracks.back();
    CalCoreBone* bone_right = GetCoreBone(track_right->getCoreBoneId());
    int size = tracks.front()->getCoreKeyframeCount();

    for (int i=0;i<size;i++) {
      double res=fabs(NoiseFunction(time(0)+i*1.0/size));
      CalVector t(0,0,res*2);
      track_left->getCoreKeyframe(i)->setTranslation(bone_left->getTranslation()+t);

      res=fabs(NoiseFunction(time(0)+0.5+i*0.1/size));
      t=CalVector(0,0,res*2);
      track_right->getCoreKeyframe(i)->setTranslation(bone_right->getTranslation()+t);
    }
    
    scheduler->run(CalScheduler::FOREGROUND,
		   mCoreAnimationId,
		   CalScheduler::ONCE,
		   1.f,
		   new CalScheduler::FadeInOut(.2f, .2f))
      ->setStopCallback(this);
  }
};

class NoiseSkull : public NoiseElement
{

  osg::Quat mInitialRotation;

 public:

  NoiseSkull(CalModel* model, const std::string& datadir) : NoiseElement(model, datadir){
    std::list<std::string> bones;
    bones.push_back("boneSkull");
    CreateCoreAnimation("noiseskull.xaf", bones);
    CalCoreTrack* track = mCoreAnimation->getListCoreTrack().front();
    CalCoreBone* bone = GetCoreBone(track->getCoreBoneId());
    CalQuaternion calq=track->getCoreKeyframe(0)->getRotation();
    mInitialRotation =osg::Quat(calq[0],calq[1],calq[2],calq[3]);
  }
  
  virtual void process(CalModel* model, CalAnimationAlt* animation) {
    if(!mKeepGoing)
      return;

    CalScheduler* scheduler = (CalScheduler*)(model->getAbstractMixer());
    CalCoreTrack* track = mCoreAnimation->getListCoreTrack().front();
    CalCoreBone* bone = GetCoreBone(track->getCoreBoneId());

    int size=track->getCoreKeyframeCount();
    for (int i=0;i<size;i++) {
      double res=NoiseFunction((time(0)+i*1.0/size));
      osg::Quat q0,q1;
      q0.makeRotate(res*0.4, osg::Vec3(0,1,0));
      double res2=NoiseFunction((time(0)+0.5+i*0.1/size));
      q1.makeRotate(res2*.05, osg::Vec3(1,0,0)); // 0.5
      q1=mInitialRotation*q1;
      q1*=q0;
      CalQuaternion q(q1[0],q1[1],q1[2],q1[3]);
      track->getCoreKeyframe(i)->setTranslation(bone->getTranslation());
      track->getCoreKeyframe(i)->setRotation(q);
    }
    
    scheduler->run(CalScheduler::FOREGROUND,
		   mCoreAnimationId,
		   CalScheduler::ONCE,
		   1.f,
		   new CalScheduler::FadeInOut(0.25f, 0.25f))
      ->setStopCallback(this);
  }
};

class NoiseMouth : public NoiseElement
{
 public:

  NoiseMouth(CalModel* model, const std::string& datadir) : NoiseElement(model, datadir){
    std::list<std::string> bones;
    bones.push_back("boneMouthBL");
    bones.push_back("boneMouthTL");
    bones.push_back("boneMouthCL");
    bones.push_back("boneMouthBR");
    bones.push_back("boneMouthTR");
    bones.push_back("boneMouthCR");
    CreateCoreAnimation("noisemouth.xaf", bones);
  }

   
  virtual void process(CalModel* model, CalAnimationAlt* animation) {
    if(!mKeepGoing)
      return;

    CalScheduler* scheduler = (CalScheduler*)(model->getAbstractMixer());
    std::list<CalCoreTrack*>& tracks_list = mCoreAnimation->getListCoreTrack();
    std::vector<CalCoreTrack*> tracks(tracks_list.begin(), tracks_list.end());
    std::vector<CalCoreBone*> bones;
    for(std::vector<CalCoreTrack*>::iterator track = tracks.begin();
	track != tracks.end();
	track++)
      bones.push_back(GetCoreBone((*track)->getCoreBoneId()));

    int size = tracks.front()->getCoreKeyframeCount();

    for (int i=0;i<size;i++) {
      double res=fabs(NoiseFunction(time(0)+i*1.0/size));
      for (int j=0;j<2;j++) {
	CalVector t(res,0,res*0.5);
	tracks[j]->getCoreKeyframe(i)->setTranslation(bones[j]->getTranslation()+t);
      }
      res=fabs(NoiseFunction(time(0)+i*1.0/size+0.2));
      tracks[2]->getCoreKeyframe(i)->setTranslation(bones[2]->getTranslation()+CalVector(res,0,res*0.5));

      res=fabs(NoiseFunction(time(0)+i*1.0/size+0.5));
      for (int j=3;j<5;j++) {
	CalVector t(res,0,res*0.5);
	tracks[j]->getCoreKeyframe(i)->setTranslation(bones[j]->getTranslation()+t);
      }
      res=fabs(NoiseFunction(time(0)+i*1.0/size+0.5+0.2));
      tracks[5]->getCoreKeyframe(i)->setTranslation(bones[5]->getTranslation()+CalVector(res,0,res*0.5));
    }

    scheduler->run(CalScheduler::FOREGROUND,
		   mCoreAnimationId,
		   CalScheduler::ONCE,
		   1.f,
		   new CalScheduler::FadeInOut(.2f, .2f))
      ->setStopCallback(this);
  }
};

class NoiseZygo : public NoiseElement
{
 public:

  NoiseZygo(CalModel* model, const std::string& datadir) : NoiseElement(model, datadir){
    std::list<std::string> bones;
    bones.push_back("boneZygoL");
    bones.push_back("boneZygoR");
    CreateCoreAnimation("noisezygo.xaf", bones);
  }

  
  virtual void process(CalModel* model, CalAnimationAlt* animation) {
    if(!mKeepGoing)
      return;

    CalScheduler* scheduler = (CalScheduler*)(model->getAbstractMixer());
    std::list<CalCoreTrack*>& tracks = mCoreAnimation->getListCoreTrack();
    CalCoreTrack* track_left = tracks.front();
    CalCoreBone* bone_left = GetCoreBone(track_left->getCoreBoneId());
    CalCoreTrack* track_right = tracks.back();
    CalCoreBone* bone_right = GetCoreBone(track_right->getCoreBoneId());
    int size = tracks.front()->getCoreKeyframeCount();

    osg::Quat q0,q1;
    for (int i=0;i<size;i++) {
      double res=fabs(NoiseFunction(time(0)+i*1.0/size));
      //    g_debug("Zygo %f\n", res);
//       res*=10;
      CalVector t(res,0,res);
      track_left->getCoreKeyframe(i)->setTranslation(bone_left->getTranslation()+t);

      res=fabs(NoiseFunction(time(0)+0.5+i*0.1/size));
//       res*=10;
      t=CalVector(res,0,res);
      track_right->getCoreKeyframe(i)->setTranslation(bone_right->getTranslation()+t);
    }

    
    scheduler->run(CalScheduler::FOREGROUND,
		   mCoreAnimationId,
		   CalScheduler::ONCE,
		   1.f,
		   new CalScheduler::FadeInOut(.2f, .2f))
      ->setStopCallback(this);
  }
};

class NoiseEpicr : public NoiseElement
{
 public:

  NoiseEpicr(CalModel* model, const std::string& datadir) : NoiseElement(model, datadir){
    std::list<std::string> bones;
    bones.push_back("boneEpicrML");
    bones.push_back("boneEpicrCL");
    bones.push_back("boneEpicrMR");
    bones.push_back("boneEpicrCR");
    CreateCoreAnimation("noiseepicr.xaf", bones);
  }

  
  void process(CalModel* model, CalAnimationAlt* animation) {
    if(!mKeepGoing)
      return;

    CalScheduler* scheduler = (CalScheduler*)(model->getAbstractMixer());
    std::list<CalCoreTrack*>& tracks_list = mCoreAnimation->getListCoreTrack();
    std::vector<CalCoreTrack*> tracks(tracks_list.begin(), tracks_list.end());
    std::vector<CalCoreBone*> bones;
    for(std::vector<CalCoreTrack*>::iterator track = tracks.begin();
	track != tracks.end();
	track++)
      bones.push_back(GetCoreBone((*track)->getCoreBoneId()));

    int size = tracks.front()->getCoreKeyframeCount();

    for (int i=0;i<size;i++) {
      double res=fabs(NoiseFunction(time(0)+i*1.0/size));
      for (int j=0;j<2;j++) {
	CalVector t(res*.5,res*0.5,0);
	tracks[j]->getCoreKeyframe(i)->setTranslation(bones[j]->getTranslation()+t);
      }
      res=fabs(NoiseFunction(time(0)+i*1.0/size+.5));
      for (int j=2;j<4;j++) {
	CalVector t(res*.5,res*0.5,0);
	tracks[j]->getCoreKeyframe(i)->setTranslation(bones[j]->getTranslation()+t);
      }
    }

    scheduler->run(CalScheduler::FOREGROUND,
		   mCoreAnimationId,
		   CalScheduler::ONCE,
		   1.f,
		   new CalScheduler::FadeInOut(.2f, .2f))
      ->setStopCallback(this);
  }
};

// Model

PokerBodyModel::PokerBodyModel(MAFApplication* application,MAFVisionData *data,MAFOSGData* seat) : mMe(false), mLoopBreath(false), mStartLookCards(0), mEndLookCards(0), mFoldCards(0), mCards(0), mDeck(0), mNbCardsToPlay(0), mDataPath(""), mLookatId(0), mNoiseAnimations(0), mFocus(""), mAnimationsEnabled(true)
{
  SetData(data);
  mCurrentAlpha=1;
  mMinAlpha=0.5;
  mNumberOfCardsReady=0;

  mFoldAnimation=new PlayFoldAnimation(this);
  mFoldSequence=new PokerFoldAnimation((PokerApplication*)application,seat);
  texmatL_ = new osg::TexMat();
  texmatR_ = new osg::TexMat();
}


float PokerBodyModel::ComputeAlphaFromDirection(const osg::Vec3 camDirection)
{
  osg::Matrix tr=MAFComputeLocalToWorld(GetArtefact());
  //  osg::Vec3 dir(tr(2,0),tr(2,1),tr(2,2));
  osg::Vec3 dir(tr(2,0),0,tr(2,2));
  
  osg::Vec3 myCam=camDirection;
  myCam[1]=0;
  myCam.normalize();

  // use only x,z no y
  float a=myCam*dir;
  float alpha=1;
  float maxAngle=cos(mAngleAlpha);
  if (a>maxAngle) {
    alpha=mMinAlpha+1-(a-maxAngle)/(1.0-maxAngle);
    if (alpha>1)
      alpha=1;
  }
  
  return alpha;
}


void PokerBodyModel::SetAlpha(float alpha) 
{
  if (alpha<0)
    alpha=0;
  if (alpha>1)
    alpha=1;
  mCurrentAlpha=alpha;

  int nb=GetArtefact()->getNumDrawables();
  if (mCurrentAlpha==1) {

    for (int i=0;i<nb;i++) {
      osg::StateSet* state=GetArtefact()->getDrawable(i)->getStateSet();
      osg::Material* material = 0;
      if (state)
	material = dynamic_cast<osg::Material*>(state->getAttribute(osg::StateAttribute::MATERIAL));
      if (material) {
// 	g_error("PokerBodyController::Update no material in the drawable of osgCal::Model");
	
	material->setAlpha(osg::Material::FRONT,1);
	state->setMode(GL_BLEND, osg::StateAttribute::OFF);
	state->setRenderingHint(osg::StateSet::OPAQUE_BIN);
      }

    }
  } else {
    for (int i=0;i<nb;i++) {
      osg::StateSet* state=GetArtefact()->getDrawable(i)->getStateSet();
      osg::Material* material = 0;
      if (state)
	material = dynamic_cast<osg::Material*>(state->getAttribute(osg::StateAttribute::MATERIAL));
      if (material) {
	material->setAlpha(osg::Material::FRONT,mCurrentAlpha);
	state->setMode(GL_BLEND, osg::StateAttribute::ON);
	state->setRenderingHint(osg::StateSet::TRANSPARENT_BIN);
      }

    }
  }
  
}

PokerBodyModel::~PokerBodyModel()
{
  for(std::vector<NoiseElement*>::iterator i = mNoiseAnimations.begin();
      i != mNoiseAnimations.end();
      i++)
    delete *i;
  if (mStartLookCards)
    delete (mStartLookCards);
  if (mEndLookCards)
    delete (mEndLookCards);
  if (mFoldCards)
    delete(mFoldCards);

  delete mFoldCards;
  delete mFoldSequence;
  delete mFoldAnimation;
}

void PokerBodyModel::Init()
{
  if(mMe) {
    std::vector<std::string> textures;
    textures.push_back("recto_0");
    textures.push_back("recto_1");
    textures.push_back("recto_2");
    textures.push_back("recto_3");
    SetPrivateTextures(textures);
    
    mStartLookCards = new CardStartLooking(this);
    mEndLookCards = new CardEndLooking(this);
  }

  UGAMEAnimatedModel::Init();

  GetNode()->setName("PokerBody");
  
  g_assert(GetCalModel() != 0);
  g_assert(GetCalModel()->getCoreModel());
  CalCoreModel* coreModel = GetCalModel()->getCoreModel();

//   CreateCoreAnimation("common.cal3d/noiseepicr.xaf", bones);
  mLookatId = coreModel->loadCoreAnimation(mDataPath + "/lookat.xaf");
  if(mLookatId < 0)
    g_error("PlayerAnimation::PlayerAnimation: could not load lookat.xaf");

  CalCoreTrack* coreTrack = GetCoreAnimation(mLookatId)->getListCoreTrack().front();
  g_assert(coreTrack != 0);
  coreTrack->setCoreBoneId(coreModel->getCoreSkeleton()->getCoreBoneId("boneSkull"));
  if (!coreModel->addAnimationName("lookat",mLookatId))
    assert(0);

  InitCardsOfPlayer();

  //
  // Facial noise animations
  //
  const std::string &dir = mDataPath;
  mNoiseAnimations.push_back(new NoiseSkull(GetCalModel(), dir));
  mNoiseAnimations.push_back(new NoiseEyes(GetCalModel(), dir));
//   mNoiseAnimations.push_back(new NoiseNose(GetCalModel(), dir));
//   mNoiseAnimations.push_back(new NoiseMouth(GetCalModel(), dir));
//   mNoiseAnimations.push_back(new NoiseZygo(GetCalModel(), dir));
//   mNoiseAnimations.push_back(new NoiseEpicr(GetCalModel(), dir));

  // hack get eyes mesh of player 	 
  MAFCal3dData::Outfit resultOfSearchEye; 	 
  if (!GetData()->GetOutfit(&resultOfSearchEye, "/cal3d/outfits/outfit[@name='" + GetOutfit() + "']/item[@name='eyes']/mesh")) 	 
	  g_critical("PokerBodyModel::Init Hey guy don't forget to put eyes on your characters !!!"); 	 

  if (resultOfSearchEye.size()>1) 	 
	  g_critical("PokerBodyModel::Init Only two eyes are required for a character !!!"); 	 

  int end=(int)resultOfSearchEye.size(); 	 
  for (int i=0;i<end;i++) { 	 
	  mEyesMeshName.push_back(resultOfSearchEye[i]["name"]); 	 
	  g_debug("PokerBodyModel::Init Eyes %s found",mEyesMeshName.back().c_str()); 	 
  }
}

struct SeekTime : CalAnimationAlt::TimeFunction
{
  float mTime;
  SeekTime() : mTime(0)
  {
  }
  SeekTime(const SeekTime &cpy) : CalAnimationAlt::TimeFunction(cpy), mTime(cpy.mTime)
  {    
  }
  virtual Function* clone() { return new SeekTime(*this); }
  void AddTime(float time) 
  { 
    float newTime = mTime + time;
    if (newTime >= 1.0f)
      newTime = 0.99f;
    else if (newTime < 0.0f)
      newTime = 0.0f;
    mTime = newTime;
  }
  virtual float process(CalAnimationAlt* pAnimation)
  {
    return mTime * pAnimation->getDuration();
  }
};

int PokerBodyModel::PlayStandUp()
{
  if(!mAnimationsEnabled) return -1;
  int standUpId = GetCoreAnimationId("seatUp");

  CalAnimationAlt *a = GetScheduler()->run(CalScheduler::FOREGROUND,
					   standUpId,
					   CalScheduler::FOREVER);

  SeekTime *seek = new SeekTime();
  a->setTimeFunction(seek);
  delete(seek);

  return a->getAnimationId();
}


void PokerBodyModel::PlayBreath(float time)
{
  if(!mAnimationsEnabled) return;
  // we play the breath only one time in loop mode
  if (!mLoopBreath)
    GetScheduler()->run(CalScheduler::FOREGROUND,
			GetCoreAnimationId("breath"),
			CalScheduler::FOREVER,
			1.f,
			new CalScheduler::FadeInOut(.3f, .3f),
			time);
  mLoopBreath=true;
}

void PokerBodyModel::PlayBlink()
{
  if(!mAnimationsEnabled) return;
  GetScheduler()->run(CalScheduler::FOREGROUND,
		      GetCoreAnimationId("blink"),
		      CalScheduler::ONCE);
}


void PokerBodyModel::PlayWin()
{
  if(!mAnimationsEnabled) return;
  //PlayStartPlaying(); now in python
  //  double duration=GetDuration("s_playWin");

  GetScheduler()->run(CalScheduler::FOREGROUND,
		      GetCoreAnimationId("win"),
		      CalScheduler::ONCE,
		      1.f,
		      new CalScheduler::FadeInOut(.3f, .8f));
  ; //   PlayEndPlaying(duration * .8f); now in python
}

void PokerBodyModel::PlayLoose()
{
  if(!mAnimationsEnabled) return;
  //PlayStartPlaying(); now in python
  //  double duration = GetDuration("s_playLose");

  GetScheduler()->run(CalScheduler::FOREGROUND,
		      GetCoreAnimationId("lose"),
		      CalScheduler::ONCE,
		      1.f,
		      new CalScheduler::FadeInOut(.3f, .8f));
  // PlayEndPlaying(duration * .8f); now in python
}


void PokerBodyModel::PlayStanding()
{
  if(!mAnimationsEnabled) return;
  Stop();

  GetScheduler()->run(CalScheduler::FOREGROUND,
		      GetCoreAnimationId("upIdle01"),
		      CalScheduler::FOREVER,
		      1.f);
}



double PokerBodyModel::PlayStartLookCards()
{
  // be sure to have card
  if (mMe) {
    if (GetNbCardsDisplayed() != mNumberOfCardsReady) {
      DettachCardsDrawableToPlayer();
      for (int i=0;i<mNumberOfCardsReady;i++)
	ShowCard(i);
    }
  }

  if(!mAnimationsEnabled) return 0.0;

  mStartLookCards->Look();
  mEndLookCards->Init();
  return 0;
}

double PokerBodyModel::PlayEndLookCards()
{
  if(!mAnimationsEnabled) return 0.0;
  mEndLookCards->Look();
  return GetDuration("lookC");
}


int PokerBodyModel::IsPlayingAnimationList(const std::vector<std::string>& animationlist)
{
  int i=0;
  int end=(int)animationlist.size();
  for (;i<end;i++) {
//     std::cout << "test id " << GetCoreAnimationId(animationlist[i]) << std::endl;
    if (GetScheduler()->isAnimationActive(GetCoreAnimationId(animationlist[i])))
      return i;
  }
  return -1;
}


void PokerBodyModel::PlayLookAt(const osg::Vec3& directionOfPlayerorigin,
				const osg::Vec3& positionOfPlayer, 
				const osg::Vec3& target)
{
  if(!mAnimationsEnabled) return;

  if (GetScheduler()->getAnimation(mLookatId))
    return;

  std::vector<std::string> animationlist;
  animationlist.push_back("fold");
  animationlist.push_back("check");
  animationlist.push_back("bet");
  float delay=0;
  int i=0;
  int end=3;
  for (;i<end;i++) {
    float a=GetCoreAnimation(animationlist[i])->getDuration();
    if (a>delay)
      delay=a;
  }

  CalQuaternion quat;

  // recompute direction to be in an axis aligned plane
  osg::Vec3 dir = directionOfPlayerorigin;
  dir.y() = 0;
  dir.normalize();

  BuildQuaternionFromTarget(dir, positionOfPlayer, target, quat);
  CalCoreTrack* track = GetCoreAnimation(mLookatId)->getListCoreTrack().front();
  g_assert(track != 0);
  CalBone* bone = GetBone("boneSkull");

  int size=track->getCoreKeyframeCount();
  for (int i=0;i<size;i++) {
    track->getCoreKeyframe(i)->setTranslation(bone->getTranslation());
    track->getCoreKeyframe(i)->setRotation(quat);
  }
  
  GetScheduler()->run(CalScheduler::FOREGROUND,
		      mLookatId,
		      CalScheduler::ONCE,
		      1.f,
		      new CalScheduler::FadeInOut(.5f, .5f),
		      delay);
  g_debug("DELAY %f",delay);
}


void PokerBodyModel::BuildQuaternionFromTarget(const osg::Vec3& _origin, const osg::Vec3& _target,CalQuaternion& _result)
{
  osg::Quat quat;

  osg::Vec3 origin(_origin[0],_origin[1],_origin[2]);
  osg::Vec3 target(_target[0],_target[1],_target[2]);
  
  osg::Vec3 f(target-origin);
  f.normalize();

  osg::Vec3 reference;
  osg::Vec3 targetSide(target);
  reference=osg::Vec3(0.f, 0.f, origin.z()) - origin;
  reference.normalize();
  osg::Vec3 repere=reference^osg::Vec3(0,1,0);
  float sign=repere*target; // give me the sign of angle

  float angle=acos(reference*f);
  angle*=(sign<0?-1:1);

  quat.makeRotate(angle, osg::Vec3(0,1,0));
  _result=CalQuaternion(quat[0],quat[1],quat[2],quat[3]);
}

void PokerBodyModel::BuildQuaternionFromTarget(const osg::Vec3& _dir,const osg::Vec3& _origin, const osg::Vec3& _target,CalQuaternion& _result)
{
  osg::Quat quat;

  osg::Vec3 origin(_origin[0],_origin[1],_origin[2]);
  osg::Vec3 target(_target[0],_target[1],_target[2]);
  
  osg::Vec3 dir=-_dir;

  // direction from player1 to player to look at
  osg::Vec3 f(target-origin);
  f.normalize();

  osg::Vec3 repere=dir^osg::Vec3(0,1,0);
  float sign=repere*target-_origin*repere; // give me the sign of angle

  float val=dir*f;
  if (val>1.0)
    val=1.0;
  else if (val<-1.0)
    val=-1.0;
  float angle=acos(val);
  angle*=(sign<0?-1:1);

  // attenuate angle
  angle*=0.6;

  quat.makeRotate(angle, osg::Vec3(0,1,0));

  _result=CalQuaternion(quat[0],quat[1],quat[2],quat[3]);
}

void PokerBodyModel::PlayFacialNoise()
{
  if(!mAnimationsEnabled) return;

  for(std::vector<NoiseElement*>::iterator noise = mNoiseAnimations.begin();
      noise != mNoiseAnimations.end();
      noise++) {
    (*noise)->SetKeepGoing(true);
    (*noise)->process(GetCalModel(), 0);
  }
}

void PokerBodyModel::StopFacialNoise()
{
  for(std::vector<NoiseElement*>::iterator noise = mNoiseAnimations.begin();
      noise != mNoiseAnimations.end();
      noise++) {
    (*noise)->SetKeepGoing(false);
    GetScheduler()->stop((*noise)->GetCoreAnimationId());
  }
}


void PokerBodyModel::Stop()
{
  GetScheduler()->stop(CalScheduler::ALL);
}

void PokerBodyModel::StopAll()
{
  StopFacialNoise();
  Stop();
}

void PokerBodyModel::InitCardsOfPlayer()
{
  // 
  // Init data to manage looking card
  //
  mCards.clear();
  for(int i = 0; i < 5; i++) {
    char tmp[32];
    CardEntry cards;
    snprintf(tmp, sizeof(tmp), "bentcard%d", i);
    UGAMEAnimatedController::Drawables* drawables = GetDrawables(tmp);
    if(!drawables || drawables->empty()) {
      g_critical("PokerBodyModel::InitCardsOfPlayer: %s: no drawables for %s", CalError::getLastErrorText().c_str(), tmp);
      continue;
    }
    if(drawables->size() != 2) {
      g_critical("PokerBodyModel::InitCardsOfPlayer: %d drawables for %s instead of 2", drawables->size(), tmp);
      continue;
    }

    for(int j = 0; j < 2; j++) {
      if((*drawables)[j]->getStateSet() == 0) {
	    g_critical("PokerBodyModel::InitCardsOfPlayer: no stateSet for drawable %d of %s", j, tmp);
	    continue;
      }
      if((*drawables)[j]->getStateSet()->getTextureAttribute(0, osg::StateAttribute::TEXTURE) == 0) {
	    g_critical("PokerBodyModel::InitCardsOfPlayer: no texture for drawable %d of %s", j, tmp);
	    continue;
	  } else {
#if 1
		osg::AlphaFunc *alphaFunc = new osg::AlphaFunc();
		alphaFunc->setReferenceValue(0.5);
		alphaFunc->setFunction(osg::AlphaFunc::GREATER);
		(*drawables)[j]->getStateSet()->setAttributeAndModes(alphaFunc);
		(*drawables)[j]->getStateSet()->setMode(GL_LIGHTING, osg::StateAttribute::OFF);
#else
	    osg::BlendFunc *bf = new osg::BlendFunc();
	    bf->setFunction(GL_SRC_COLOR, GL_ONE_MINUS_SRC_ALPHA);
	    (*drawables)[j]->getStateSet()->setAttributeAndModes(bf, osg::StateAttribute::ON);
//		(*drawables)[j]->getStateSet()->setRenderingHint(osg::StateSet::TRANSPARENT_BIN);
//		(*drawables)[j]->getStateSet()->setMode(GL_LIGHTING, osg::StateAttribute::OFF);
#endif
 	  }
      if(j == 0)
	cards.front = (*drawables)[j].get();
      else
	cards.back = (*drawables)[j].get();
    }
    // stack
    mCards.push_back(cards);
  }

  DettachCardsDrawableToPlayer();
  g_debug("Cards found %d\n",mCards.size());
}

void PokerBodyModel::UpdateCardsOfPlayer(const std::vector<int>& cards)
{
  std::vector<int> cardsValue;
  std::vector<osg::Image*> cardsImages;

  mNumberOfCardsReady=cards.size();

  if (!mMe)
    return;

  for (std::vector<int>::const_iterator i = cards.begin();
       i != cards.end();
       i++)
    cardsImages.push_back(mDeck->GetImage(*i));

  if (cards.empty())
    return;

  unsigned int nb=mNbCardsToPlay;
  if (mCards.size()<nb) {
    g_critical("Data of player missing not enough cards (wanted %d currently %d)\n",nb,mCards.size());
    nb=mCards.size();
  }

  for (unsigned int i=0;i<nb;i++) {
    osg::StateSet* state = mCards[i].front->getStateSet();
    if (!state)
      g_error("PokerBodyModel::UpdateCardsOfPlayer osg::state not found for a cards of player. check materials card of player");
    g_assert(state != 0);
    osg::Texture2D* current_texture = dynamic_cast<osg::Texture2D*>(state->getTextureAttribute(0, osg::StateAttribute::TEXTURE));
    g_assert(current_texture != 0);
    current_texture->setImage(cardsImages[i]);
  }
}


void PokerBodyModel::DettachCardsDrawableToPlayer()
{
  osgCal::Model* model = GetArtefact();

  int nb=mCards.size();
  for (int i=0;i<nb;i++) {
    model->removeDrawable(mCards[i].front.get());
    model->removeDrawable(mCards[i].back.get());
  }
}

int PokerBodyModel::GetNbCardsDisplayed()
{
  int nbCards=0;
  osgCal::Model* model = GetArtefact();
  for (int i=0;i<(int)mCards.size();i++) {
    if (model->getDrawableIndex(mCards[i].front.get())!=model->getNumDrawables())
      nbCards++;
  }
  return nbCards;
}

void PokerBodyModel::ShowCard(int i)
{
  if (i>=mNumberOfCardsReady)
    return;
  assert(i>=0 && i<(int)mCards.size());
  osgCal::Model* model = GetArtefact();
  if (model->getDrawableIndex(mCards[i].front.get())==model->getNumDrawables())
    model->addDrawable(mCards[i].front.get());
  if (model->getDrawableIndex(mCards[i].back.get())==model->getNumDrawables())
    model->addDrawable(mCards[i].back.get());
}

void PokerBodyModel::HideCard(int i)
{
  assert(i>=0 && i<(int)mCards.size());
  osgCal::Model* model = GetArtefact();
  model->removeDrawable(mCards[i].front.get());
  model->removeDrawable(mCards[i].back.get());
}


void PokerBodyModel::DisableCurrentAnimationCallback(const std::string &name)
{
  DisableCurrentAnimationCallback(GetCoreAnimationId(name));
}

void PokerBodyModel::DisableCurrentAnimationCallback(int id)
{
  if (GetScheduler()->isAnimationActive(id)) {
    CalAnimationAlt* a = GetScheduler()->getAnimation(id);
    a->setStopCallback(0);
  }
}




// at the end animation start the PokerFoldSequence
// see file PokerFoldSequence.h/cpp
void PlayFoldAnimation::process(CalModel* model, CalAnimationAlt* animation)
{
  int nb=mBody->mNbCardsToPlay;
  for (int i=0;i<nb;i++) {
    mBody->HideCard(i);
    mBody->mFoldSequence->ShowCard(i);
  }
    
  mBody->mFoldSequence->StartSequence();
}


// animation must be a fold animation.
void PokerBodyModel::PlayFold(const std::string& foldAnimation)
{
  if(!mAnimationsEnabled) return;
  int id=GetCoreAnimationId(foldAnimation);
  CalAnimationAlt *a;
  a=GetScheduler()->run(CalScheduler::FOREGROUND,
			id,
			CalScheduler::ONCE); // 1e4 means that this animation will be the one seen by the user
  a->setStopCallback(mFoldAnimation);
}
  





// Controller 

PokerBodyController::PokerBodyController(MAFApplication* application,MAFVisionData *data,MAFOSGData* seat)
{
  SetModel(new PokerBodyModel(application,data,seat));

  PokerApplication *game = dynamic_cast<PokerApplication *>(application);
  const std::string &baseAlphaStr=game->HeaderGet("sequence", "/sequence/player/@baseAlpha");
  if (baseAlphaStr.length())
    GetModel()->mMinAlpha=atof(baseAlphaStr.c_str());
  else
    g_error("PokerBodyController::PokerBodyController /sequence/player/@baseAlpha not found in config file");

  const std::string &angleAlphaStr=game->HeaderGet("sequence", "/sequence/player/@angleAlpha");
  if (angleAlphaStr.length())
    GetModel()->mAngleAlpha=osg::PI/180.0*atof(angleAlphaStr.c_str());
  else
    g_error("PokerSeatController::Init /sequence/player/@angleAlpha not found in config file");

}

void PokerBodyController::HandleHit(MAFHit& hit) 
{
  typedef osgCal::SubMeshSoftware subMesh;
  subMesh *mesh = dynamic_cast<subMesh *>(hit.mHit._drawable.get());
  if (mesh != 0)
    GetModel()->mFocus = mesh->getName();
  else
    GetModel()->mFocus = "";
}



bool PokerBodyController::Update(MAFApplication* application)
{
  PokerApplication *game = dynamic_cast<PokerApplication *>(application);

  if (game->HasEvent()) {
    SDL_Event* event = game->GetLastEvent(this);
    int originalAnimationEnabled=GetModel()->mAnimationsEnabled;
    if(game->GetFocus() == this  && event && event->type == SDL_KEYDOWN) {
      if (!originalAnimationEnabled)
	GetModel()->EnableAnimations();
      switch(event->key.keysym.sym) {
      case SDLK_c:
// 	PlayIdlePlaying();
	break;
      case SDLK_d:
// 	PlayBet();
	break;
      case SDLK_e:
// 	PlayCheck();
	break;
      case SDLK_f:
	PlayWin();
	break;
      case SDLK_g:
	PlayLoose();
	break;
      case SDLK_h:
	PlayFacialNoise();
	break;
      case SDLK_i:
// 	PlayFold();
	break;
      case SDLK_j:
// 	PlayIdleWaiting();
	break;
      case SDLK_k:
// 	PlayWaitNoise();
	break;
      case SDLK_l:
	PlayBlink();
	break;
      case SDLK_m:
	PlayStartLookCards();
	break;
      case SDLK_n:
	break;
      case SDLK_o:
	PlayEndLookCards();
	break;
      case SDLK_p:
	PlayBreath(0);
	break;
      case SDLK_q:
// 	PlayWaiting();
	break;
      case SDLK_r:
	PlayStanding();
	break;
      case SDLK_s:
	//	PlayStartPlaying();
	break;
      case SDLK_t:
// 	PlayEndPlaying();
	break;
      case SDLK_z:
	StopAll();
	break;
      default:
	break;
      }
      if (!originalAnimationEnabled)
	GetModel()->DisableAnimations();
    }
    return true;
  }

   if (GetModel()->mMe)
     if (game->GetFocus() != this)
       GetModel()->mFocus = "";


   GetModel()->mFoldSequence->Update(0);

  PokerCameraModel* camera=(dynamic_cast<PokerCameraController*>(game->GetScene()->GetModel()->mCamera.get()))->GetModel();
  osg::Vec3 camDirection=camera->GetTarget()-camera->GetPosition();
  camDirection.normalize();

  float alpha=1;
  alpha=GetModel()->ComputeAlphaFromDirection(camDirection);
  
  //float currentAlpha=GetModel()->GetAlpha(); // unused 

  float camDuraction=camera->mTimer.GetDuration();
  float camCurrentTime=camera->mTimer.GetCurrentTime();

  if (GetModel()->mMe) {
    switch (camera->GetMode()) {
    case PokerCameraModel::CAMERA_DIRECT_MODE: 
    case PokerCameraModel::CAMERA_GAME_MODE: 
      {
      osg::Vec3 camPrevDirection=camera->mCamPrevTarget-camera->mCamPrevPosition;
      camPrevDirection.normalize();
      float srcAlpha=GetModel()->ComputeAlphaFromDirection(camPrevDirection);
      float dstAlpha=1;
      float timeState=camCurrentTime/camDuraction;
      timeState=camera->mLengthBInterpolator.GetT(timeState);
      alpha=srcAlpha+(dstAlpha-srcAlpha)*timeState;
//        g_debug("ENTERMODE srcAlpha %f currentAlpha %f timestate %f",srcAlpha,alpha,timeState);
      }
      break;
    case PokerCameraModel::CAMERA_LEAVE_MODE:
      {
      osg::Vec3 camPrevDirection=camera->mCamNextTarget-camera->mCamNextPosition;
      camPrevDirection.normalize();
      float dstAlpha=GetModel()->ComputeAlphaFromDirection(camPrevDirection);
      float srcAlpha=1;
      float timeState=camCurrentTime/camDuraction;
      timeState=camera->mLengthBInterpolator.GetT(timeState);
      alpha=srcAlpha+(dstAlpha-srcAlpha)*timeState;
//        g_debug("LEAVEMODE srcAlpha %f currentAlpha %f timestate %f",dstAlpha,alpha,timeState);
      }
      break;
    default:
//       g_debug("NOMODE %f",alpha);
      break;
    }
  }

  GetModel()->SetAlpha(alpha);


  PokerBodyModel *bm = GetModel();
  CalBone *bone = bm->GetBone("boneEyeL");
  const CalQuaternion &quat = bone->getRotation();
  CalVector vec(0, 0, 1);
  vec *= quat;

  // hack for eyes
  g_assert(GetModel()->mEyesMeshName.size());
  osgCal::Model::Drawables *draws = bm->GetDrawables(GetModel()->mEyesMeshName[0]);
  if (!draws)
    g_critical("PokerBodyController::Update no mesh eyes found for outfit %s",GetModel()->GetOutfit().c_str());
  else {
    if (!draws->size())
      g_critical("PokerBodyController::Update mesh eyes found but empty for outfit %s",GetModel()->GetOutfit().c_str());
    else {
      int end=(int)draws->size();
      for (int i=0;i<end;i++) {
	osg::Drawable *draw = (*draws)[i].get();
	osg::Matrix mat;
	mat = osg::Matrix::translate(-vec.x*(24/256.0f), -vec.y*(24/256.0f), 0);
	bm->texmatL_->setMatrix(mat);
	draw->getOrCreateStateSet()->setTextureAttributeAndModes(0, bm->texmatL_, osg::StateAttribute::ON);
      }
    }
  }

  return true;
}

void PokerBodyController::AddTimeSitIn(float time)
{
  int sitDownId = GetModel()->GetCoreAnimationId("seatDown");
  AddTimeSit(sitDownId, time);
}

void PokerBodyController::AddTimeSitOut(float time)
{
  int sitOutId = GetModel()->GetCoreAnimationId("seatUp");
  AddTimeSit(sitOutId, time);
}

void PokerBodyController::AddTimeSit(int id, float time)
{
  CalAnimationAlt *a = GetModel()->GetScheduler()->getAnimation(id);
  if (a)
    {
      CalAnimationAlt::TimeFunction *timeFunction = a->getTimeFunction();
      bool seeked = (timeFunction != 0);
      if (seeked)
	{
	  SeekTime *seek = static_cast<SeekTime *>(timeFunction);
	  seek->AddTime(time); 
	}
    }
}

// CardStartEndLooking

CardStartLooking::CardStartLooking(PokerBodyModel *body) : mBody(body)
{
}

void CardStartLooking::process(CalModel* model, CalAnimationAlt* animation)
{
  mBody->GetScheduler()->run(CalScheduler::FOREGROUND,
			     mBody->GetCoreAnimationId(std::string("lookB")),
			     CalScheduler::FOREVER,
			     1e4); // 1e4 means that this animation will be the one seen by the user
}

void CardStartLooking::Look()
{
  CalAnimationAlt*a;
  a=mBody->GetScheduler()->run(CalScheduler::FOREGROUND,
			       mBody->GetCoreAnimationId(std::string("lookA")),
			       CalScheduler::ONCE,
			       1e4); // 1e4 means that this animation will be the one seen by the user
  a->setStopCallback(this);
}

void CardEndLooking::process(CalModel* model, CalAnimationAlt* animation)
{
  mBody->GetScheduler()->run(CalScheduler::FOREGROUND,
			     mBody->GetCoreAnimationId(std::string("lookC")),
			     CalScheduler::ONCE,
			     1e4); // 1e4 means that this animation will be the one seen by the user
}

void CardEndLooking::Look()
{
  int lookBId = mBody->GetCoreAnimationId(std::string("lookB"));
  int lookId = mBody->GetCoreAnimationId(std::string("lookA"));
  
  if (mBody->GetScheduler()->isAnimationActive(lookBId))
    {
      CalAnimationAlt* a = mBody->GetScheduler()->getAnimation(lookBId);
      a->setStopCallback(this);
      mBody->GetScheduler()->stop(lookBId,
				  NULL,
				  0.2);
    }
  else if (mBody->GetScheduler()->isAnimationActive(lookId))
    {
      CalAnimationAlt* a = mBody->GetScheduler()->getAnimation(lookId);
      a->setStopCallback(this);
    }  
}












void PokerBodyModel::BuildAnimationSoundMap(std::map<std::string,osg::ref_ptr<MAFAudioController> >& source)
{
  mAnimation2Sounds.clear();
  typedef std::map<std::string,osg::ref_ptr<MAFAudioController> >::iterator MapIt;
  source.size();
  for (MapIt it=source.begin();it!=source.end();it++) {
    std::string name=it->first;
    int id=GetCoreAnimationId(name);
    if (id!=-1) {
      MAFAudioController* sound=it->second.get();
      mAnimation2Sounds[id]=sound;
      g_debug("PokerBodyModel::BuildAnimationSoundMap assign sound %s to id %d",name.c_str(),id);
    } else {
      g_debug("PokerBodyModel::BuildAnimationSoundMap sound %s not assigned to animation",name.c_str());
    }
  }

#ifdef HACK_SOUND_IN_CAL3D
  GetScheduler()->mAnimation2Sounds=&mAnimation2Sounds;
#endif
}
