/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Cedric PINSON <cpinson@freesheep.org>
 *  Loic Dachary <loic@gnu.org>
 *
 */

#include "undercal3dStdAfx.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifdef WIN32
#include "config_win32.h"
#endif

#include <math.h>

#ifndef WIN32
#include "cal3d/animationalt.h"
#endif // WIN32

#include "cal3d/error.h"
#include "cal3d/mixer.h"
#include "cal3d/coremodel.h"
#include "cal3d/corebone.h"
#include "cal3d/coreanimation.h"
#include "cal3d/coretrack.h"
#include "cal3d/corekeyframe.h"
#include "cal3d/model.h"
#include "cal3d/skeleton.h"
#include "cal3d/bone.h"

float CalAnimationAlt::FadeInOut::process(CalAnimationAlt* pAnimation)
{
  if(pAnimation->getState() == CalAnimationAlt::WANT_TO_STOP)
    {
      pAnimation->setLength(pAnimation->getLength() + m_fadeOut);
      pAnimation->setState(STOPPING);
    }

  float weight = 1.f;

  float cumulatedRunningTime = pAnimation->getCumulatedTime();

  if(m_fadeIn > 0.0001f && cumulatedRunningTime < m_fadeIn)
    {
      weight *= cumulatedRunningTime / m_fadeIn;
    }

  if(pAnimation->getLength() >= 0.f) {
    if(cumulatedRunningTime < pAnimation->getLength())
      {
	if(m_fadeOut > 0.0001f &&
	   cumulatedRunningTime > pAnimation->getLength() - m_fadeOut)
	  {
	    float delayBeforeEnd = pAnimation->getLength() - cumulatedRunningTime;
	    weight *= delayBeforeEnd / m_fadeOut;
	  }
      }
    else
      {
	pAnimation->setState(WANT_TO_STOP);
      }
  }
    
  return weight;
}

CalAnimationAlt::Info::Info()
{
  m_state = RUNNING;
  m_channel = CalBlender::BACKGROUND;
  m_pStopCallback = 0;
  m_pWeightFunction = 0;
  m_pTimeFunction = 0;
  m_coreAnimationId = -1;
  m_animationId = -1;
  m_length = 0.f;
  m_weight = -1;
}

CalAnimationAlt::CalAnimationAlt() :
  m_loopCount(0),
  m_cumulatedTime(0.f),
  m_time(-1),
  m_weight(1.f),
  m_pCoreAnimation(0)
{
}

CalAnimationAlt::CalAnimationAlt(CalCoreAnimation* pCoreAnimation) :
  m_loopCount(0),
  m_cumulatedTime(0.f),
  m_time(-1),
  m_weight(1.f)
{
  setCoreAnimation(pCoreAnimation);
}

CalAnimationAlt::~CalAnimationAlt()
{
  setWeightFunction(0);
  setTimeFunction(0);  
  //  std::cout << "cache " << number_of_good*1.0/number_of_call << std::endl;
}

CalAnimationAlt& CalAnimationAlt::setWeightFunction(WeightFunction* pWeightFunction) 
{
  if(getWeightFunction())
    delete getWeightFunction();

  if(pWeightFunction)
    m_info.m_pWeightFunction = (WeightFunction*)pWeightFunction->clone();
  else
    m_info.m_pWeightFunction = 0;

  return *this;
}

CalAnimationAlt& CalAnimationAlt::setTimeFunction(TimeFunction* pTimeFunction)
{
  if(getTimeFunction())
    delete getTimeFunction();

  if(pTimeFunction)
    m_info.m_pTimeFunction = (TimeFunction*)pTimeFunction->clone();
  else
    m_info.m_pTimeFunction = 0;

  return *this;
}

/**
 *  Fix keyframes if shorter than duration of animation
 */
void CalAnimationAlt::FixAnimation()
{
  CalCoreAnimation *pCoreAnimation = getCoreAnimation();
  assert(pCoreAnimation);

  float duration = pCoreAnimation->getDuration();
  std::list<CalCoreTrack*>& listCoreTrack = pCoreAnimation->getListCoreTrack();

  std::list<CalCoreTrack *>::iterator itr;
  for(itr = listCoreTrack.begin(); itr != listCoreTrack.end(); itr++)
  {
    CalCoreTrack *coreTrack = *itr;
    CalCoreKeyframe *lastKeyframe = coreTrack->getCoreKeyframe(coreTrack->getCoreKeyframeCount() - 1);

    if(lastKeyframe->getTime() < duration)
    {
      CalCoreKeyframe *firstKeyframe = coreTrack->getCoreKeyframe(0);
      CalCoreKeyframe *newKeyframe = new CalCoreKeyframe();
      newKeyframe->setTranslation(firstKeyframe->getTranslation());
      newKeyframe->setRotation(firstKeyframe->getRotation());
      newKeyframe->setTime(pCoreAnimation->getDuration());
      coreTrack->addCoreKeyframe(newKeyframe);
    }
  }
}

/**
 *  update time of an animation
 *
 *  there are new feature here:
 *  user can specify a time fonction that modify time (typically for slowdown animation or x2 speed)
 *  user can specify a weight fonction that modify the weight of animation in time
 *
 */
int CalAnimationAlt::update(float deltaTimeInSecond)
{
  float modifiedTime;

  m_cumulatedTime += deltaTimeInSecond;

  if(getState() == WANT_TO_STOP)
    setLength(m_cumulatedTime);

  if(getTimeFunction())
  {
    modifiedTime = getTimeFunction()->process(this);
  }
  else
  {
    modifiedTime = m_cumulatedTime;
  }

  float animationDuration = getDuration();

  m_loopCount = (int)floorf(modifiedTime / animationDuration);

  if(getLength() < 0.f || getLength() >= m_cumulatedTime)
  {
    m_time = fmodf(modifiedTime, animationDuration);
  }
  else
  {
    m_time = animationDuration;
    setState(STOPPED);
  }

  if(getWeightFunction())
  {
    m_weight = getWeightFunction()->process(this);
  }
  else
  {
    m_weight = m_info.m_weight;
  }

  if(getState() == WANT_TO_STOP)
    setState(STOPPED);

  return getState() == STOPPED;
}


/**
 *  set core animation for this animation alt
 *
 *  Create a cache of keyframe for each core track. the cache contain
 *  the key upper of the current time asked by getUpperBound
 */
void CalAnimationAlt::setCoreAnimation(CalCoreAnimation* pCoreAnimation)
{
  m_pCoreAnimation=pCoreAnimation;
#ifdef USE_CACHE_KEYFRAME
  m_cacheKeyframe.resize(pCoreAnimation->getListCoreTrack().size());
#endif
}


#ifdef USE_CACHE_KEYFRAME
bool CalAnimationAlt::getState(CalCoreTrack* track,int indexOfCoreTrack,float time, CalVector& translation, CalQuaternion& rotation) 
{
  // remove iterator because this function is the bottleneck
  std::vector<CalCoreKeyframe*>::iterator iteratorCoreKeyframeBefore;
  std::vector<CalCoreKeyframe*>::iterator iteratorCoreKeyframeAfter;

  // get the keyframe after the requested time
  //    iteratorCoreKeyframeAfter = getUpperBound(time);
  getUpperBound(track,indexOfCoreTrack,time,iteratorCoreKeyframeAfter);

  // check if the time is after the last keyframe
  if(iteratorCoreKeyframeAfter == track->getCoreKeyframes().end())
    {
      // return the last keyframe state
      --iteratorCoreKeyframeAfter;
	
      rotation = (*iteratorCoreKeyframeAfter)->getRotation();
      translation = (*iteratorCoreKeyframeAfter)->getTranslation();

      return true;
    }

  // check if the time is before the first keyframe
  if(iteratorCoreKeyframeAfter == track->getCoreKeyframes().begin())
    {
      // return the first keyframe state
      rotation = (*iteratorCoreKeyframeAfter)->getRotation();
      translation = (*iteratorCoreKeyframeAfter)->getTranslation();

      return true;
    }

  // get the keyframe before the requested one
  iteratorCoreKeyframeBefore = iteratorCoreKeyframeAfter;
  --iteratorCoreKeyframeBefore;

  // get the two keyframe pointers
  CalCoreKeyframe *pCoreKeyframeBefore;
  pCoreKeyframeBefore = *iteratorCoreKeyframeBefore;
  CalCoreKeyframe *pCoreKeyframeAfter;
  pCoreKeyframeAfter = *iteratorCoreKeyframeAfter;

  // calculate the blending factor between the two keyframe states
  float blendFactor;
  blendFactor = (time - pCoreKeyframeBefore->getTime()) / (pCoreKeyframeAfter->getTime() - pCoreKeyframeBefore->getTime());

  // blend between the two keyframes
  translation = pCoreKeyframeBefore->getTranslation();
  translation.blend(blendFactor, pCoreKeyframeAfter->getTranslation());

  rotation = pCoreKeyframeBefore->getRotation();
  rotation.blend(blendFactor, pCoreKeyframeAfter->getRotation());

  return true;
}

void CalAnimationAlt::getUpperBound(CalCoreTrack* track,int indexOfCoreTrack,float time,std::vector<CalCoreKeyframe*>::iterator& result) 
{

  //  number_of_call++;

  unsigned int* cacheArray=&m_cacheKeyframe.front();
  unsigned int upperBound = track->getCoreKeyframes().size()-1;
  unsigned int lowerBound = 0;
  unsigned int candidate=cacheArray[indexOfCoreTrack];
  CalCoreKeyframe** k=&track->getCoreKeyframes().front();

  // do it only if it's not the key 0 in this case it's fast enough to not using the cache
  if (candidate && k[candidate]->getTime()>time && k[candidate-1]->getTime()<time) {
    result=track->getCoreKeyframes().begin() + candidate;
    //    number_of_good++; //std::cout << "use cache " << candidate << std::endl;
    return;
  } else if (k[candidate]->getTime()<time)
    lowerBound=candidate;
  else if (k[candidate]->getTime()>time)
    upperBound=candidate;

  while(lowerBound<upperBound-1)
    {
      unsigned int middle = (lowerBound+upperBound)>>1;

      if(time >= k[middle]->getTime())
	{
	  lowerBound=middle;
	}
      else
	{
	  upperBound=middle;
	}
    }
  //  number_of_miss++;
  //  std::cout << "don't use cache " << candidate << " key used " << upperBound << std::endl;
  cacheArray[indexOfCoreTrack]=upperBound;
  result=track->getCoreKeyframes().begin() + upperBound;
}
#endif
