#!/usr/bin/perl

package Main;

# This file is part of the pnopaste program
# Copyright (C) 2008-2009 Patrick Matthäi <patrick@linux-dev.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

use warnings;
no warnings 'once';
use strict;
use CGI qw/:standard/;
use lib::HTML;
use lib::Database;
use lib::Security;
use lib::Highlighting;
use lib::Expire;


# CGI class.
my $CGI = new CGI;

# Security settings for the cgi module.
$CGI::DISABLE_UPLOADS = 1;
$CGI::POST_MAX = 1024 * 1024 * 8;


Main();


### Main function - it decides where we go.
sub Main {

	# Block all banned IPs directly.
	if(Security::Blacklist_Host($CGI->remote_host()) == 1){
		HTML::Error(2);
	}

	if(Expire::Get_Cleanup()){
		# Clean up database.
		Expire::Delete_Old_Entrys();
	}


	# Get all GET parameters.
	my $Params = $CGI->url_param('keywords');

	# Seems like someone wants a code.
	if($Params and $Params =~ /^\d/){

		my $Download = 0;
		if($Params =~ /download$/){
			$Download = 1;
		}

		HTML::Code(int($Params), $Download, $CGI->url(-full =>1));
	}

	# New code comes in.
	elsif($CGI->param('code')){
		Add_Code();
	}

	else {
		# Else just show the add page.
		HTML::Add($CGI->url(-full => 1));
	}

}


### Adds a new code to the database.
sub Add_Code {
	# Get POST parameters.
	my $Code = $CGI->param('code')			|| '';
	my $Desc = $CGI->param('description')	|| '';
	my $Name = $CGI->param('name')			|| '';
	my $Expi = $CGI->param('expires')		|| $Expire::Standard;
	my $Synt = $CGI->param('language')		|| 'Plain';

	# Get remote host address.
	my $Remote_Addr = $CGI->remote_host();


	# Now some security checks.
	if(Expire::Valid_Time($Expi) == 0){
		HTML::Error(1);
	}
	elsif(Security::Blacklist_Words($Code) == 1){
		HTML::Error(3);
	}
	elsif(Highlighting::Is_Valid($Synt) == 0){
		HTML::Error(4);
	}

	# Create query with all needed data.
	my $Query = 'INSERT INTO `nopaste` (name, description, code, time, ip, expires, language) VALUES (?, ?, ?, ?, ?, ?, ?)';
	$Query = $Database::dbh->prepare($Query);
	$Query->execute($Name, $Desc, $Code, time(), $Remote_Addr, $Expire::Time_List{$Expi}, $Synt);

	# Get the last MySQL auto_increment ID.
	my $Last_ID = $Database::dbh->{'mysql_insertid'};

	# Show us the confirm page.
	HTML::Confirm($Last_ID, $CGI->url(-full => 1));
}


# Close the connection.
$Database::dbh->disconnect();

1;

