/* pngcomp.c

A simple test program for png image quantisation. 
  
This program compares the pixel colors of two images and prints out 
statistics on the differences between the images. 

Statistics printed include:
mean error
standard deviation of error
maximum error
	     
The error is calculated as the linear distance between two colors in RGBA space.

Copyright (C) 2006 by Stuart Coyle

*/
#define VERSION "0.1"
#define PNGCOMP_USAGE "usage: pngcomp [-vVh] image1.png image2.png\n\
  options: v - verbose, does nothing as yet.\n\
           V - version, prints version information.\n\
           h - help, prionts this message.\n\
  inputs: image1.png and image2.png are the two images that are to be compared.\n\
          it is required that they be the same size.\n\
\n\
  This program give some basic statistics about the difference between two images.\n\
  It was created as a measure of various color quantization methods.\n"

#define MAX_COLOR_VAL 256
#define SQRT_3 1.73205

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include <ctype.h>

#include "png.h"
#include "rwpng.h"

typedef struct {
  uch r, g, b, a;
} pixel;

struct statistics {
  double max_error;
  double mean_error;
  double stddev_error;
  ulg   n_pixels;
  ulg   correct_pixels;
};

/* Image information structs */
static mainprog_info image1_info;
static mainprog_info image2_info;

float *imagediff(char* file1_name, char* file2_name);
float errval(pixel *p1, pixel *p2);
struct statistics *gather_stats(float *error_data);
void printstats(struct statistics* stats);


int main(int argc, char** argv)
{
  int verbose = 0;
 
  char *file1_name = NULL;
  char *file2_name = NULL;

  int c; /* argument count */

  int retval = 0;
  float* err_image = NULL;

  /* Parse arguments */
  while((c = getopt(argc,argv,"hVv"))!=-1){
    switch(c){
    case 'v':
      verbose = 1;
      break;
    case 'V':
      fprintf(stderr,"pngcomp %s\n",VERSION);
      rwpng_version_info();
      exit(EXIT_SUCCESS);
      break;
    case 'h':
      fprintf(stderr,PNGCOMP_USAGE);
      exit(EXIT_SUCCESS);
      break;
    case '?':      
      if (isprint(optopt))
	fprintf (stderr, "  unknown option `-%c'.\n", optopt);
      else
	fprintf (stderr,
		 "  unknown option character `\\x%x'.\n",
		 optopt);
    default:
      fprintf(stderr,PNGCOMP_USAGE);
      exit(EXIT_FAILURE);
    }
  }


  /* determine input files */
  if(optind == argc){
    fprintf(stderr,"  pngcomp requires two input file names.\n");
    exit(EXIT_FAILURE);
  }
  else{
    file1_name=argv[optind];
    optind++;
    if(optind == argc){
      fprintf(stderr,"  pngcomp requires two file names.\n");
      exit(EXIT_FAILURE);
    }
    else{
      file2_name=argv[optind];
      optind++;
    }
  }

  err_image = imagediff(file1_name,file2_name);
  if(err_image != NULL){
    struct statistics *stats = gather_stats(err_image);
    printstats(stats);
  }

  exit(retval);

}


float *imagediff(char* file1_name, char* file2_name){
  
  FILE *file1 = NULL;
  FILE *file2 = NULL;  
    
  ulg cols, rows;
  ulg row;

  float* error_data = NULL;

  /* Open the image files */
  if((file1 = fopen(file1_name, "rb"))==NULL){
    fprintf(stderr,"  error: cannot open %s for reading.\n",file1_name);
    fflush(stderr);
    return NULL;
  }

  if((file2 = fopen(file2_name, "rb"))==NULL){
    fprintf(stderr,"  error: cannot open %s for reading.\n",file2_name);
    fflush(stderr);
    return NULL;
  }

  /* Read each image */
  rwpng_read_image(file1,&image1_info);
  fclose(file1);
  if (image1_info.retval) {
    fprintf(stderr, "  rwpng_read_image() error\n");
    fflush(stderr);
    return NULL; 
  }

  rwpng_read_image(file2,&image2_info);
  fclose(file2);
  if (image2_info.retval) {
    fprintf(stderr, "  rwpng_read_image() error\n");
    fflush(stderr);
    return NULL; 
  }
 
  /* Can't do images that differ in size */
  /* Is there any point? */
  cols = image1_info.width;
  rows= image1_info.height;

  if(image2_info.width != cols || image2_info.height != rows){
    fprintf(stderr, "  images differ in size. cannot continue. \n");
    return(NULL);
  }
  
   
  if(!image1_info.rgba_data || !image2_info.rgba_data)
    {
      fprintf(stderr,"  no pixel data found.");
      return(NULL);
    }

  error_data = (float *)calloc(cols*rows*sizeof(float),sizeof(float));
  if(error_data == NULL){
    fprintf(stderr,"  cannot allocate error buffer.");
    return(NULL);
  }

  
  /* Calculate error value for each pixel */
  for(row=0;(ulg)row < rows; ++row){
    int col;
    pixel p1, p2;
    ulg offset = row*cols*4;

    for( col=0;(ulg)col<cols;++col){
      p1.r = image1_info.rgba_data[col*4+offset];
      p1.g = image1_info.rgba_data[col*4+offset+1];
      p1.b = image1_info.rgba_data[col*4+offset+2];
      p1.a = image1_info.rgba_data[col*4+offset+3];

      p2.r = image2_info.rgba_data[col*4+offset];
      p2.g = image2_info.rgba_data[col*4+offset+1];
      p2.b = image2_info.rgba_data[col*4+offset+2];
      p2.a = image2_info.rgba_data[col*4+offset+3];

      error_data[col*row] = errval(&p1,&p2);
      
    }
  }
  return error_data;
}


/* Calculates cartesian distance of pixels in rgba space */
float errval(pixel *p1, pixel *p2){
  long err;
 
  long err_r = p1->r-p2->r;
  long err_g = p1->g-p2->g;
  long err_b = p1->b-p2->b;
  long err_a = p1->a-p2->a;
  err = err_r*err_r + err_g*err_g + err_b*err_b+ err_a*err_a;
  return sqrt((double)err);
}


struct statistics *gather_stats(float *error_data){

  int count;
  struct statistics *stats = malloc(sizeof(struct statistics));
  
  if(stats == NULL){
    fprintf(stderr,"  Cannot allocate statistics struct.");
    return(NULL);
  }
  
 
  stats->max_error = 0.0;
  stats->mean_error = 0.0;
  stats->stddev_error = 0.0;
  stats->n_pixels = image1_info.width*image1_info.height;
  stats->correct_pixels = (ulg)0;

  /* Basic stats */
  for(count=0;count<stats->n_pixels;count++){
    float err = error_data[count];
    if(err > stats->max_error) stats->max_error = err;
    stats->mean_error += err;
    if(err <= 0.0) stats->correct_pixels++;
  }
  stats->mean_error = (float)(stats->mean_error)/(float)(stats->n_pixels);
  
  /* Standard deviation */
 for(count=0;count<stats->n_pixels;count++){
    double err= error_data[count];
    stats->stddev_error += (err-stats->mean_error)*(err-stats->mean_error);
 }
 stats->stddev_error = sqrt(stats->stddev_error)/stats->n_pixels;
 
 return stats;

}


void printstats(struct statistics* stats){
  printf("RGBA image color difference statistics.\n");
  printf("Mean error: %f \n",stats->mean_error);
  printf("Maximum error: %f \n",stats->max_error);
  printf("Standard Deviation of Error: %f\n",stats->stddev_error);
  printf("Image Dimensions %d x %d \n",image1_info.width,image1_info.height);
  printf("Number of pixels: %d \n",stats->n_pixels);
  printf("Number of correct pixels: %d\n",stats->correct_pixels);
  printf("Percentage correct pixels: %f%\n",(float)stats->correct_pixels/(float)stats->n_pixels*100.0);

}

