/*  
    pmacct (Promiscuous mode IP Accounting package)
    pmacct is Copyright (C) 2004 by Paolo Lucente
*/

/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* defines */
#define __PMACCTD_C

/* includes */
#include "pmacct.h"
#include "pmacct-data.h"
#include "plugin_hooks.h"
#include "pkt_handlers.h"

/* variables to be exported away */
int debug, got_alrm_signal;
struct configuration config; /* global configuration */ 
struct plugins_list_entry *plugins_list = NULL; /* linked list of each plugin configuration */ 
struct channels_list_entry channels_list[MAX_N_PLUGINS]; /* communication channels: core <-> plugins */
int have_num_memory_pools; /* global getopt() stuff */

/* Functions */
void usage_daemon(char *prog_name)
{
  printf("%s\n", PMACCTD_USAGE_HEADER);
  printf("Usage: %s [-D] [-b buckets] [-i interface] [filter]\n", prog_name);
  printf("       %s [-f config_file]\n", prog_name);
  printf("       %s [-h]\n", prog_name);
  printf("\n");
  printf("Options:\n");
  printf("  -f  \tspecify a configuration file (see EXAMPLES file for valid config keys)\n");
  printf("  -c  \t[src_mac|dst_mac|src_host|dst_host|src_net|dst_net|sum|src_port|dst_port|proto] \n\tcounts source, destination or total IP traffic (default src_host)\n");
  printf("  -D  \tdaemonize the process\n"); 
  printf("  -N  \tdon't use promiscuous mode\n");
  printf("  -n  \tpath for the file containing network definitions; to be used in conjunction with 'src_net' or 'dst_net'\n");
  printf("  -P  \t[memory|mysql|pgsql] \n\tactivate specified plugins\n"); 
  printf("  -p  \tpath for client/server fifo (used by in-memory table plugin only)\n");
  printf("  -d  \tenables debug mode\n");
  printf("  -b  \tnumber of buckets\n");
  printf("  -i  \tinterface used for listening\n");
  printf("  -m  \tnumbers of memory pools (see INTERNALS file)\n");
  printf("  -r  \trefresh time of data into SQL database from in-memory cache (in seconds)\n");
  printf("  -s  \tsize of each memory pool\n");
  printf("  -S  \t[auth|mail|daemon|kern|user|local[0-7]] \n\tenables syslog logging to the specified facility\n");
  printf("  -h  \tprints this page\n");
  printf("\n");
  printf("Examples:\n");
  printf("  Daemonize the process; listen on eth0; write statst in a MySQL database\n"); 
  printf("  pmacctd -c dst_port -i eth0 -D -P mysql\n\n");
  printf("  Debug mode; listen on ee1; collect stats aggregating for destination host in a in-memory\n");
  printf("  structure with 8 memory pools of 64Kb each\n");
  printf("  pmacctd -c dst_host -i ee1 -d -m 8 -s 65535\n");
  printf("\n");
  printf("  See EXAMPLES file in the distribution for more examples\n");
  printf("\n");
  printf("For suggestions, critics, bugs, contact me: %s.\n", MANTAINER);
}


int main(int argc,char **argv)
{
  bpf_u_int32 localnet, netmask;  /* pcap library stuff */
  struct bpf_program filter;
  struct pcap_device device;
  char errbuf[PCAP_ERRBUF_SIZE];
  int index, logf;

  struct plugins_list_entry *list;
  char config_file[SRVBUFLEN];
  int psize = DEFAULT_SNAPLEN;

  /* getopt() stuff */
  extern char *optarg;
  extern int optind, opterr, optopt;
  int errflag, cp; 

  if (getuid() != 0) { 
    Log(LOG_ERR, "ERROR: You need superuser privileges to run pmacct daemon.\nExiting ...\n\n");
    exit(0);
  }

  umask(077);

  /* a bunch of default definitions */ 
  have_num_memory_pools = FALSE;
  errflag = 0;

  memset(&config, 0, sizeof(struct configuration));
  memset(&device, 0, sizeof(struct pcap_device));
  memset(&config_file, 0, sizeof(config_file));

  /* zeroing 'cfg array' */ 
  for (rows = 0; rows < SRVBUFLEN; rows++) memset(cfg[rows], 0, SRVBUFLEN); 
  rows = 0;

  /* getting commandline values */
  while (!errflag && ((cp = getopt(argc, argv, ARGS_DAEMON)) != -1)) {
    switch (cp) {
    case 'P':
      rows++;
      strcpy(cfg[rows], "plugins: ");
      strncat(cfg[rows], optarg, CFG_LINE_LEN(cfg[rows]));
      break;
    case 'D':
      rows++;
      strcpy(cfg[rows], "daemonize: true");
      break;
    case 'd':
      debug = TRUE;
      rows++;
      strcpy(cfg[rows], "debug: true");
      break;
    case 'n':
      rows++;
      strcpy(cfg[rows], "networks_file: ");
      strncat(cfg[rows], optarg, CFG_LINE_LEN(cfg[rows]));
      break;
    case 'N':
      rows++;
      strcpy(cfg[rows], "promisc: false");
      break;
    case 'f':
      strlcpy(config_file, optarg, sizeof(config_file));
      break;
    case 'F':
      rows++;
      strcpy(cfg[rows], "pidfile: ");
      strncat(cfg[rows], optarg, CFG_LINE_LEN(cfg[rows]));
      break;
    case 'c':
      rows++;
      strcpy(cfg[rows], "aggregate: ");
      strncat(cfg[rows], optarg, CFG_LINE_LEN(cfg[rows]));
      break;
    case 'b':
      rows++;
      strcpy(cfg[rows], "imt_buckets: ");
      strncat(cfg[rows], optarg, CFG_LINE_LEN(cfg[rows]));
      break;
    case 'm':
      rows++;
      strcpy(cfg[rows], "imt_mem_pools_number: ");
      strncat(cfg[rows], optarg, CFG_LINE_LEN(cfg[rows]));
      have_num_memory_pools = TRUE;
      break;
    case 'p':
      rows++;
      strcpy(cfg[rows], "imt_path: ");
      strncat(cfg[rows], optarg, CFG_LINE_LEN(cfg[rows]));
      break;
    case 'r':
      rows++;
      strcpy(cfg[rows], "sql_refresh_time: ");
      strncat(cfg[rows], optarg, CFG_LINE_LEN(cfg[rows]));
      break;
    case 's':
      rows++;
      strcpy(cfg[rows], "imt_mem_pools_size: ");
      strncat(cfg[rows], optarg, CFG_LINE_LEN(cfg[rows]));
      break;
    case 'S':
      rows++;
      strcpy(cfg[rows], "syslog: ");
      strncat(cfg[rows], optarg, CFG_LINE_LEN(cfg[rows]));
      break;
    case 'i':
      rows++;
      strcpy(cfg[rows], "interface: ");
      strncat(cfg[rows], optarg, CFG_LINE_LEN(cfg[rows]));
      break;
    case 'h':
      usage_daemon(argv[0]);
      exit(1);
      break;
    default:
      usage_daemon(argv[0]);
      exit(1);
      break;
    }
  }

  /* post-checks and resolving conflicts */
  if (strlen(config_file)) {
    if (parse_configuration_file(config_file) != SUCCESS) 
      exit(1);
  }
  else {
    if (parse_configuration_file(NULL) != SUCCESS)
      exit(1);
  }
    
  /* XXX: glue; i'm conscious it's a dirty solution from an engineering viewpoint;
     someday later i'll fix this */
  list = plugins_list;
  while(list) {
    if (!strcmp(list->name, "default") || !strcmp(list->type.string, "core")) {
      memcpy(&config, &list->cfg, sizeof(struct configuration)); 
      break;
    }
    list = list->next;
  }

  if (config.daemon) {
    if (debug || config.debug)
      printf("WARN: debug is enabled; forking in background. Console logging will get lost.\n"); 
      daemonize();
      signal(SIGINT, SIG_IGN);
  }

  if (config.syslog) {
    logf = parse_log_facility(config.syslog);
    if (logf == ERR) {
      config.syslog = NULL;
      Log(LOG_WARNING, "WARN: specified syslog facility is not supported; logging to console.\n");
    }
    else openlog(NULL, LOG_PID, logf);
    Log(LOG_INFO, "INFO: Start logging ...\n");
  }

  if (config.pidfile) write_pid_file(config.pidfile);

  /* Enforcing policies over aggregation methods */
  list = plugins_list;
  while (list) {
    if (!strcmp(list->type.string, "core"));  
    else {
      if (list->cfg.what_to_count & COUNT_SUM_HOST) {
        if (list->cfg.what_to_count != COUNT_SUM_HOST) {
          config.what_to_count = COUNT_SUM_HOST;
          Log(LOG_WARNING, "WARN: using *only* sum aggregation method in '%s-%s'.\n", list->name, list->type.string);
	}
      }
      else if (!list->cfg.what_to_count) {
	Log(LOG_WARNING, "WARN: defaulting to src_host aggregation in '%s-%s'.\n", list->name, list->type.string);
	list->cfg.what_to_count = COUNT_SRC_HOST;
      }
      else if ((list->cfg.what_to_count & COUNT_SRC_NET) || (list->cfg.what_to_count & COUNT_DST_NET)) {
	if (!list->cfg.networks_file) {
	  Log(LOG_ERR, "ERROR: net aggregation method has been selected but no networks file specified. Exiting...\n\n");
	  exit(0);
	}
	else {
	  if (list->cfg.what_to_count & COUNT_SRC_NET) list->cfg.what_to_count |= COUNT_SRC_HOST;
	  if (list->cfg.what_to_count & COUNT_DST_NET) list->cfg.what_to_count |= COUNT_DST_HOST;
	}
      }
    } 
    list = list->next;
  }

  if (!config.dev) {
    Log(LOG_WARNING, "WARN: selecting a suitable device.\n");
    config.dev = pcap_lookupdev(errbuf); 
    if (!config.dev) {
      Log(LOG_WARNING, "WARN: unable to find a suitable device. Exiting ...\n");
      exit(0);
    }
  }

  /* reading filter; if it exists, we'll take an action later */
  if (!strlen(config_file)) config.clbuf = copy_argv(&argv[optind]);

  /* starting the wheel */
  if ((device.dev_desc = pcap_open_live(config.dev, psize, config.promisc, 1000, errbuf)) == NULL) {
    Log(LOG_ERR, "ERROR: pcap_open_live(): %s\n", errbuf);
    exit(1);
  } 
  device.active = TRUE;
  glob_pcapt = device.dev_desc; /* SIGINT/stats handling */ 

  device.link_type = pcap_datalink(device.dev_desc); 
  for (index = 0; _devices[index].link_type != -1; index++) {
    if (device.link_type == _devices[index].link_type)
      device.data = &_devices[index];
  }

  /* we need to solve some link constraints */
  if (device.data == NULL) {
    Log(LOG_ERR, "ERROR: data link not supported: %d\n", device.link_type); 
    exit(0);
  }
  else Log(LOG_INFO, "OK: link type is: %d\n", device.link_type); 

  if (device.link_type != DLT_EN10MB) {
    if ((config.what_to_count & COUNT_SRC_MAC) || 
	(config.what_to_count & COUNT_DST_MAC)) {
      Log(LOG_ERR, "ERROR: MAC aggregation methods not available for this link type: %d\n", device.link_type);
      exit(0);
    }
  }
  
  /* doing pcap stuff */
  if (pcap_lookupnet(config.dev, &localnet, &netmask, errbuf) < 0) {
    localnet = 0;
    netmask = 0;
    Log(LOG_WARNING, "WARN: %s\n", errbuf);
  }

  if (pcap_compile(device.dev_desc, &filter, config.clbuf, 0, netmask) < 0)
    Log(LOG_WARNING, "WARN: %s\nWARN: going on without a filter\n", pcap_geterr(device.dev_desc));
  else {
    if (pcap_setfilter(device.dev_desc, &filter) < 0)
      Log(LOG_WARNING, "WARN: %s\nWARN: going on without a filter\n", pcap_geterr(device.dev_desc));
  }

  /* signal handling we want to inherit to plugins */
  signal(SIGCHLD, handle_falling_child); /* this eliminates zombies */
  signal(SIGHUP, reload); /* handles reopening of syslog channel */
  signal(SIGPIPE, SIG_IGN); /* we want to exit gracefully when a pipe is broken */

  /* plugins glue: creation */
  load_plugins(&device);
  evaluate_packet_handlers();

  /* signals to be handled only by pmacctd;
     we set proper handlers after plugin creation */
  if (!config.daemon) signal(SIGINT, my_sigint_handler);

  /* Main loop: if pcap_loop() exits maybe an error occurred; we will try closing
     and reopening again our listening device */
  while(1) {
    if (!device.active) {
      Log(LOG_WARNING, "WARN: %s become unavailable; throttling ...\n", config.dev);
      sleep(5); /* XXX: to get fixed */
      if ((device.dev_desc = pcap_open_live(config.dev, psize, config.promisc, 1000, errbuf)) == NULL) {
        Log(LOG_ERR, "ERROR: pcap_open_live(): %s\n", errbuf);
        exit(1);
      }
      pcap_setfilter(device.dev_desc, &filter);
      device.active = TRUE;
    }
    pcap_loop(device.dev_desc, -1, pcap_cb, (u_char *) &device);
    pcap_close(device.dev_desc);
    device.active = FALSE;
  }
}

void pcap_cb(u_char *user, const struct pcap_pkthdr *pkthdr, const u_char *buf)
{
  struct packet_ptrs pptrs;
  struct pcap_device *device = (struct pcap_device *) user;

  /* We process the packet with the appropriate
     data link layer function */
  if (buf) { 
    pptrs.pkthdr = pkthdr;
    pptrs.packet_ptr = (u_char *) buf;
    (*device->data->handler)(pkthdr, &pptrs);
    if (pptrs.iph_ptr) {
      ip_handler(&pptrs);
      exec_plugins(&pptrs);
    }
  }
} 

void ip_handler(register struct packet_ptrs *pptrs)
{
  register u_int8_t len = 0;
  
  /* len: number of 32bit words forming the header */
  len = IP_HL(((struct my_iphdr *) pptrs->iph_ptr));
  pptrs->tlh_ptr = pptrs->iph_ptr+(len*4); 
}
