/************************************************************************
 * $Id: bridge.c,v 1.3 2002/08/13 00:06:25 DemonLord Exp $
 ************************************************************************
 *
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2002  Kevin P. Lawton
 *
 *  bridge.c: Dispatch IO port events
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <stdio.h>

#include "io_thread.h"
#include "bridge.h"
#include "plex86.h"
#include "plugin.h"

#define MAX_HANDLER	0x20
#define NO_HANDLER	0xFF

#define EVENT_PORT_READ		0
#define EVENT_PORT_WRITE	1

typedef struct _port_event_t
{
  Bit32u	address;
  unsigned int	io_len;
  union
  {
    Bit32u	value;
    Bit32u      *ret_val;
  } io_value;
} port_event_t;

static void* port_handler_read(void* data);
static void* port_handler_write(void* data);

static void port_write(void *this_ptr, Bit32u address, Bit32u value, unsigned int io_len);
static Bit32u port_read(void *this_ptr, Bit32u address, unsigned int io_len);


int bridgeRegisterIOReadHandler(void *thisPtr, ioReadHandler_t callback, unsigned base, const char *name, unsigned len);
int bridgeRegisterIOWriteHandler(void *thisPtr, ioWriteHandler_t callback, unsigned base, const char *name, unsigned len);

int (*realRegisterIOReadHandler)(void *thisPtr, ioReadHandler_t callback, unsigned base, const char *name, unsigned len);
int (*realRegisterIOWriteHandler)(void *thisPtr, ioWriteHandler_t callback, unsigned base, const char *name, unsigned len);

static int read_id[64*1024];
static int write_id[64*1024];
static ioReadHandler_t read_callback[MAX_HANDLER];
static ioWriteHandler_t write_callback[MAX_HANDLER];
static void* read_object[MAX_HANDLER];
static void* write_object[MAX_HANDLER];
static int read_count;
static int write_count;

static eventAction handlers[2] = {
    port_handler_read, port_handler_write
  };

int plugin_init(plugin_t *plugin, int argc, char *argv[])
{
  int i;

  set_io_handlers(handlers);

  thread_init();

  for (i = 0; i < 64*1024; i++)
    read_id[i] = write_id[i] = NO_HANDLER;

  read_count = write_count = 0;

  pluginRegisterIOReadHandler(plugin, port_read, 0xFFFFFFFF, "Bridge", 1);
  pluginRegisterIOWriteHandler(plugin, port_write, 0xFFFFFFFF, "Bridge", 1);

  realRegisterIOReadHandler = pluginRegisterIOReadHandler;
  realRegisterIOWriteHandler = pluginRegisterIOWriteHandler;

  pluginRegisterIOReadHandler = bridgeRegisterIOReadHandler;
  pluginRegisterIOWriteHandler = bridgeRegisterIOWriteHandler;

  return 0;
}

void plugin_fini(void)
{
  pluginRegisterIOReadHandler = realRegisterIOReadHandler;
  pluginRegisterIOWriteHandler = realRegisterIOWriteHandler;
}

static void* port_handler_read(void* data)
{
  port_event_t *evt = (port_event_t*) data;
  int id = read_id[evt->address];

  if (id == NO_HANDLER)
  {
//    fprintf(stderr, "Bridge :\tUnhandled in port 0x%x\n", address);
    *(evt->io_value.ret_val) = 0x80;
    return NULL;
  }

  *(evt->io_value.ret_val) = read_callback[id](read_object[id], evt->address, evt->io_len);
  return NULL;
}

static void* port_handler_write(void* data)
{
  port_event_t *evt = (port_event_t*) data;
  int id = write_id[evt->address];
 
  if (id == NO_HANDLER)
  {
//    fprintf(stderr, "Bridge :\tUnhandled out port 0x%x\n", evt->address);
    return NULL;
  }
  write_callback[id](write_object[id], evt->address, evt->io_value.value, evt->io_len);
  return NULL;
}  

static void port_write(void *this_ptr, Bit32u address, Bit32u value, unsigned int io_len)
{
  port_event_t evt;

  evt.address		= address;
  evt.io_value.value	= value;
  evt.io_len		= io_len;

  post_evt (EVENT_PORT_WRITE, &evt, sizeof(port_event_t));
}

static Bit32u port_read(void *this_ptr, Bit32u address, unsigned int io_len)
{
  port_event_t evt;
  Bit32u ret_val;

  evt.address		= address;
  evt.io_value.ret_val	= &ret_val;
  evt.io_len		= io_len;

  post_evt (EVENT_PORT_READ, &evt, sizeof(port_event_t));
  queue_flush();
  return ret_val;
}

int bridgeRegisterIOReadHandler(void *thisPtr, ioReadHandler_t callback, unsigned base, const char *name, unsigned len)
{
  int i;

  for (i = 0; i < read_count; i++)
    if (callback == read_callback[i] && thisPtr == read_object[i])
      break;

  if (i >= MAX_HANDLER)
  {
    fprintf(stderr, "Bridge :\tCan't register port %d: %s (maximum handler count reached)\n", base, name);
    return -1;
  }

  if (i >= read_count)
  {
    read_count++;
    read_callback[i] = callback;
    read_object[i] = thisPtr;
  }

  if (read_id[base] != NO_HANDLER)
    fprintf(stderr, "Bridge :\tWARNING! Redefining port %d\n", base);

  read_id[base] = i;

  return i;
}

int bridgeRegisterIOWriteHandler(void *thisPtr, ioWriteHandler_t callback, unsigned base, const char *name, unsigned len)
{
  int i;

  for (i = 0; i < write_count; i++)
    if (callback == write_callback[i] && thisPtr == write_object[i])
      break;

  if (i >= MAX_HANDLER)
  {
    fprintf(stderr, "Bridge :\tCan't register port %d: %s (maximum handler count reached)\n", base, name);
    return -1;
  }

  if (i >= write_count)
  {
    write_count++;
    write_callback[i] = callback;
    write_object[i] = thisPtr;
  }

  if (write_id[base] != NO_HANDLER)
    fprintf(stderr, "Bridge :\tWARNING! Redefining port %d\n", base);

  write_id[base] = i;

  return i;
}
