/************************************************************************
 * $Id: plex86.h,v 1.2 2002/07/21 23:14:54 DemonLord Exp $
 ************************************************************************
 *
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999  Kevin P. Lawton
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#ifndef __PLEX86_H__
#define __PLEX86_H__

#include "config.h"
#include "descriptor.h"


#define PLEX86_INIT_TEST   10
#define PLEX86_INIT_LINUX  11

/*
 * the eflags field looks like this:
 * bit:  0  1 2  3 4  5 6  7  8  9  A  B  C/D  E  F 10 11 12 13  14  15 16
 * flg:  CF 1 PF 0 AF 0 ZF SF TF IF DF OF IOPL NT 0 RF VM AC VIF VIP ID 0
 */

#define FLG_CF    (1<<0)
#define FLG_PF    (1<<2)
#define FLG_AF    (1<<4)
#define FLG_ZF    (1<<6)
#define FLG_SF    (1<<7)
#define FLG_TF    (1<<8)
#define FLG_IF    (1<<9)
#define FLG_DF    (1<<10)
#define FLG_OF    (1<<11)
#define FLG_IOPL  (3<<12)
#define FLG_NT    (1<<14)
#define FLG_RF    (1<<16)
#define FLG_VM    (1<<17)
#define FLG_AC    (1<<18)
#define FLG_VIF   (1<<19)
#define FLG_VIP   (1<<20)
#define FLG_ID    (1<<21)



typedef Bit64u icount_t;
#define ICOUNT_INDEFINITE  ((icount_t) 0)
#define ICOUNT_CONTINUE   (((icount_t) 0) - 1)


typedef struct {
  selector_t   sel;
  descriptor_t des;
  unsigned     valid;
  } guest_sreg_t;

typedef struct {
  Bit32u eax;
  Bit32u ebx;
  Bit32u ecx;
  Bit32u edx;
  Bit32u ebp;
  Bit32u esi;
  Bit32u edi;
  Bit32u esp;
  Bit32u eflags;
  Bit32u eip;
  guest_sreg_t cs;
  guest_sreg_t ss;
  guest_sreg_t ds;
  guest_sreg_t es;
  guest_sreg_t fs;
  guest_sreg_t gs;
  guest_sreg_t ldtr;
  guest_sreg_t tr;
  struct { Bit32u base, limit; } gdtr;
  struct { Bit32u base, limit; } idtr;
  Bit32u dr0, dr1, dr2, dr3, dr6, dr7;
  Bit32u tr3, tr4, tr5, tr6, tr7;
  Bit32u cr0, cr1, cr2, cr3, cr4;
  unsigned inhibit_mask;
  } guest_cpu_t;






/* ========================================================== */
/* Messages which are passed between the user program (u) and */
/* the monitor (m) of the VM. */
/* ========================================================== */
#define VMMessageNone           0

//#define VMMessageIACRequest     1 /* m->u */
//#define VMMessageIACResponse    2 /* u->m */
#define VMMessageIntRequest     3 /* m->u */
#define VMMessageIntResponse    4 /* u->m */

#define VMMessageIOInRequest    5 /* m->u */
#define VMMessageIOInResponse   6 /* u->m */
#define VMMessageIOOutRequest   7 /* m->u */

#define VMMessageMemMapIOReadRequest  8  /* m->u */
#define VMMessageMemMapIOReadResponse 9  /* u->m */
#define VMMessageMemMapIOWriteRequest 10 /* m->u */

#define VMMessagePanic         11 /* m->u */
#define VMMessagePrintBuf      12 /* m->u */
#define VMMessageRunGuestN     13 /* u->m */

#define VMMessageTimeElapsed   14 /* m->u */

#define VMMessageDisasm        15 /* m->u */

#define VMMessageEOICount      16 /* m->u */
#define VMMessageReqComplete   17 /* m->u */

#define VMMessageIOBatchRequest    18 /* m->u */
#define VMMessageIOBatchResponse   19 /* u->m */

#define MaxVMMessages 32 /* Large enough to contain all the above. */


#define MAX_VM_MESSAGE_PACKET  128

#define PrescanRing3Off   0
#define PrescanRing3Auto  1
#define PrescanRing3On    2

typedef struct {
  struct header_tag {
    unsigned msg_type;
    unsigned msg_len;
    } header;
  unsigned char msg[MAX_VM_MESSAGE_PACKET];
  } vm_messages_t;

typedef struct {
  Bit32u   port;
  unsigned len;
  unsigned op;
  Bit32u   data;
  void    *thisPtr;
  void    *callback;
  } IO_msg_t;

typedef struct {
  Bit32u   port;
  unsigned len;
  unsigned op;
  unsigned n;
  Bit32u   paddr;
  void    *thisPtr;
  void    *callback;
  } IOBatch_msg_t;

typedef struct {
  Bit32u   addr;
  unsigned len;
  unsigned op;
  Bit32u   data;
  } memMapIO_msg_t;

typedef struct {
  unsigned vector;
  unsigned reflect;
  } INT_msg_t;


/* There are 3 ways to instruct the VM to run the guest.  The first
 * is for running the guest indefinitely in the normal fashion, and
 * must pass an instruction count of ICOUNT_INDEFINITE.
 *
 *   Execute: Run normally.  Most code executes natively, some
 *     code is necessarily virtualized (and emulated) by the VM monitor.
 *
 * The other 2 are for running the guest for exactly N instructions.
 *
 *   Emulate: The VM monitor operates in a pure emulation mode, executing
 *     exactly 'icount' instructions.
 *   Breakpoint: The VM monitor executes the guest natively where
 *     possible, but turns on the Trap Flag, so that it can execute
 *     instructions one-at-a-time.
 */

#define RunGuestNMethodExecute    10
#define RunGuestNMethodEmulate    11
#define RunGuestNMethodBreakpoint 12

typedef struct {
  icount_t icount;
  unsigned method;
  } run_guest_n_t;

typedef struct {
  void     *thisPtr;
  void     *callback;
  unsigned  callbackID;
  } timerCallback_t;

typedef struct {
  void     *thisPtr;
  void     *callback;
  Bit32u    useconds;
  Boolean   continuous;
  Boolean   active;
  } timerRegister_t;

#define IO_IN  10 // xxx Fix these, duplicate stuff below
#define IO_OUT 11

/* An IO handler registers itself using a bitlist requesting
 * that it handles read and/or writes of a given port addresss.
 */
#define ioHandlerOpRead    0x01
#define ioHandlerOpWrite   0x02

typedef struct {
  void    *thisPtr;
  void    *callback;
  Bit32u   base;
  unsigned len;
  unsigned op;
  } ioRegister_t;

typedef struct {
  unsigned data;
  } event_msg_t;

typedef struct {
  Bit32u cs;
  Bit32u eip;
  Bit32u laddr;
  Bit32u seg32;
  } EOICount_t;

typedef struct {
  unsigned irq;
  unsigned on;
  } irqMsg_t;

/*
 *  ioctl() names
 */

#if defined(__linux__) || defined(__NetBSD__) || defined(__FreeBSD__)
#ifdef __linux__
#include <asm/ioctl.h>
#else
#include <sys/ioccom.h>
#endif
#define PLEX86_ALLOCVPHYS   _IO('k', 2)
#define PLEX86_ALLOCINT     _IO('k', 3)
#define PLEX86_RELEASEINT   _IO('k', 4)
#define PLEX86_RESET        _IO('k', 5)
#define PLEX86_MESSAGEQ     _IO('k', 6)
#define PLEX86_TEARDOWN     _IO('k', 7)
#define PLEX86_SETINTR      _IO('k', 8)
#define PLEX86_PRESCANDEPTH _IO('k', 9)
#define PLEX86_SET_CPU      _IOW('k', 12, guest_cpu_t)
#define PLEX86_RESET_CPU    _IO('k', 13)
#define PLEX86_GET_CPU      _IOR('k', 14, guest_cpu_t)
#define PLEX86_FORCE_INT    _IO('k', 15)
#define PLEX86_SET_A20      _IO('k', 16)
#define PLEX86_PHYMEM_MOD   _IO('k', 17)
#define PLEX86_PRESCANRING3 _IO('k', 18)
#define PLEX86_GENERIC      _IO('k', 19)
#define PLEX86_REGTIMER     _IO('k', 20)
#define PLEX86_ACTTIMER     _IO('k', 21)
#define PLEX86_DEACTTIMER   _IO('k', 22)
#define PLEX86_REGIO        _IO('k', 23)
#define PLEX86_IRQ          _IO('k', 24)
#else
#define PLEX86_ALLOCVPHYS   0x6b02
#define PLEX86_ALLOCINT     0x6b03
#define PLEX86_RELEASEINT   0x6b04
#define PLEX86_RESET        0x6b05
#define PLEX86_MESSAGEQ     0x6b06
#define PLEX86_TEARDOWN     0x6b07
#define PLEX86_SETINTR      0x6b08
#define PLEX86_PRESCANDEPTH 0x6b09
#define PLEX86_SET_CPU      0x6b0c
#define PLEX86_RESET_CPU    0x6b0d
#define PLEX86_GET_CPU      0x6b0e
#define PLEX86_FORCE_INT    0x6b0f
#define PLEX86_SET_A20      0x6b10
#define PLEX86_PHYMEM_MOD   0x6b11
#define PLEX86_PRESCANRING3 0x6b12
#define PLEX86_REGTIMER     0x6b13
#define PLEX86_ACTTIMER     0x6b14
#define PLEX86_DEACTTIMER   0x6b15
#define PLEX86_REGIO        0x6b16
#define PLEX86_IRQ          0x6b17
#endif


#endif  /* #ifndef __PLEX86_H__ */
