/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001  Kevin P. Lawton
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <signal.h>

#include "dt.h"

static void timer_handler(int i);


/*
 * Some notes for myself.  Some may be old.
 * 
 * - Invalidate linear addr range (all, PDE, PTE, arbitrary from user space)
 * - Invalidate physical addr range.
 *   This would invalidate the Liaddr to Tiaddr table entries.  Would have
 *   to either dump that whole table or cycle through the local meta page
 *   table and hash all the linear addresses to find TLB entries to
 *   invalidate.  Perhaps for single page range hits, this would be good,
 *   and for big spans, dump the table.  Dumping the table could be
 *   expensive since we''d have to rebuild it.
 *   If we store only one laddr for each phy page meta info section, then
 *   we have to limit to only one translation per page.  Store Metai
 *   in phy meta info section so we can correlate.  Especially, DMA
 *   writes which have no associated laddr need correlation.
 * - Exceptions from DT code.
 *   o If tAddr within range of DT buffer:
 * - simple 1st effort would be to do exhaustive search of tables
 *   for that page.
 * - could have reverse lookup table
 *   o If tAddr within range of handlers:
 *     get Giaddr from storage area
 * 
 * - Ways to accelerate the shim code inserted for branches and certain
 *   other instruction:
 * 
 *   o If SS naturally has access to handler storage area, then could
 *     make use of that to accelerate branch handling.
 *   o Use other segment to do the same.  Even use specific handler
 *     data segment and patch use of that segment in guest code. (use
 *     infrequently used one).
 *   o Make use of guest stack.
 * 
 * Laundry list:
 * 
 * - Invalidation of Hash tables.  Dynamic zeroing?
 * - Incrementing of PDBR reload context switch ID.
 */


descriptor_t CS;
unsigned CPL;


int
main(int argc, char *argv[])
{
  Bit32u gOff, tcodeAddr, guest_EFLAGS, r3h_EFLAGS;

  {
    /*
     *  Hacks to pretend that the ring3 handler routines have
     *  a separate set of code and data segments than the
     *  guest code
     */

    /* Get the current DS and copy it to the pretend handler SS. */
    asm volatile (
      "pushl %%ds \n\t"
      "popl  %0"
       :"=g" (r3h_DS)
       :
       :"memory"
      );

    guest_SS = r3h_DS;

    /* The handler ESP will point into our stack area. */
    r3h_ESP = (unsigned) &r3h_stack[R3H_STACK_SIZE];
    r3h_ESP_empty = r3h_ESP;
    guest_ESP = (unsigned) &guest_stack[GUEST_STACK_SIZE];
    globalID = 1;

    /*
     *  Pretend GS is a virtualized segment register, and is
     *  the data segment for the R3H code.  Really, just copy
     *  the user data segment.
     */

    asm volatile (
      "pushl %ds \n\t"
      "popl  %gs \n\t"
      );
  }


  dtInitialize();
  CS.base = 0;
  CS.limit = 0xffffffff;


  hack_guest_code();

#if DT_DEBUG
  atexit(debug_dump);
  signal(SIGSEGV, debug_signal);
#endif

#if DT_ON

  {
    /* Assume execution begins at beginning of guest_page0 */
    gOff = (Bit32u) & guest_page0[0];

    /* Start by translating at that location */
    printf("Priming: translating initial guest code.\n");
    tcodeAddr = dtTranslateG2T(gOff);

    /*
     *  Transfer control to the tcode.  This would be a
     *  monitor-to-ring3 tcode transition in the real VM.
     */
    }

  {
    /*
     *  Prime the r3h stack for an initial transition, as if
     *  the r3h code had called the monitor and was executing
     *  the tail end of the __r3h_branch stub.
     */

    asm volatile (
      "pushfl; popl %0\n"
       :"=g" (guest_EFLAGS)
      );

    r3h_EFLAGS = guest_EFLAGS;

    /* The guest to r3h context saves */
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = guest_SS;   /* DS */
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = guest_SS;   /* ES */
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = 0;  /* General regsiters */
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = 0;
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = 0;
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = 0;
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = 0;
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = 0;
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = 0;
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = 0;
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = guest_EFLAGS;

    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = (Bit32u) __r3h_prime;
    r3h_target_EIP = tcodeAddr;

    /* The r3h to monitor saves */
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = r3h_EFLAGS;
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = 0;  /* r3h general regsiters */
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = 0;
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = 0;
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = 0;
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = 0;
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = 0;
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = 0;
    r3h_ESP -= 4;
    *((Bit32u *) r3h_ESP) = 0;
    }

  printf("Calling initial tcode.\n");

  {
    /*
     *  Set a timer to go off @ N Hz.  The handler can then
     *  simulate things that need to be done for context
     *  switches in the guest, with respect to tcode.
     */

    struct sigaction sg_act;
    struct itimerval timer;

    memset(&sg_act, 0, sizeof(sg_act));
    sg_act.sa_handler = timer_handler;
    sg_act.sa_flags = SA_RESTART;
    sigaction(SIGVTALRM, &sg_act, NULL);

    memset(&timer, 0, sizeof(timer));
    timer.it_value.tv_sec = 0;
    timer.it_value.tv_usec = DT_GuestTimeslice;
    timer.it_interval.tv_sec = 0;
    timer.it_interval.tv_usec = DT_GuestTimeslice;
    setitimer(ITIMER_VIRTUAL, &timer, NULL);
    }

  while (1) {
    __mon2r3h();

    switch (r3h_request) {
      case R3HToMonRequestG2T:
        r3h_data = dtTranslateG2T(r3h_data);
        break;

      case R3HToMonRequestTerminate:
        printf("Guest code finished execution.\n");
        return 0;
        break;

      default:
        printf("R3H Request default, data=0x%x\n", (unsigned) r3h_data);
        return 1;
        break;
      }
    }

#else /* DT_ON */

  {
    extern Bit32u loop_count;

    __execute_guest_code_native();

    printf("Control returns OK from native execution, count=%u.\n",
           (unsigned) loop_count);

    return 0;
    }

#endif /* DT_ON */

}


#if DT_ON

void
timer_handler(int i)
{
  globalID++;
}

#endif
