/*
    Bear Engine - Editor library

    Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/sprite.cpp
 * \brief Implementation of the bf::sprite class.
 * \author Julien Jorge
 */
#include "bf/sprite.hpp"

#include "bf/compiled_file.hpp"
#include "bf/path_configuration.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Default constructor.
 */
bf::sprite::sprite()
  : m_top(0), m_left(0), m_clip_width(0), m_clip_height(0)
{

} // sprite::sprite()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the name of the image resource to use.
 * \param name The new value.
 */
void bf::sprite::set_image_name( const std::string& name )
{
  m_image_name = name;
} // sprite::set_image_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the Y-coordinate in the image resource.
 * \param top The new value.
 */
void bf::sprite::set_top( const unsigned int top )
{
  m_top = top;
} // sprite::set_top()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the X-coordinate in the image resource.
 * \param left The new value.
 */
void bf::sprite::set_left( const unsigned int left )
{
  m_left = left;
} // sprite::set_left()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the width in the image resource.
 * \param width The new value.
 */
void bf::sprite::set_clip_width( const unsigned int width )
{
  m_clip_width = width;
} // sprite::set_clip_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set the height in the image resource.
 * \param height The new value.
 */
void bf::sprite::set_clip_height( const unsigned int height )
{
  m_clip_height = height;
} // sprite::set_clip_height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the left x-ccordinate of the sprite.
 */
unsigned int bf::sprite::get_left() const
{
  return m_left;
} // sprite::get_left()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the top y-coordinate of the sprite.
 */
unsigned int bf::sprite::get_top() const
{
  return m_top;
} // sprite::get_top()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the width of the sprite in the image source.
 */
unsigned int bf::sprite::get_clip_width() const
{
  return m_clip_width;
} // sprite::get_clip_width()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the height of the sprite in the image source.
 */
unsigned int bf::sprite::get_clip_height() const
{
  return m_clip_height;
} // sprite::get_clip_height()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the image name of the sprite.
 */
const std::string& bf::sprite::get_image_name() const
{
  return m_image_name;
} // sprite::get_image_name()

/*----------------------------------------------------------------------------*/
/**
 * \brief Compile the sprite.
 * \param f The stream in which we write the compiled sprite.
 */
void bf::sprite::compile( compiled_file& f ) const
{
  std::string image_path(m_image_name);

  if ( path_configuration::get_instance().expand_file_name(image_path) )
    path_configuration::get_instance().get_relative_path(image_path);

  f << image_path << m_left << m_top << m_clip_width << m_clip_height;

  bitmap_rendering_attributes::compile(f);
} // sprite::compile()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if two sprites are the same.
 * \param that The other sprite.
 */
bool bf::sprite::operator==( const sprite& that ) const
{
  return that.bitmap_rendering_attributes::operator==(*this)
    && (m_image_name == that.m_image_name)
    && (m_top == that.m_top)
    && (m_left == that.m_left)
    && (m_clip_width == that.m_clip_width)
    && (m_clip_height == that.m_clip_height);
} // sprite::operator==()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if two sprites are different.
 * \param that The other sprite.
 */
bool bf::sprite::operator!=( const sprite& that ) const
{
  return !(*this == that);
} // sprite::operator!=()
