/*
  Bear Engine

  Copyright (C) 2005-2010 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file block.cpp
 * \brief Implementation of the bear::block class.
 * \author Sebastien Angibaud
 */
#include "generic_items/block.hpp"

#include "universe/collision_info.hpp"
#include "engine/export.hpp"

BASE_ITEM_EXPORT( block, bear )

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 */
bear::block::block()
: m_collision_threshold(10)
{
  set_weak_collisions(false);
} // block::block()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a field of type \c real.
 * \param name The name of the field.
 * \param value The new value of the field.
 * \return false if the field "name" is unknow, true otherwise.
 */
bool
bear::block::set_u_integer_field( const std::string& name, unsigned int value )
{
  bool result(true);

  if ( name == "block.collision_threshold" )
    m_collision_threshold = value;
  else
    result = super::set_u_integer_field(name, value);

  return result;
} // block::set_u_integer_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Check if the collision is on a solid side and align the other item.
 * \param that The other item of the collision.
 * \param info Some informations about the collision.
 */
void bear::block::collision_check_and_align
( engine::base_item& that, universe::collision_info& info )
{
  if ( !satisfy_collision_condition(that) )
    return;

  bool align(false);
  double f;

  switch( choose_alignment_side(that, info) )
    {
    case universe::zone::bottom_zone:
      align = bottom_side_is_active() && collision_align_bottom(info);
      f = get_bottom_friction();
      break;
    case universe::zone::top_zone:
      align = top_side_is_active() && collision_align_top(info);
      f = get_top_friction();
      break;
    case universe::zone::middle_left_zone:
      align = left_side_is_active() && collision_align_left(info);
      f = get_left_friction();
      break;
    case universe::zone::middle_right_zone:
      align = right_side_is_active() && collision_align_right(info);
      f = get_right_friction();
      break;
    case universe::zone::middle_zone:
      align = default_collision(info);
      break;
    default: { CLAW_FAIL( "Invalid collision side." ); }
    }

  if ( align )
      {
        that.set_contact_friction(f);
        that.set_system_angle(0);
        z_shift(that);
      }
} // block::collision_check_and_align()

/*----------------------------------------------------------------------------*/
/**
 * \brief Choose the size on which the other item should be aligned.
 * \param that The other item of the collision.
 * \param info Some informations about the collision.
 */
bear::universe::zone::position
bear::block::choose_alignment_side
( const engine::base_item& that, const universe::collision_info& info ) const
{
  universe::zone::position result( info.get_collision_side() );

  switch( info.get_collision_side() )
    {
    case universe::zone::bottom_zone:
    case universe::zone::top_zone:
      {
        if ( (info.other_previous_state().get_left() >= get_right())
             && (that.get_left() >= get_right() - m_collision_threshold) )
          result = universe::zone::middle_right_zone;
        else if ( (info.other_previous_state().get_right() <= get_left())
                  && (that.get_right() <= get_left() + m_collision_threshold ) )
          result = universe::zone::middle_left_zone;
      }
      break;
    case universe::zone::middle_left_zone:
    case universe::zone::middle_right_zone:
      {
        if ( (info.other_previous_state().get_bottom() >= get_top())
             && (that.get_bottom() >= get_top() - m_collision_threshold) )
          result = universe::zone::top_zone;
        else if ( (info.other_previous_state().get_top() <= get_bottom())
                  && (that.get_top() <= get_bottom() + m_collision_threshold ) )
          result = universe::zone::bottom_zone;
      }
      break;
    case universe::zone::middle_zone:
      break;
    default: { CLAW_FAIL( "Invalid collision side." ); }
    }

  return result;
} // block::choose_alignment_side()

/*----------------------------------------------------------------------------*/
/**
 * \brief Call collision_check_and_align().
 * \param that The other item of the collision.
 * \param info Some informations about the collision.
 */
void bear::block::collision
( engine::base_item& that, universe::collision_info& info )
{
  collision_check_and_align(that, info);
} // block::collision()
