/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file plee.hpp
 * \brief The class describing the player (Plee).
 * \author Julien Jorge
 */
#ifndef __PTB_PLEE_HPP__
#define __PTB_PLEE_HPP__

#include "ptb/monster_item.hpp"
#include "ptb/item/plee/state_plee.hpp"
#include "ptb/item/plee/gauge.hpp"
#include "ptb/item/stone/stone.hpp"
#include "ptb/item_brick/item_with_single_player_control_reader.hpp"
#include "ptb/item_brick/item_that_speaks.hpp"

#include "engine/messageable_item.hpp"
#include "engine/model.hpp"
#include "communication/typed_message.hpp"

#include <set>
#include <list>

#include "engine/export.hpp"

namespace ptb
{
  /**
   * \brief The class describing the player (Plee).
   * \author Julien Jorge
   */
  class plee :
    public item_with_single_player_control_reader
    < item_that_speaks
      < monster_item
        < bear::engine::model
          < bear::engine::messageable_item
            < bear::engine::base_item> > > > >
  {
    DECLARE_BASE_ITEM(plee, ptb);
    
  public:
    /**
     * \brief A message to get a pointer on the instance of a player.
     * \author Sebastien Angibaud
     */
    class get_instance_message:
      public bear::communication::typed_message<plee>
    {
    public:
      get_instance_message();
      
      void clear();

      bool apply_to( plee& that );
      
      plee* get_instance() const;
      
    private:
      /** \brief Pointer on the instance of the plee. */
      plee* m_plee_instance;

    }; // class get_instance_message
    
    /** \brief The type of the parent class. */
    typedef item_with_player_control_reader
    < item_that_speaks
      < monster_item
        < bear::engine::model
          < bear::engine::messageable_item
            < bear::engine::base_item > > > > > super;
    
    enum plee_state_name
      {
        walk_state = 0,
        idle_state,
        jump_state,
        fall_state,
        dead_state,
	game_over_state,
        roar_state,
        run_state,
        slap_state,
        start_jump_state,
        vertical_jump_state,
        look_upward_state,
        crouch_state,
        captive_state,
        throw_state,
        maintain_state,
        wait_state,
        injured_state,
        start_cling_state,
        cling_state,
        clung_jump_state,
        start_hang_state,
        hang_state,
        swimming_state,
        sink_state,
        float_state
      }; // enum plee_state
    
    enum gauge_type
      {
        oxygen_gauge = 0,
        fire_gauge,
        ice_gauge
      }; // enum gauge_type

    enum object_type
      {
        none_object = 0,
        stone_object,
        air_honeypot_object,
        fire_honeypot_object,
        water_honeypot_object,
        azelnut_object
      }; // enum object_type
    
  private:
    typedef void (plee::*progress_function_type)(bear::universe::time_type);
    
  public:
    plee();
    ~plee();
    
    void progress( bear::universe::time_type elapsed_time );
    void get_visual( std::list<bear::engine::scene_visual>& visuals ) const;
    void pre_cache();
    void build();
    
    bool is_valid() const;
    bool set_u_integer_field( const std::string& name, unsigned int value );   
    void save_position( const bear::universe::position_type& p );
    
    void start_action( player_action::value_type a );
    void do_action
    ( bear::universe::time_type elapsed_time, player_action::value_type a );
    void stop_action( player_action::value_type a );
    void start_action_model(const std::string& action);
    
    const gauge* get_gauge(gauge_type t) const;
    
    double get_jump_time_ratio() const;
    void update_throw_time_ratio();
    void set_throw_down(bool value);
    void set_throw_up(bool value);
    void throw_azelnut();
    
    bool is_a_marionette() const ;
    void set_marionette( bool b);

    void set_air_float(bool status);

    void set_status_look_upward(bool status);
    bool get_status_look_upward() const;
    void set_status_crouch(bool status);
    bool get_status_crouch() const;
    void do_start_look_upward();
    void do_start_crouch();
    void do_start_throw();
    void do_start_drop();
    void set_can_cling(bool status);
    
    virtual void set_invincible(const bool invincible);
    void receive_energy(double energy);
    void receive_oxygen(double oxygen);    
    
    void add_air_stone(stone* s);
    void remove_air_stone(stone* s);
    void can_throw_power(bool b, monster::attack_type a);
    
    bear::universe::position_type hot_spot() const;
    unsigned int get_index() const;
    void set_index( unsigned int index );
    static std::string player_name( unsigned int player_index );
    
    void set_spot_minimum
    ( bear::universe::coordinate_type x, bear::universe::coordinate_type y );
    void set_spot_maximum
    ( bear::universe::coordinate_type x, bear::universe::coordinate_type y );
    void set_spot_balance_move
    ( bear::universe::coordinate_type x, bear::universe::coordinate_type y );
    
    void add_spot_gap( const bear::universe::position_type& gap );
    void balance_spot( bool x, bool y );
    
  public:
    void apply_drop();
    void apply_move_right();
    void apply_move_left();
    void apply_impulse_jump();
    void apply_jump();
    void apply_start_jump();
    void apply_vertical_jump();
    void apply_fall();
    void apply_idle();
    void apply_walk();
    void apply_run();
    void apply_slap();
    void apply_attack();
    void apply_chain();
    void apply_unchain();
    void apply_injured();
    void apply_throw();
    void apply_maintain();
    void apply_throw_object();
    void apply_blast_stone();
    void apply_wait();
    void apply_crouch();
    void apply_look_upward();
    void apply_die();
    void apply_roar();
    void apply_game_over();
    void apply_start_cling();
    void apply_cling();
    void apply_clung_jump();
    void apply_start_hang();
    void apply_hang();
    void apply_swimming();
    void apply_sink();
    void apply_swim_up();
    void apply_swim_down();
    void apply_float();
    void apply_swim_jump();
    void apply_dive();

  protected:
    virtual void execute_function( const std::string& name );
    virtual void inform_no_energy(const monster& attacker);
    virtual void injure
    (const monster& attacker, const bear::universe::collision_info& info);
    virtual void finish_injure();
    virtual unsigned int get_offensive_coefficient
    ( unsigned int index, const monster& other,
      const bear::universe::collision_info& info ) const;

    void to_string( std::string& str ) const;

  private:
    void progress_walk( bear::universe::time_type elapsed_time );
    void progress_idle( bear::universe::time_type elapsed_time );
    void progress_jump( bear::universe::time_type elapsed_time );
    void progress_fall( bear::universe::time_type elapsed_time );
    void progress_dead( bear::universe::time_type elapsed_time );
    void progress_game_over( bear::universe::time_type elapsed_time );
    void progress_roar( bear::universe::time_type elapsed_time );
    void progress_run( bear::universe::time_type elapsed_time );
    void progress_slap( bear::universe::time_type elapsed_time );
    void progress_start_jump( bear::universe::time_type elapsed_time );
    void progress_vertical_jump( bear::universe::time_type elapsed_time );
    void progress_crouch( bear::universe::time_type elapsed_time );
    void progress_look_upward( bear::universe::time_type elapsed_time );
    void progress_captive( bear::universe::time_type elapsed_time );
    void progress_throw( bear::universe::time_type elapsed_time );
    void progress_maintain( bear::universe::time_type elapsed_time );
    void progress_wait( bear::universe::time_type elapsed_time );
    void progress_start_cling( bear::universe::time_type elapsed_time );
    void progress_cling( bear::universe::time_type elapsed_time );
    void progress_clung_jump( bear::universe::time_type elapsed_time );
    void progress_start_hang( bear::universe::time_type elapsed_time );
    void progress_hang( bear::universe::time_type elapsed_time );
    void progress_swimming( bear::universe::time_type elapsed_time );
    void progress_sink( bear::universe::time_type elapsed_time );
    void progress_float( bear::universe::time_type elapsed_time );
    void progress_injured( bear::universe::time_type elapsed_time );

    void set_state(plee_state_name state);
    bool is_crushed() const;
    void regenerate();
    bool is_in_floating() const;
    bool has_a_power() const;

    void update_orientation();
    void update_power( bool b, monster::attack_type a);
    void update_powers();

    void do_get_camera();
    void do_stop_look_upward();
    void do_stop_crouch();
    void progress_spot( bear::universe::time_type elapsed_time );
    void progress_gauges( bear::universe::time_type elapsed_time );
    void progress_oxygen_gauge( bear::universe::time_type elapsed_time );
    void progress_fire_gauge( bear::universe::time_type elapsed_time );
    void progress_ice_gauge( bear::universe::time_type elapsed_time );

    bool test_walk();
    bool test_bottom_contact();
    bool test_in_sky_or_swimm();
    void brake();

    bear::universe::force_type get_stone_force() const;
    void create_stone( const bear::universe::force_type& a );
    void create_small_honeypot(const bear::universe::force_type& a );
    void create_azelnut();

    void render_halos
    ( std::list<bear::engine::scene_visual>& visuals ) const;
    void render_jump_halo
    ( std::list<bear::engine::scene_visual>& visuals ) const;
    void render_hand_halo
    ( std::list<bear::engine::scene_visual>& visuals ) const;
    void render_throw
    ( std::list<bear::engine::scene_visual>& visuals ) const;
    void get_visuals_without_invincibility
    ( std::list<bear::engine::scene_visual>& visuals ) const;
    void progress_invincibility( bear::universe::time_type elapsed_time );
    void progress_air_float( bear::universe::time_type elapsed_time );

  public:
    /** \brief The mass of Plee. */
    static const double s_mass;

    /** \brief Number of oxygen's units. */
    static const double s_max_oxygen;

    /** \brief Number of fire's units in gauge. */
    static const double s_max_fire_gauge;

    /** \brief Number of ice's units in gauge. */
    static const double s_max_ice_gauge;

    /** \brief The density of Plee. */
    static const double s_density;

  private:
    /** \brief What Plee is doing. */
    player_action::value_type m_current_action;

    /** \brief Last saved position of the center of mass. */
    bear::universe::position_type m_saved_position;

    /** \brief Last bottom_left_position. */
    bear::universe::position_type m_last_bottom_left;

    /** \brief The different states of Plee. */
    std::vector<state_plee*> m_states;

    /** \brief The current state. */
    plee_state_name m_current_state;

    /** \brief Current progress function. */
    progress_function_type m_progress;

    /** \brief Indicates the time since the last invincibily visual. */
    double m_last_visual_time;

    /** \brief Indicates the ratio of the length of preparation of the jump. */
    double m_jump_time_ratio;

    /** \brief Indicates the ratio of the preparation of throw state. */
    double m_throw_time_ratio;
    
    /** \brief The object that Plee has in its hand. */
    object_type m_current_object;

    /** \brief The current powers. */
    std::vector<gauge*> m_gauges;

    /** \brief Indicates if Plee look up. */
    bool m_status_look_upward;

    /** \brief Indicates the time of the look_upward. */
    bear::universe::time_type m_look_upward_time;

    /** \brief Indicates if Plee crouch. */
    bool m_status_crouch;

    /** \brief Indicates if Plee can cling. */
    bool m_can_cling;

    /** \brief The animation of the halo. */
    bear::visual::animation* m_halo_animation;

    /** \brief The animation of the halo on the hand. */
    bear::visual::animation* m_halo_hand_animation;

    /** \brief The list of last visual when Plee is invincible. */
    std::list< std::list<bear::engine::scene_visual> > m_last_visuals;

    /** \brief Indicates the time of the crouch. */
    bear::universe::time_type m_crouch_time;

    /** \brief Indicates the time since the state changement. */
    bear::universe::time_type m_state_time;

    /** \brief Indicates the time since plee want to run. */
    bear::universe::time_type m_run_time;

    /** \brief Indicates the time since plee is invincible. */
    bear::universe::time_type m_invincible_time;

    /** \brief Indicates if Plee want to throw up. */
    bool m_throw_up;

    /** \brief Indicates if Plee want to throw down. */
    bool m_throw_down;

    /** \brief The map of current air stones. */
    std::set<stone*> m_air_stones;

    /** \brief Indicates if Plee want to move right. */
    bool m_move_right;

    /** \brief Indicates if Plee want to move left. */
    bool m_move_left;

    /** \brief The current force in x axis when Plee move right. */
    double m_right_move_force;

    /** \brief The index of this player. */
    unsigned int m_index;

    /** \brief The number of succesive iteration for which this player
        has a bottom contact. */
    unsigned int m_nb_bottom_contact;

    /** \brief Indicates if Plee is a marionette. */
    bool m_marionette;

    /** \brief Indicates if Plee float in air. */
    bool m_air_float;

    /** \brief The time since the last change of air float status. */
    bear::universe::time_type m_air_float_time;

    /** \brief The animation to display in the hand. */
    bear::visual::animation* m_animation_to_throw;
    
    /** \brief Position of the hot spot, relative to the center of mass. */
    bear::universe::position_type m_hot_spot_position;
    
    /** \brief Minimum acceptable value for the position of the hot spot. */
    bear::universe::position_type m_hot_spot_minimum;
    
    /** \brief Maximum acceptable value for the position of the hot spot. */
    bear::universe::position_type m_hot_spot_maximum;
    
    /** \brief Maximum movement allowed when balancing the hot spot. */
    bear::universe::position_type m_hot_spot_balance_move;

    /** \brief A map that indicates if Plee can throw a power. */
    std::map<monster::attack_type,bool> m_can_throw_power;

    /** \brief Number of states considered. */
    static const unsigned int s_states_cardinality;

    /** \brief Number of gauges. */
    static const unsigned int s_gauges_cardinality;

    /** \brief The speed over which Plee is running. */
    static const int s_speed_to_run;

    /** \brief Number of oxygen's units loss in the water during one second. */
    static const double s_oxygen_loss_speed;

    /** \brief Number of oxygen's units won in the water during one second. */
    static const double s_oxygen_inspiration_speed;

    /** \brief Number of fire's units loss in the fire environment
     * during one second. */
    static const double s_fire_loss_speed;

    /** \brief Number of fire's units won in the fire environment
     * during one second. */
    static const double s_fire_increase_speed;

    /** \brief Number of ice's units loss in the ice environment
     * during one second. */
    static const double s_ice_loss_speed;

    /** \brief Number of ice's units won in the ice environment
     * during one second. */
    static const double s_ice_increase_speed;

    /** \brief The maximum halo height. */
    static const bear::universe::size_type s_max_halo_height;

    /** \brief The maximum halo hand width. */
    static const bear::universe::size_type s_max_halo_hand_width;

    /** \brief  The time to crouch. */
    static const bear::universe::time_type s_time_to_crouch;

    /** \brief  The time to look_upward. */
    static const bear::universe::time_type s_time_to_look_upward;

    /** \brief The time after which Plee is waiting. */
    static const bear::universe::time_type s_time_to_wait;

    /** \brief The time over which Plee jump automatically. */
    static const bear::universe::time_type s_time_to_jump;
    
        /** \brief The time over which Plee can run. */
    static const bear::universe::time_type s_time_to_run;

    /** \brief The time over which Plee throw automatically. */
    static const bear::universe::time_type s_time_to_start_throw;

    /** \brief The time over which Plee cannot cling. */
    static const bear::universe::time_type s_max_time_to_cling;

    /** \brief The time over which Plee cannot hang. */
    static const bear::universe::time_type s_max_time_to_hang;

    /** \brief The time over which Plee cannot float in air. */
    static const bear::universe::time_type s_max_time_air_float;
   }; // class plee
} // namespace ptb

#endif // __PTB_PLEE_HPP__
