/*
    Bear Engine - Level editor

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/level_file_xml_reader.cpp
 * \brief Implementation of the bf::level_file_xml_reader class.
 * \author Julien Jorge
 */
#include "bf/level_file_xml_reader.hpp"

#include "bf/gui_level.hpp"
#include "bf/item_class_pool.hpp"
#include "bf/layer.hpp"
#include "bf/wx_facilities.hpp"
#include "bf/xml/exception.hpp"
#include "bf/xml/item_instance_fields_node.hpp"
#include "bf/xml/reader_tool.hpp"

#include <claw/logger.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Load a level.
 * \param pool The pool of item classes in which we take the item classes.
 * \param file_path The path to the level file.
 */
bf::gui_level* bf::level_file_xml_reader::load
( const item_class_pool& pool, const wxString& file_path ) const
{
  wxXmlDocument doc;

  if ( !doc.Load(file_path) )
    throw std::ios_base::failure
      ( "Cannot load the XML file '" + wx_to_std_string(file_path) + "'" );

  wxXmlNode* node = doc.GetRoot();

  if ( node == NULL )
    throw xml::missing_node("level");

  return load_level( pool, node );
} // level_file_xml_reader::load()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load a node of type "level".
 * \param pool The pool of item classes in which we take the item classes.
 * \param node The node to parse.
 */
bf::gui_level* bf::level_file_xml_reader::load_level
( const item_class_pool& pool, wxXmlNode* node ) const
{
  if ( node->GetName() != wxT("level") )
    throw xml::bad_node( wx_to_std_string(node->GetName()) );

  unsigned int width, height;
  wxString val;

  width = xml::reader_tool::read_uint(node, wxT("width"));
  height = xml::reader_tool::read_uint(node, wxT("height"));

  val = node->GetPropVal( wxT("music"), wxT("") );

  gui_level* lvl = NULL;

  try
    {
      lvl = new gui_level( width, height, wx_to_std_string(val) );
      load_layers( pool, *lvl, node->GetChildren() );
    }
  catch( std::exception& e )
    {
      delete lvl;
      throw;
    }

  return lvl;
} // level_file_xml_reader::load_level()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the layers of the level.
 * \param pool The pool of item classes in which we take the item classes.
 * \param lvl The level in which we store the layers.
 * \param node The node to parse.
 */
void bf::level_file_xml_reader::load_layers
( const item_class_pool& pool, gui_level& lvl, wxXmlNode* node ) const
{
  for ( ; node!=NULL; node=node->GetNext() )
    if ( node->GetName() == wxT("layer") )
      load_layer(pool, lvl, node);
    else if ( node->GetName() != wxT("comment") )
      claw::logger << claw::log_warning << "Ignored node '"
                   << wx_to_std_string(node->GetName()) << "'" << std::endl;
} // level_file_xml_reader::load_layers()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load one layer.
 * \param pool The pool of item classes in which we take the item classes.
 * \param lvl The level in which we store the layer.
 * \param node The node to parse.
 */
void bf::level_file_xml_reader::load_layer
( const item_class_pool& pool, gui_level& lvl, wxXmlNode* node ) const
{
  CLAW_PRECOND( node->GetName() == wxT("layer") );

  unsigned int width, height;
  wxString val;

  width = xml::reader_tool::read_uint(node, wxT("width"));
  height = xml::reader_tool::read_uint(node, wxT("height"));

  if ( !node->GetPropVal( wxT("class_name"), &val ) )
    throw xml::missing_property( "class_name" );

  layer& lay = lvl.add_layer( width, height, wx_to_std_string(val) );

  load_layer_content( pool, lay, node->GetChildren() );
} // level_file_xml_reader::load_layer()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the content of a layer.
 * \param pool The pool of item classes in which we take the item classes.
 * \param lay The layer in which we store the instances.
 * \param node The node to parse.
 */
void bf::level_file_xml_reader::load_layer_content
( const item_class_pool& pool, layer& lay, wxXmlNode* node ) const
{
  for ( ; node!=NULL; node=node->GetNext() )
    if ( node->GetName() == wxT("item") )
      load_item(pool, lay, node);
    else if ( node->GetName() != wxT("comment") )
      claw::logger << claw::log_warning << "Ignored node '"
                   << wx_to_std_string(node->GetName()) << "'" << std::endl;
} // level_file_xml_reader::load_layer_content()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load an item of a layer.
 * \param pool The pool of item classes in which we take the item classes.
 * \param lay The layer in which we store the item.
 * \param node The node to parse.
 */
void bf::level_file_xml_reader::load_item
( const item_class_pool& pool, layer& lay, wxXmlNode* node ) const
{
  CLAW_PRECOND( node->GetName() == wxT("item") );

  wxString val;

  if ( !node->GetPropVal( wxT("class_name"), &val ) )
    throw xml::missing_property( "class_name" );

  std::string class_name( wx_to_std_string(val) );
  item_instance* item(NULL);

  try
    {
      item = new item_instance( pool.get_item_class_ptr(class_name) );
      item->set_fixed
        ( xml::reader_tool::read_bool_opt(node, wxT("fixed"), false) );
      item->set_id( wx_to_std_string(node->GetPropVal( wxT("id"), wxT("") )) );

      load_fields( *item, node->GetChildren() );

      lay.add_item(item);
    }
  catch( std::exception& e )
    {
      claw::logger << claw::log_error << e.what() << std::endl;
      delete item;
    }
} // level_file_xml_reader::load_item()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the fields of an item.
 * \param item The item in which we set the fields.
 * \param node The node to parse.
 */
void bf::level_file_xml_reader::load_fields
( item_instance& item, const wxXmlNode* node ) const
{
  xml::item_instance_fields_node reader;
  node = xml::reader_tool::skip_comments(node);

  if ( node->GetName() == wxT("fields") )
    reader.read(item, node);
  else
    claw::logger << claw::log_warning << "Ignored node '"
                 << wx_to_std_string(node->GetName()) << "'" << std::endl;
} // level_file_xml_reader::load_fields()
