/*
    Bear Engine - Editor library

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/xml/xml_to_value.cpp
 * \brief Implementation of the bf::xml::xml_to_value class.
 * \author Julien Jorge
 */
#include "bf/xml/xml_to_value.hpp"

#include "bf/wx_facilities.hpp"
#include "bf/xml/reader_tool.hpp"

#include <claw/logger.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Read the value from a xml value node.
 * \param spr (out) The sprite we have read.
 * \param node The node from which we read the value.
 */
void bf::xml::xml_to_value<bf::sprite>::operator()
  ( sprite& spr, const wxXmlNode* node ) const
{
  CLAW_PRECOND( node != NULL );

  wxString image_name;

  if ( !node->GetPropVal( wxT("image"), &image_name ) )
    throw missing_property("image");

  spr.set_image_name( wx_to_std_string(image_name) );

  spr.set_left( xml::reader_tool::read_uint(node, wxT("x")) );
  spr.set_top( xml::reader_tool::read_uint(node, wxT("y")) );
  spr.set_clip_width( xml::reader_tool::read_uint(node, wxT("clip_width")) );
  spr.set_clip_height( xml::reader_tool::read_uint(node, wxT("clip_height")) );
  spr.set_width
    ( xml::reader_tool::read_uint_opt
      (node, wxT("width"), spr.get_clip_width()) );
  spr.set_height
    ( xml::reader_tool::read_uint_opt
      (node, wxT("height"), spr.get_clip_height()) );
  spr.mirror( xml::reader_tool::read_bool_opt(node, wxT("mirror"), false));
  spr.flip( xml::reader_tool::read_bool_opt(node, wxT("flip"), false));
  spr.set_opacity( xml::reader_tool::read_real_opt(node, wxT("opacity"), 1) );
  spr.set_angle( xml::reader_tool::read_real_opt( node, wxT("angle"), 0) );

  double r = xml::reader_tool::read_real_opt(node, wxT("red_intensity"), 1 );
  double g = xml::reader_tool::read_real_opt(node, wxT("green_intensity"), 1 );
  double b = xml::reader_tool::read_real_opt(node, wxT("blue_intensity"), 1 );

  spr.set_intensity( r, g, b );
} // xml_to_value::operator()() [sprite]




/*----------------------------------------------------------------------------*/
/**
 * \brief Read the value from a xml value node.
 * \param anim (out) The animation we have read.
 * \param node The node from which we read the value.
 */
void bf::xml::xml_to_value<bf::animation>::operator()
  ( animation& anim, const wxXmlNode* node ) const
{
  CLAW_PRECOND( node != NULL );

  anim.set_loops( xml::reader_tool::read_uint(node, wxT("loops")) );
  anim.set_first_index
    ( xml::reader_tool::read_uint(node, wxT("first_index")) );
  anim.set_last_index
  ( xml::reader_tool::read_uint(node, wxT("last_index")) );
  anim.set_loop_back
  ( xml::reader_tool::read_bool_opt(node, wxT("loop_back"), false));

  anim.mirror( xml::reader_tool::read_bool_opt(node, wxT("mirror"), false));
  anim.flip( xml::reader_tool::read_bool_opt(node, wxT("flip"), false));
  anim.set_opacity( xml::reader_tool::read_real_opt(node, wxT("opacity"), 1) );

  anim.set_angle( xml::reader_tool::read_real_opt( node, wxT("angle"), 0) );

  double r = xml::reader_tool::read_real_opt(node, wxT("red_intensity"), 1 );
  double g = xml::reader_tool::read_real_opt(node, wxT("green_intensity"), 1 );
  double b = xml::reader_tool::read_real_opt(node, wxT("blue_intensity"), 1 );

  anim.set_intensity( r, g, b );

  load_frames(anim, node->GetChildren());

  anim.set_width
    ( xml::reader_tool::read_uint_opt
      (node, wxT("width"), anim.get_max_size().x) );
  anim.set_height
    ( xml::reader_tool::read_uint_opt
      (node, wxT("height"), anim.get_max_size().y) );

} // xml_to_value::operator()() [animation]

/*----------------------------------------------------------------------------*/
/**
 * \brief Load the frames of an animation.
 * \param anim The animation in which we set a list of frames.
 * \param node The node to parse.
 */
void bf::xml::xml_to_value<bf::animation>::load_frames
( animation& anim, const wxXmlNode* node ) const
{
  for ( ; node!=NULL; node=node->GetNext() )
    if ( node->GetName() == wxT("frame") )
      load_frame(anim, node);
    else if ( node->GetName() != wxT("comment") )
      claw::logger << claw::log_warning << "Ignored node '"
                   << wx_to_std_string(node->GetName()) << "'" << std::endl;
} // bf::xml::xml_to_value<bf::animation>::load_frames()

/*----------------------------------------------------------------------------*/
/**
 * \brief Load a frame of an animation.
 * \param anim The animation in which we add the frame.
 * \param node The node to parse.
 */
void bf::xml::xml_to_value<bf::animation>::load_frame
( animation& anim, const wxXmlNode* node ) const
{
  CLAW_PRECOND( node->GetName() == wxT("frame") );

  wxString val;
  animation_frame frame;
  sprite spr;

  if ( !node->GetPropVal( wxT("duration"), &val ) )
    throw missing_property( "duration" );

  frame.set_duration
    ( xml::reader_tool::read_real_opt(node, wxT("duration"), 40) );

  wxXmlNode* children = node->GetChildren();
  if ( children != NULL )
    {
      if ( children->GetName() == wxT("sprite") )
        {
          xml::xml_to_value<sprite> xml_conv;
          xml_conv(spr,children);
          frame.set_sprite(spr);
          anim.add_frame() = frame;
        }
      else if ( node->GetName() != wxT("comment") )
        claw::logger << claw::log_warning << "Ignored node '"
                     << wx_to_std_string(children->GetName())
                     << "'" << std::endl;
    }
  else
    throw missing_node("sprite");
} // bf::xml::xml_to_value<bf::animation>::load_frame()
