/*
    Bear Engine - Level editor

    Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

    This program is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This program is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    contact: plee-the-bear@gamned.org

    Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/impl/item_field_edit.tpp
 * \brief Implementation of the template methods of the bf::item_field_edit
 *        class.
 * \author Julien Jorge
 */
#include "bf/default_value.hpp"
#include "bf/dialog_maker.hpp"
#include "bf/free_edit.hpp"
#include "bf/human_readable.hpp"
#include "bf/set_edit.hpp"
#include "bf/interval_edit.hpp"
#include "bf/item_reference_edit.hpp"

#include <claw/assert.hpp>

/*----------------------------------------------------------------------------*/
/**
 * \brief Show the adequate dialog for editing a given field.
 * \param f The type of the field we are editing.
 * \param type The name of the type of the field.
 */
template<typename Type>
void bf::item_field_edit::show_simple_property_dialog
( const type_field& f, const wxString& type )
{
   switch ( f.get_range_type() )
    {
    case type_field::field_range_free:
      show_property_dialog< free_edit<Type> >(f, type);
      break;
    case type_field::field_range_set:
      show_property_dialog< set_edit<Type> >(f, type);
      break;
    case type_field::field_range_interval:
      show_property_dialog< interval_edit<Type> >(f, type);
      break;
    default:
      {
        CLAW_ASSERT(false, "range type is not valid.");
      }
    }
} // item_field_edit::show_simple_property_dialog()

/*----------------------------------------------------------------------------*/
/**
 * \brief Show the adequate dialog for editing a given field.
 * \param f The type of the field we are editing.
 * \param type The name of the type of the field.
 */
template<typename Control>
void bf::item_field_edit::show_property_dialog
( const type_field& f, const wxString& type )
{
  if ( f.is_list() )
    edit_field< Control, std::list<typename Control::value_type> >(f, type);
  else
    edit_field<Control, typename Control::value_type>(f, type);
} // item_field_edit::show_property_dialog()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create and show the dialog for editing the field and update the item.
 * \param f The field to edit.
 * \param type The name of the type of the field.
 */
template<typename Control, typename Type>
void
bf::item_field_edit::edit_field( const type_field& f, const wxString& type )
{
  typedef dialog_maker<Control, Type> dialog_maker_type;
  typedef typename dialog_maker_type::dialog_type dialog_type;

  dialog_type* dlg;

  if ( m_item->has_value(f) )
    {
      Type v;
      m_item->get_value( f.get_name(), v );
      dlg = dialog_maker_type::create(*this, type, f, v);
    }
  else
    dlg = dialog_maker_type::create(*this, type, f, default_value<Type>::get());

  show_dialog(f.get_name(), *dlg);
  dlg->Destroy();
} // item_field_edit::edit_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create and show the dialog for editing the field and update the item.
 * \param f The field to edit.
 * \param values The valid item identifiers for this field.
 */
template<typename Type>
void bf::item_field_edit::edit_item_reference_field
( const type_field& f, const wxArrayString& values )
{
  typedef value_editor_dialog<item_reference_edit, Type> dialog_type;

  dialog_type* dlg;

  if ( m_item->has_value(f) )
    {
      Type v;
      m_item->get_value( f.get_name(), v );
      dlg = new dialog_type(*this, _("item"), values, v);
    }
  else
    dlg = new dialog_type(*this, _("item"), values, default_value<Type>::get());

  show_dialog(f.get_name(), *dlg);
  dlg->Destroy();
} // item_field_edit::edit_item_reference_field()

/*----------------------------------------------------------------------------*/
/**
 * \brief Show a created dialog for editing the field and update the item.
 * \param field_name The name of the edited field to edit.
 * \param type The name of the type of the field.
 */
template<typename DialogType>
void bf::item_field_edit::show_dialog
( const std::string& field_name, DialogType& dlg )
{
  if ( dlg.ShowModal() == wxID_OK )
    {
      m_proxy.set_field_value( *m_item, field_name, dlg.get_value() );
      update_values();
    }
} // item_field_edit::show_dialog()

/*----------------------------------------------------------------------------*/
/**
 * \brief Convert the value of a field in a string.
 * \param field_name The name of the field for which we want the value.
 */
template<typename Type>
wxString bf::item_field_edit::convert_value_to_text
( const std::string& field_name ) const
{
  Type val;
  m_item->get_value( field_name, val );

  return human_readable<Type>::convert( val );
} // item_field_edit::convert_value_to_text()
