/*
  Bear Engine - Editor library

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [Bear] in the subject of your mails.
*/
/**
 * \file bf/code/class_tree_ctrl.cpp
 * \brief Implementation of the bf::class_tree_ctrl class.
 * \author Julien Jorge
 */
#include "bf/class_tree_ctrl.hpp"

#include "bf/item_class_pool.hpp"
#include <wx/dnd.h>

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param pool The classes to display in the tree.
 * \param parent Pointer to the owner.
 * \param id Identifier of the control.
 * \param show_abstract Tell if the abstract classes are displayed too.
 */
bf::class_tree_ctrl::class_tree_ctrl
( const item_class_pool& pool, wxWindow* parent, int id, bool show_abstract )
  : wxTreeCtrl( parent, id, wxDefaultPosition, wxDefaultSize,
                wxTR_DEFAULT_STYLE | wxTR_FULL_ROW_HIGHLIGHT | wxTR_HIDE_ROOT )
{
  create_tree(pool, show_abstract);
} // class_tree_ctrl::class_tree_ctrl()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the tree of classes.
 * \param pool The classes to display in the tree.
 * \param show_abstract Tell if the abstract classes are displayed too.
 */
void bf::class_tree_ctrl::create_tree
( const item_class_pool& pool, bool show_abstract )
{
  AddRoot( _("Item classes") );

  tree_builder tb;

  create_categories_tree(pool, tb, show_abstract);
  tb.create_wxTree( *this );
  ExpandAll();
} // class_tree_ctrl::create_tree()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the tree of classes.
 * \param pool The classes to display in the tree.
 * \param tb (out) The tree builder.
 * \param show_abstract Tell if the abstract classes are displayed too.
 */
void bf::class_tree_ctrl::create_categories_tree
( const item_class_pool& pool, tree_builder& tb, bool show_abstract ) const
{
  item_class_pool::const_iterator it = pool.begin();
  const item_class_pool::const_iterator eit = pool.end();

  for ( ; it!=eit; ++it )
    if ( (it->get_category() != "-abstract-") || show_abstract )
      {
        const std::string cat( it->get_category() );

        if ( cat.empty() )
          tb.add_entries( it->get_class_name(), '/' );
        else
          tb.add_entries( cat + '/' + it->get_class_name(), '/' );
      }
} // class_tree_ctrl::create_categories_tree()

/*----------------------------------------------------------------------------*/
/**
 * \brief Select a class in the tree.
 * \param all_children Tell if we expand the whole subtree (if selected item is
 *        not a leaf).
 */
void bf::class_tree_ctrl::select_class(bool all_children)
{
  wxTreeItemId item = GetSelection();

  if ( item.IsOk() )
    if ( ItemHasChildren(item) )
      {
        if ( IsExpanded(item) )
          {
            if ( all_children )
              CollapseAllChildren(item);
            else
              Collapse(item);
          }
        else
          {
            if ( all_children )
              ExpandAllChildren(item);
            else
              Expand(item);
          }
      }
} // class_tree_ctrl::select_class()

/*----------------------------------------------------------------------------*/
/**
 * \brief Procedure called when a double click occurs on a tree.
 * \param event This event occured.
 */
void bf::class_tree_ctrl::on_double_click(wxMouseEvent& event)
{
  select_class(event.ShiftDown());
} // class_tree_ctrl::on_double_click()

/*----------------------------------------------------------------------------*/
/**
 * \brief The user pressed a keybord key.
 * \param event The keyboard event that occured.
 */
void bf::class_tree_ctrl::on_key_up(wxKeyEvent& event)
{
  switch( event.GetKeyCode() )
    {
    case WXK_RETURN:
    case WXK_NUMPAD_ENTER:
      select_class( event.ShiftDown() );
      break;
    default:
      event.Skip();
    }
} // class_tree_ctrl::on_key_up()

/*----------------------------------------------------------------------------*/
/**
 * \brief Procedure called when the mouse moves over the control.
 * \param event This event occured.
 */
void bf::class_tree_ctrl::on_mouse_move(wxMouseEvent& event)
{
  if ( event.LeftIsDown() )
    {
      wxTreeItemId item = GetSelection();

      if ( item.IsOk() )
        if ( !ItemHasChildren(item) )
          {
            wxTextDataObject class_name(GetItemText(item));
            wxDropSource drag_source( this );

            drag_source.SetData( class_name );
            drag_source.DoDragDrop( TRUE );
          }
    }
  else
    event.Skip();
} // class_tree_ctrl::on_mouse_move()

/*----------------------------------------------------------------------------*/
BEGIN_EVENT_TABLE(bf::class_tree_ctrl, wxTreeCtrl)
  EVT_LEFT_DCLICK( bf::class_tree_ctrl::on_double_click )
  EVT_KEY_UP( bf::class_tree_ctrl::on_key_up )
  EVT_MOTION( bf::class_tree_ctrl::on_mouse_move )
END_EVENT_TABLE()
