/***************************************************************************
 *   Copyright (C) 2008 by Vinnie Futia <rowancompsciguy@gmail.com>      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .        *
 ***************************************************************************/

#include "plasma-weather.h"
#include "config.h"

#include <QPainter>
#include <QFontMetrics>
#include <QStyleOptionGraphicsItem>
#include <QSizeF>
#include <QRectF>
#include <QFile>
#include <KGlobalSettings>
#include <QProcess>
#include <QDomDocument>
#include <QTimer>
#include <QDir>
#include <KConfigGroup>

#include <plasma/svg.h>
#include <plasma/theme.h>

Plasma_Weather::Plasma_Weather(
    QObject *parent, const QVariantList &args  )
    : Plasma::Applet(parent, args)
{
   titleFont = QFont("FreeSans", 10, false);
   tempFont = QFont("FreeSans", 24, false);
   timer = new QTimer(this);
   resize(250, 125);
   setHasConfigurationInterface(true);

   connect(timer, SIGNAL(timeout()), this, SLOT(refresh()));
}
 
Plasma_Weather::~Plasma_Weather()
{
   timer->stop();
   delete timer;
}

void Plasma_Weather::init()
{
   refresh();
}

void Plasma_Weather::refresh()
{
   bool error = false;

   readConfigData();
   collectData();
   parseData();
   update();

   timer->stop();
   timer->setInterval(updateFrequency.toInt(&error) * 60 * 1000);
   timer->start();
}

void Plasma_Weather::readConfigData()
{
   KConfigGroup cg = config();

   iconTheme = cg.readEntry("iconTheme", iconTheme);
   zip = cg.readEntry("zip", "USDC0001");
   tempType = cg.readEntry("tempType", "F");
   updateFrequency = cg.readEntry("updateFrequency", "5");
} 
 
void Plasma_Weather::collectData()
{
    //retrive the weather file
    QProcess* process = new QProcess(this);
    QString command = "wget -O " + QDir::homePath() + "/.weather.xml http://weather.yahooapis.com/forecastrss?p=" + zip;

    if(tempType == "C")
    {
       command.append("&u=c");
    }
    
    process->start(command);
    process->waitForFinished();
    delete process;
}

void Plasma_Weather::parseData()
{
    //get the invidual elements from the XML file
    QDomDocument doc("weather");
    QFile file(QDir::homePath() + "/.weather.xml");

    //read the data into the XML parser and close the file
    doc.setContent(&file);
    file.close();

    QDomElement root = doc.documentElement();

    QDomNode n = root.firstChild();

    n = root.firstChild();
    n = n.firstChild();

    while(!n.isNull())
    {
       QDomElement e = n.toElement();

       if(!e.isNull())
       {
          if(e.tagName() == "yweather:location")
          {
             city = e.attribute("city", "");
             region = e.attribute("region", "");
             country = e.attribute("country", "");
          }
          else if(e.tagName() == "yweather:wind")
          {
             temperature = e.attribute("chill", "");
          }
          else if(e.tagName() == "yweather:atmosphere")
          {
             humidity = e.attribute("humidity", "");
          }
          else if(e.tagName() == "item")
          {
             n = n.firstChild();

             while(!n.isNull())
             {
                QDomElement e = n.toElement();

                if(!e.isNull())
                {
                   if(e.tagName() == "yweather:condition")
                   {
                      condition = e.attribute("text", "");
                      code = e.attribute("code", "");
                   }
                }
               
                 n = n.nextSibling();
             } 
           
             n = n.parentNode();
          }
       }

       n = n.nextSibling();
    }    
}

void Plasma_Weather::paintInterface(QPainter *p,
        const QStyleOptionGraphicsItem *option, const QRect &contentsRect)
{
    Q_UNUSED( option );

    QRect textRect(contentsRect.left(), contentsRect.top(), 
          (int)(contentsRect.width() / 2.0), (int)contentsRect.height());
    QRect imageRect(contentsRect.left() + (int)(contentsRect.width() / 2.0), contentsRect.top(), 
          (int)contentsRect.height(), (int)contentsRect.height());

    p->setRenderHint(QPainter::SmoothPixmapTransform);
    p->setRenderHint(QPainter::Antialiasing);
 
    // We place the icon and text
    int width = (int)contentsRect.width();
    int height = (int)contentsRect.height();
    paintImage(p, imageRect);
    p->save();
    
    p->setPen(Qt::white);
    p->setFont(titleFont);

    if(region != "")
    {
       p->drawText(textRect,
          Qt::AlignTop | Qt::AlignHCenter,
          city + ", " + region); 
    }
    else
    {
       p->drawText(textRect,
          Qt::AlignTop | Qt::AlignHCenter,
          city + ", " + country); 
    }

    p->drawText(textRect,
       Qt::AlignBottom | Qt::AlignHCenter,
       condition);

    p->setFont(tempFont);

    p->drawText(textRect,
       Qt::AlignVCenter | Qt::AlignHCenter,
       temperature + char(176) + tempType);

    p->restore();
}

void Plasma_Weather::paintImage( QPainter *p, const QRect &imageRect )
{
    QString svg_file = QString();

    if(iconTheme == "")
    {
       svg_file = "widgets/weather";
    }
    else
    {
       svg_file = iconTheme;  
    }

    Plasma::Svg* svg;
    svg = new Plasma::Svg(this);
    svg->setImagePath( svg_file );
    svg->setContainsMultipleImages(true);
    svg->resize(imageRect.width(), imageRect.height());

    svg->paint(p, imageRect, code);
}

void Plasma_Weather::createConfigurationInterface(KConfigDialog* parent)
{
   bool error = false;
   conf = new ConfigDialog();
   parent->setButtons(KDialog::Ok | KDialog::Cancel | KDialog::Apply);
   parent->addPage(conf, parent->windowTitle(), icon());

   conf->zipText->setText(zip);
   conf->iconThemeText->setText(iconTheme);
   conf->freqChooser->setValue(updateFrequency.toInt(&error));
   
   if(tempType == "F")
   {
      conf->fahrenheitSelect->setChecked(true);
   }
   else
   {
      conf->celsiusSelect->setChecked(true);
   }

   connect(parent, SIGNAL(applyClicked()), this, SLOT(configAccepted()));
   connect(parent, SIGNAL(okClicked()), this, SLOT(configAccepted()));
   connect(parent, SIGNAL(cancelClicked()), this, SLOT(configRejected()));
}

void Plasma_Weather::configAccepted()
{
   KConfigGroup cg = config();
   cg.writeEntry("zip", conf->zipText->text());

   if(conf->fahrenheitSelect->isChecked())
   {
      cg.writeEntry("tempType", "F");
   }
   else
   {
      cg.writeEntry("tempType", "C");
   }

   cg.writeEntry("iconTheme", conf->iconThemeText->text());
   cg.writeEntry("updateFrequency", updateFrequency.setNum(conf->freqChooser->value()));

   emit configNeedsSaving();

   refresh();
   
   delete conf;
}

void Plasma_Weather::configRejected()
{
   delete conf;
}

Qt::Orientations Plasma_Weather::expandingDirections() const
{
    // no use of additional space in any direction
    return 0;
}

#include "plasma-weather.moc" 
