/******************************************************************************
*                             plasma-am4rok                                   *
*******************************************************************************
*                                                                             *
*            Copyright (C) 2008 Giulio Camuffo				      *                             *									      *
*                                                                             *
*                                                                             *
*   This program is free software; you can redistribute it and/or modify      *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 2 of the License, or         *
*   (at your option) any later version.                                       *
*                                                                             *
*   This program is distributed in the hope that it will be useful,           *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License         *
*   along with this program; if not, write to the Free Software               *
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA  *
*                                                                             *
*                                                                             *
*                                                                             *
*   For further information contact me at giuliocamuffo@gmail.com	      *
******************************************************************************/

#include "player_controller.h"
#include "mmbuttonwidget.h"

#include <QDBusInterface>
#include <QDBusReply>
#include <QProcess>
#include <QVariant>

QDBusInterface amarok2PlayerDbus( "org.kde.amarok", "/Player", "org.kde.Amarok.Player" );
QDBusInterface amarok2PlaylistDbus("org.kde.amarok", "/Playlist", "org.kde.Amarok.Playlist");
QDBusInterface jukPlayerDbus( "org.kde.juk", "/Player", "org.kde.juk.player" );

PlayerController::PlayerController(){

  int status_interval=1000;
  timer.connect(&timer,SIGNAL(timeout(void)),
		this,SLOT(playerStatus()));
  timer.start(status_interval);

  m_status = 0;
}

PlayerController::~PlayerController(){
  timer.stop();
}

QString PlayerController::upFirstLetter(const QString &string){

  QString new_string;
  for (int i=0; i < string.size(); i++){
    if (i == 0)
      new_string = string[0].toUpper();
    else
      new_string += string[i];
  }
    
  return new_string;
}

void PlayerController::setPlayer(int &player){
  m_player = player;
}

int PlayerController::player(void){
  return m_player;
}

int PlayerController::playerIntQuery(const QString &query){

  if (m_player == 0){
    QDBusReply<int> reply = amarok2PlayerDbus.call( query );
    if ( reply.isValid() ) {
      return reply.value();
    }
    else
	return -1;
  }
  if (m_player == 1){
    if (query == "trackCurrentTime")
      return playerIntQuery("currentTime");
    if (query == "trackTotalTime")
      return playerIntQuery("totalTime");

    QDBusReply<int> reply = jukPlayerDbus.call( query );
    if ( reply.isValid() ) {
      return reply.value();
    }
    else
	return -1;
  }

  return -2;
}

QString PlayerController::playerStringQuery(const QString &query){

  if (m_player == 0){
    QDBusReply<QString> reply = amarok2PlayerDbus.call( query );
    if ( reply.isValid() ) {
		return reply.value();
    }
    else
	return QString("");
  }
  if (m_player == 1){
    if (query == "currentTime"){
      int cur_time = playerIntQuery("currentTime");
      if (cur_time != -1){
		QVariant modInt = QVariant(cur_time%60);
		QString mod ;
		if (modInt.toInt() < 10)
		  mod = QString("0" + modInt.toString());
		else
		  mod = modInt.toString();
		return QString(QVariant(cur_time/60).toString() + ":" + mod);
      }
      return QString("");
    }
    if (query == "totalTime"){
      int tot_time = playerIntQuery("totalTime");
      if (tot_time != -1){
		QVariant modInt = QVariant(tot_time%60);
		QString mod ;
		if (modInt.toInt() < 10)
		  mod = QString("0" + modInt.toString());
		else
		  mod = modInt.toString();
		return QString(QVariant(tot_time/60).toString() + ":" + mod);
      }
      return QString("");
    }
    QDBusReply<QString> reply = jukPlayerDbus.call("trackProperty", upFirstLetter(query));
    if ( reply.isValid() ) {
      return reply.value();
    }
    else
	return QString("");
  }

  return QString(" ");

}

int PlayerController::playlistIntQuery(const QString &query){

  if (m_player == 0){
    QDBusReply<int> reply = amarok2PlaylistDbus.call( query );
    if ( reply.isValid() ) {
      return reply.value();
    }
    else
	return -1;
  }
  if (m_player == 1){
    if (query == "trackCurrentTime")
      return playerIntQuery("currentTime");
    if (query == "trackTotalTime")
      return playerIntQuery("totalTime");

    QDBusReply<int> reply = jukPlayerDbus.call( query );
    if ( reply.isValid() ) {
      return reply.value();
    }
    else
	return -1;
  }

  return -2;
}

QString PlayerController::playlistStringQuery(const QString &query){

  if (m_player == 0){
    QDBusReply<QString> reply = amarok2PlaylistDbus.call( query );
    if ( reply.isValid() ) {
		return reply.value();
    }
    else
	return QString("");
  }
  if (m_player == 1){
    QDBusReply<QString> reply = jukPlayerDbus.call("trackProperty", upFirstLetter(query));
    if ( reply.isValid() ) {
      return reply.value();
    }
    else
	return QString("");
  }

  return QString(" ");

}

QString PlayerController::playerTimeQuery(void){
    QString current_time;
    if ((current_time = playerStringQuery("currentTime")) != ""){
      QString total_time = playerStringQuery("totalTime");
      QString time = current_time + "/" + total_time;
      return time;
    }
    else
      return QString("");
}

QString PlayerController::playerTrackQuery(void){
  QVariant current_track = (playlistIntQuery("getActiveIndex") +1);
  QVariant total_tracks = playlistIntQuery("getTotalTrackCount");
  if (total_tracks != -1)
    return QString(current_track.toString() + "/" + total_tracks.toString());
  else
    return QString("");
}

void PlayerController::playerQuery(const QString &query){
  if (m_player == 0){
    if (query == "seekForward"){
      amarok2PlayerDbus.call("seekRelative", 3);
      return;
    }
    if (query == "seekBack"){
      amarok2PlayerDbus.call("seekRelative", -3);
      return;
    }
    amarok2PlayerDbus.call(query);
    return;
  }
  if (m_player == 1){
    if (query == "prev"){
      jukPlayerDbus.call("back");
      return;
    }
    if (query == "next"){
      jukPlayerDbus.call("forward");
      return;
    }
    jukPlayerDbus.call(query);
    return;
  }
}

void PlayerController::playerQuery(const QString &query, const int &arg){
  if (m_player == 0){
    amarok2PlayerDbus.call(query, arg);
    return;
  }
  if (m_player == 1){
    jukPlayerDbus.call(query, arg);
    return;
  }
}

void PlayerController::changeState(const QString &state){
    
    if (playerIntQuery("status") == -1){

      if(state=="playPause"){
		if (m_player == 0)
	  		QProcess::execute("amarok");
		else{
	  		if (m_player == 1)
	   			QProcess::execute("juk");
	}
	// if it's closed we launch it,
	// and start playing (unless resume is on
	// and it starts on its own)
	int status=playerIntQuery("status");
	if(status!=MMButtonWidget::PLAYING)
	      playerQuery("playPause");
      }
   }

  //here calls the dbus metods  
  playerQuery(state);

  playerStatus();
}

void PlayerController::updateInfos(void){

    double cur_time=0,
           tot_time=0;
    int intTime=0;

    if((cur_time=playerIntQuery("trackCurrentTime"))>=0 &&
       (tot_time=playerIntQuery("trackTotalTime"))>0){
        intTime = ((cur_time/tot_time)*100.0);
    }
    if (intTime != m_intTime)
      emit intTimeChanged(intTime);

    
    QString title = "";
    if ((title = playerStringQuery("track")) != "")
      title = (playerStringQuery("track") +  " - " +playerStringQuery("title"));
    QString artist = playerStringQuery("artist");
    QString album = playerStringQuery("album");
    QString time = playerTimeQuery();
    QString track = playerTrackQuery();
	
    if (time != m_time){
      emit timeChanged(time);
      m_time = time;
    }
    
    if (title != m_title){
      emit titleChanged(title);
      m_title = title;
    }
    if (artist != m_artist){
      emit artistChanged(artist);
      m_artist = artist;
    }
    if (album != m_album){
      emit albumChanged(album);
      m_album = album;
    }
    if (track != m_track){
      emit trackChanged(track);
      m_track = track;
    }    
}

void PlayerController::forceUpdateInfos(void){

    double cur_time=0,
           tot_time=0;
    int intTime=0;

    if((cur_time=playerIntQuery("trackCurrentTime"))>=0 &&
       (tot_time=playerIntQuery("trackTotalTime"))>0){
        intTime = ((cur_time/tot_time)*100.0);
    }
    emit intTimeChanged(intTime);

    QString title = (playerStringQuery("track") +  " - " +playerStringQuery("title"));
    QString artist = playerStringQuery("artist");
    QString album = playerStringQuery("album");
    QString time = playerTimeQuery();
    QString track = playerTrackQuery();
    int status=playerIntQuery("status");

    emit timeChanged(time);
    m_time = time;
     
    emit titleChanged(title);
    m_title = title;
  
    emit artistChanged(artist);
    m_artist = artist;
  
    emit albumChanged(album);
    m_album = album;  
    
    emit trackChanged(track);
    m_track = track;      

    QString s_stat;
    switch(status){
        case MMButtonWidget::PLAYING:
            s_stat="PLAYING";
            break;
        case MMButtonWidget::STOPPED:
            s_stat="STOPPED";
            break;
        case MMButtonWidget::PAUSED:
            s_stat="PAUSED";
            break;
        default:
            s_stat="CLOSED";
    }
    
    emit statusChanged(s_stat);
    m_status = status;    
}

void PlayerController::playerStatus(void){

    QString playstr,
            s_stat;
    int status=playerIntQuery("status");

    switch(status){
        case MMButtonWidget::PLAYING:
            s_stat="PLAYING";
            break;
        case MMButtonWidget::STOPPED:
            s_stat="STOPPED";
            break;
        case MMButtonWidget::PAUSED:
            s_stat="PAUSED";
            break;
        default:
            s_stat="CLOSED";
    }

  if (m_status != status){
      emit statusChanged(s_stat);
      m_status = status;
  }

  if (status != 0)
    updateInfos();
  else
    stopped();

}

void PlayerController::stopped(void){

  emit intTimeChanged(0);

  emit timeChanged("");
  m_time = QString("");
 
  emit titleChanged("");
  m_title = QString("");

  emit artistChanged("");
  m_artist = QString("");

  emit albumChanged("");
  m_album = QString("");
  
  emit trackChanged("");
  m_track = QString("");
 
}

#include "player_controller.moc"
