/***********************************************************************************
* Run Command: Simple plasmoid to run commands with support for runners.
* Copyright (C) 2008 - 2009 Michal Dutkiewicz aka Emdek <emdeck@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "RunCommandApplet.h"
#include "RunCommandItem.h"

#include <QDir>
#include <QRegExp>
#include <QFileInfo>
#include <QScriptEngine>
#include <QGraphicsView>
#include <QGraphicsLinearLayout>

#include <KRun>
#include <KUrl>
#include <KIcon>
#include <KShell>
#include <KService>
#include <KLineEdit>
#include <KAuthorized>
#include <KCompletion>
#include <NETRootInfo>
#include <KWindowInfo>
#include <KWindowSystem>
#include <KStandardDirs>
#include <KProtocolInfo>
#include <KGlobalSettings>
#include <KToolInvocation>
#include <KServiceTypeTrader>
#include <kworkspace/kworkspace.h>
#include <kworkspace/kdisplaymanager.h>

#include <Plasma/Theme>
#include <Plasma/ComboBox>

K_EXPORT_PLASMA_APPLET(runcommand, RunCommandApplet)

RunCommandApplet::RunCommandApplet(QObject *parent, const QVariantList &args) : Plasma::Applet(parent, args)
{
    setAspectRatioMode(Plasma::IgnoreAspectRatio);

    resize(200, 50);
}

void RunCommandApplet::init()
{
    KConfig runnersConfiguration("krunnerrc", KConfig::SimpleConfig);
    QStringList commands = KConfigGroup(&runnersConfiguration, "General").readEntry("PastQueries", QStringList());

    m_enableRunnersAction = new QAction(i18n("Enable Runners"), this);
    m_enableRunnersAction->setCheckable(true);

    m_manager = new Plasma::RunnerManager(this);

    m_resultsDialog = new Plasma::Dialog;

    m_resultsUi.setupUi(m_resultsDialog);
    m_resultsUi.resultsLayout->setAlignment(Qt::AlignCenter);

    m_lastRun = QDateTime::currentDateTime();

    m_comboBox = new KHistoryComboBox(false);
    m_comboBox->setAttribute(Qt::WA_NoSystemBackground);
    m_comboBox->setSizeAdjustPolicy(QComboBox::AdjustToMinimumContentsLength);
    m_comboBox->setHistoryItems(commands);
    m_comboBox->setEditable(true);
    m_comboBox->clearEditText();

    KLineEdit *lineEdit = static_cast<KLineEdit*>(m_comboBox->lineEdit());
    lineEdit->installEventFilter(this);
    lineEdit->setCompletionObject(new KCompletion());
    lineEdit->setCompletionMode(KGlobalSettings::CompletionAuto);
    lineEdit->completionObject()->insertItems(commands);
    lineEdit->completionObject()->setOrder(KCompletion::Sorted);

    Plasma::ComboBox *comboBox = new Plasma::ComboBox(this);

    m_comboBox->setStyle(comboBox->widget()->style());

    comboBox->setWidget(m_comboBox);

    QGraphicsLinearLayout *layout = new QGraphicsLinearLayout;
    layout->setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding));
    layout->setSpacing(0);
    layout->addItem(comboBox);

    setLayout(layout);

    setPreferredWidth(QWIDGETSIZE_MAX);

    constraintsEvent(Plasma::FormFactorConstraint);

    toggleEnableRunners(config().readEntry("enableRunners", true));

    updateTheme();

    connect(this, SIGNAL(activate()), this, SLOT(focusWidget()));
    connect(this, SIGNAL(destroyed()), m_resultsDialog, SLOT(deleteLater()));
    connect(m_enableRunnersAction, SIGNAL(toggled(bool)), this, SLOT(toggleEnableRunners(bool)));
    connect(m_comboBox, SIGNAL(cleared()), this, SLOT(clearHistory()));
    connect(m_comboBox, SIGNAL(returnPressed()), this, SLOT(getCommand()));
    connect(m_comboBox, SIGNAL(activated(QString)), this, SLOT(runCommand(QString)));
    connect(m_comboBox->lineEdit(), SIGNAL(textChanged(QString)), this, SLOT(queryChanged(QString)));
    connect(m_manager, SIGNAL(matchesChanged(QList<Plasma::QueryMatch>)), this, SLOT(resultsChanged(QList<Plasma::QueryMatch>)));
    connect(Plasma::Theme::defaultTheme(), SIGNAL(themeChanged()), this, SLOT(updateTheme()));
}

void RunCommandApplet::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    if (event->button() == Qt::LeftButton)
    {
        focusWidget();
    }
}

void RunCommandApplet::constraintsEvent(Plasma::Constraints constraints)
{
    if (constraints & Plasma::FormFactorConstraint)
    {
        qreal left, top, right, bottom;

        getContentsMargins(&left, &top, &right, &bottom);

        layout()->setContentsMargins(3, 0, 3, 0);

        if (formFactor() == Plasma::Horizontal || formFactor() == Plasma::Vertical)
        {
            m_comboBox->setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding));

            setMinimumHeight(-1);
            setMaximumHeight(-1);
        }
        else
        {
            m_comboBox->setSizePolicy(QSizePolicy(QSizePolicy::Preferred, QSizePolicy::Fixed));

            qreal height = (layout()->itemAt(0)->preferredHeight() + top + bottom);

            setMinimumHeight(height);
            setMaximumHeight(height);
        }

        setMinimumWidth(m_comboBox->sizeHint().width() + 25 + left + right);
    }

    if (constraints & Plasma::SizeConstraint && formFactor() == Plasma::Horizontal)
    {
        qreal margin = ((boundingRect().height() - layout()->itemAt(0)->preferredHeight()) / 2);

        layout()->setContentsMargins(3, margin, 3, margin);
    }
}

void RunCommandApplet::focusWidget()
{
    if (scene())
    {
        QGraphicsView *parentView = NULL;
        QGraphicsView *possibleParentView = NULL;

        foreach (QGraphicsView *view, scene()->views())
        {
            if (view->sceneRect().intersects(sceneBoundingRect()) || view->sceneRect().contains(scenePos()))
            {
                if (view->isActiveWindow())
                {
                    parentView = view;

                    break;
                }
                else
                {
                    possibleParentView = view;
                }
            }
        }

        if (!parentView)
        {
            parentView = possibleParentView;
        }

        if (parentView)
        {
            KWindowSystem::forceActiveWindow(parentView->winId());
        }
    }

    raise();

    m_comboBox->setFocus();

    QTimer::singleShot(250, m_comboBox, SLOT(setFocus()));
}

void RunCommandApplet::clearHistory()
{
    KConfig runnersConfiguration("krunnerrc", KConfig::SimpleConfig);
    KConfigGroup configuration = KConfigGroup(&runnersConfiguration, "General");

    configuration.deleteEntry("PastQueries");
    configuration.sync();

    static_cast<KLineEdit*>(m_comboBox->lineEdit())->completionObject()->clear();

    m_comboBox->clearHistory();
}

void RunCommandApplet::resetColor()
{
    m_comboBox->setStyleSheet("QComboBox {color: auto;}");
}

void RunCommandApplet::queryChanged(QString query)
{
    KConfigGroup configuration = config();

    if (query.length() > 1 && configuration.readEntry("enableRunners", true))
    {
        m_manager->launchQuery(query);
    }
    else
    {
        resultsChanged(QList<Plasma::QueryMatch>());
    }
}

void RunCommandApplet::resultsChanged(const QList<Plasma::QueryMatch> &matches)
{
    for (int i = 0; i < m_resultsUi.resultsLayout->count(); ++i)
    {
        m_resultsUi.resultsLayout->removeItem(m_resultsDialog->layout()->itemAt(i));
    }

    QHashIterator<RunCommandItem*, QString> iterator(m_widgets);

    while (iterator.hasNext())
    {
        iterator.next();
        iterator.key()->deleteLater();
    }

    m_widgets.clear();

    if (!matches.count())
    {
        m_resultsDialog->hide();

        return;
    }

    for (int i = 0; i < matches.count(); ++i)
    {
        RunCommandItem *item = new RunCommandItem(matches.at(i).icon(), matches.at(i).text(), matches.at(i).subtext(), m_manager->actionsForMatch(matches.at(i)), m_resultsDialog);

        m_widgets[item] = matches.at(i).id();

        m_resultsUi.resultsLayout->addWidget(item);

        connect(item, SIGNAL(run()), this, SLOT(runQuery()));
    }

    m_resultsDialog->move(popupPosition(m_resultsDialog->size()));
    m_resultsDialog->show();

    QTimer::singleShot(500, m_resultsDialog, SLOT(adjustSize()));

    focusWidget();
}

void RunCommandApplet::getCommand()
{
    if (m_comboBox->lineEdit()->text().isEmpty())
    {
        return;
    }

    runCommand(m_comboBox->lineEdit()->text());
}

void RunCommandApplet::runCommand(QString command)
{
    if (m_lastRun.secsTo(QDateTime::currentDateTime()) < 1)
    {
        return;
    }

    m_lastRun = QDateTime::currentDateTime();

    KDisplayManager displayManager;
    KConfig runnersConfiguration("krunnerrc", KConfig::SimpleConfig);
    KConfigGroup configuration = KConfigGroup(&runnersConfiguration, "General");
    KConfig uriConfiguration("kuriikwsfilterrc", KConfig::NoGlobals);
    KConfigGroup uriGeneralGroup(&uriConfiguration, "General");
    QString delimiter = uriGeneralGroup.readPathEntry("KeywordDelimiter", QString(":"));
    QString path = QDir::cleanPath(KShell::tildeExpand(command));
    QFileInfo *location = new QFileInfo(path);
    QRegExp webAddress("^((ht|f)tps?://|www\\.).+");
    int index;
    bool commandStatus = true;
    bool addToHistory = true;

    if (command.length() > 3 && command.contains(delimiter))
    {
        index = command.indexOf(delimiter);

        if (index != (command.length() - 1))
        {
            KService::List offers = KServiceTypeTrader::self()->query("SearchProvider", QString("'%1' in Keys").arg(command.left(index)));

            if (!offers.isEmpty())
            {
                QString query = offers.at(0)->property("Query").toString();

                command = query.replace("\\{@}", command.right(command.length() - index - 1));
            }
        }
    }

    KUrl url(command);

    if (command.startsWith('=') || command.startsWith(QString("bin=")) || command.startsWith(QString("oct=")) || command.startsWith(QString("dec=")) || command.startsWith(QString("hex=")))
    {
        int base = 10;
        QString result;

        addToHistory = false;

        if (!command.startsWith('='))
        {
            if (command.startsWith(QString("bin=")))
            {
                base = 2;
            }
            else if (command.startsWith(QString("oct=")))
            {
                base = 8;
            }
            else if (command.startsWith(QString("hex=")))
            {
                base = 16;
            }

            command.remove(0, 3);
        }

        command.remove(0, 1).remove(' ');

        if (command.contains(KGlobal::locale()->decimalSymbol(), Qt::CaseInsensitive))
        {
            command.replace(KGlobal::locale()->decimalSymbol(), QChar('.'), Qt::CaseInsensitive);
        }

        if (command.contains("0x"))
        {
            int position = 0;
            QString hex;

            while ((position = command.indexOf("0x")) > -1)
            {
                hex.clear();

                for (int i = (position + 2); i < command.size(); ++i)
                {
                    if (((command[i] <= '9') && (command[i] >= '0')) || ((command[i] <= 'f') && (command[i] >= 'a')) || ((command[i] <= 'F') && (command[i] >= 'A')))
                    {
                        hex.append(command[i]);
                    }
                    else
                    {
                        break;
                    }
                }

                command.replace("0x" + hex, QString::number(hex.toInt(NULL, 16)));
            }
        }

        command.replace(QRegExp("([a-zA-Z]+)"), "Math.\\1");

        QScriptEngine engine;
        QScriptValue value = engine.evaluate(command);

        if (!value.isError())
        {
            result = ((base == 16)?"0x":"") + QString::number(value.toString().toInt(), base).toUpper();

            m_comboBox->lineEdit()->setText(result);
        }
        else
        {
            commandStatus = false;
        }
    }
    else if (location->exists())
    {
        command = path;

        new KRun(path, NULL);
    }
    else if ((url.protocol() != "file" && KProtocolInfo::isKnownProtocol(url.protocol())) || webAddress.exactMatch(command))
    {
        if (url.protocol().isEmpty())
        {
            index = command.indexOf('/');

            url.clear();
            url.setHost(command.left(index));

            if (index != -1)
            {
                url.setPath(command.mid(index));
            }

            url.setProtocol("http");

            command.prepend("http://");
        }

        KToolInvocation::invokeBrowser(url.url());
    }
    else if (command.startsWith(QString("mailto:")))
    {
        KToolInvocation::invokeMailer(command.remove(0, 7), NULL);
    }
    else if (command == "logout")
    {
        KWorkSpace::requestShutDown(KWorkSpace::ShutdownConfirmDefault, KWorkSpace::ShutdownTypeNone);
    }
    else if (command == "shutdown")
    {
        KWorkSpace::requestShutDown(KWorkSpace::ShutdownConfirmDefault, KWorkSpace::ShutdownTypeHalt);
    }
    else if (command == "restart" || command == "reboot")
    {
        KWorkSpace::requestShutDown(KWorkSpace::ShutdownConfirmDefault, KWorkSpace::ShutdownTypeReboot);
    }
    else if (command == "switch" && KAuthorized::authorizeKAction("start_new_session") && displayManager.isSwitchable() && displayManager.numReserve() > 0)
    {
        displayManager.startReserve();
    }
    else if (command == "lock")
    {
        KRun::runCommand("dbus-send --print-reply --dest=org.freedesktop.ScreenSaver /ScreenSaver org.freedesktop.ScreenSaver.Lock", NULL);
    }
    else
    {
        if (command.startsWith('$'))
        {
            KToolInvocation::invokeTerminal(command.remove(0, 1));
        }
        else
        {
            QString binaryName = KRun::binaryName(command, true);

            if (!QFile(binaryName).exists() && KStandardDirs::findExe(binaryName).isEmpty())
            {
                commandStatus = false;
            }
            else
            {
                commandStatus = KRun::runCommand(command, NULL);
            }
        }
    }

    if (commandStatus)
    {
        if (addToHistory)
        {
            m_comboBox->addToHistory(command);
            m_comboBox->clearEditText();

            configuration.writeEntry("PastQueries", m_comboBox->historyItems());
            configuration.sync();
        }
    }
    else
    {
        m_comboBox->setStyleSheet("QComboBox {color: red;}");

        QTimer::singleShot(1500, this, SLOT(resetColor()));
    }

    delete location;
}

void RunCommandApplet::runQuery()
{
    RunCommandItem *item = qobject_cast<RunCommandItem*>(sender());

    if (m_widgets.contains(item))
    {
        m_manager->run(m_widgets[item]);
    }

    m_comboBox->lineEdit()->clear();
}

void RunCommandApplet::toggleEnableRunners(bool enable)
{
    config().writeEntry("enableRunners", enable);

    m_enableRunnersAction->setChecked(enable);

    if (enable && !m_comboBox->currentText().isEmpty())
    {
        queryChanged(m_comboBox->currentText());
    }
}

void RunCommandApplet::updateTheme()
{
    QPalette palette = m_resultsDialog->palette();
    palette.setColor(QPalette::WindowText, Plasma::Theme::defaultTheme()->color(Plasma::Theme::TextColor));
    palette.setColor(QPalette::ButtonText, Plasma::Theme::defaultTheme()->color(Plasma::Theme::ButtonTextColor));
    palette.setColor(QPalette::Background, Plasma::Theme::defaultTheme()->color(Plasma::Theme::BackgroundColor));
    palette.setColor(QPalette::Button, palette.color(QPalette::Background).lighter(250));

    m_resultsDialog->setPalette(palette);
}

QList<QAction*> RunCommandApplet::contextualActions()
{
    QList<QAction*> actions;
    actions.append(m_enableRunnersAction);

    return actions;
}

bool RunCommandApplet::eventFilter(QObject *object, QEvent *event)
{
    Q_UNUSED(object)

    if (event->type() == QEvent::MouseButtonPress && static_cast<QMouseEvent*>(event)->button() == Qt::LeftButton)
    {
        focusWidget();
    }

    return false;
}

#include "RunCommandApplet.moc"
