/******************************************************************************
*                                PlayWolf                                     *
*******************************************************************************
*                                                                             *
*                   Copyright (C) 2008-2009 Giulio Camuffo		      *
*                                                                             *
*   This program is free software; you can redistribute it and/or modify      *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 2 of the License, or         *
*   (at your option) any later version.                                       *
*                                                                             *
*   This program is distributed in the hope that it will be useful,           *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License along   *
*   with this program; if not, write to the Free Software Foundation, Inc.,   *
*   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA                *
*                                                                             *
*                                                                             *
*                                                                             *
*   For further information contact me at giuliocamuffo@gmail.com	      *
******************************************************************************/

#include "flowlayout.h"

FlowLayout::FlowLayout(QGraphicsLayoutItem *parent)
    : QGraphicsLayout(parent)
{
    primarySpacing = 0;
    secondarySpacing = 0;
}

FlowLayout::~FlowLayout() {
    LayoutItem *item;
    while ((item = takeAt(0)))
	delete item;
}

void FlowLayout::addItem(QGraphicsLayoutItem *item) {
    LayoutItem *myItem = new LayoutItem;
    myItem->item = item;
    itemList.append(myItem);

    doLayout(geometry());
}

int FlowLayout::count() const {
    return itemList.size();
}

LayoutItem *FlowLayout::takeAt(int index) {
    if ((index >= 0) and (index < itemList.size()))
	return itemList.takeAt(index);
    else
	return 0;
}

QGraphicsLayoutItem *FlowLayout::itemAt(int index) const {
    if ((index >= 0 ) and (index < itemList.size()))
	return itemList.at(index)->item;
    else
	return 0;
}

int FlowLayout::indexAt(int row, int column) const {
    for (int i=0; i<itemList.count(); ++i) {
	if ((itemList.at(i)->row == row) and (itemList.at(i)->column == column))
	    return i;
    }  
    return -1;
}

void FlowLayout::removeAt(int index) {
    LayoutItem *item = takeAt(index);
    delete item;
    
    doLayout(geometry());
}

void FlowLayout::removeItem(QGraphicsLayoutItem *item) {
    for (int i=0;i<itemList.count();i++) {
	if (itemList.at(i)->item == item) {
	    removeAt(i);
	    break;
	}
    }
}

void FlowLayout::setPrimarySpacing(int spacing) {
    primarySpacing = spacing;
    doLayout(contentsRect());
}

void FlowLayout::setSecondarySpacing(int spacing) {
    secondarySpacing = spacing;
    doLayout(contentsRect());
}

void FlowLayout::setGeometry(const QRectF &rect) {
    QGraphicsLayout::setGeometry(rect);
    doLayout(rect);
}

QSizeF FlowLayout::sizeHint(Qt::SizeHint which, const QSizeF &constraint) const {

    QSizeF s(0,0);
    int n = itemList.count();
    for (int i=0; i<n;i++) {
	LayoutItem *myItem = itemList.at(i);
	s = s.expandedTo(myItem->item->effectiveSizeHint(which, constraint));
    }

    qreal left, top, right, bottom;
    getContentsMargins(&left, &top, &right, &bottom);
    return s + QSizeF(left + right, top + bottom);

}

int FlowLayout::doLayout(const QRectF &rect) const {
    qreal sp = primarySpacing/2.0;

    if (orientation == Horizontal) {
	int x = rect.x();
	int y = rect.y();
	int lineHeight = 0;
	int row = 0;
	int column = -1;

	LayoutItem *myItem;
	foreach (myItem, itemList) {
	    QSizeF size = myItem->item->minimumSize();
	    int nextX = x + size.width() + sp;
	    if (nextX - sp > rect.right() && lineHeight > 0) {
		x = rect.x();
		y = y + lineHeight + sp;
		nextX = x + size.width() + sp;
		lineHeight = 0;
		row++;
		column = -1;
	    }

	    column++;

	    myItem->row = row;
	    myItem->column = column;

	    x = nextX;
	if (lineHeight < size.height())
	    lineHeight = size.height();
	}

	for (int i=1; i<rowCount();++i) {
	    int prevColumns = columnsPerRow(i-1);
	    int thisColumns = columnsPerRow(i);
	    int gap = prevColumns - thisColumns;

	    if ((gap/2) > 0) {
		for (int j=thisColumns-1;j>-1;--j) {
		    LayoutItem *item = itemList[indexAt(i,j)];
		    item->column = j+1;
		}

		for (int j=(prevColumns-(gap/2));j<prevColumns;++j) {
		    LayoutItem *item = itemList[indexAt(i-1,j)];
		    item->row = i;
		    item->column = j-gap;
		}
	    }
	}

	int rows = rowCount();
	int columns = columnCount();
	qreal width = (rect.width() / columns);
	qreal height = (rect.height() / rows);

	foreach (myItem, itemList) {
	    int row = myItem->row;
	    int column = myItem->column;

	    qreal x = width * column + (columns - columnsPerRow(row))*width/2.0 + sp;
	    qreal y = height * row + secondarySpacing/2.0;

	    myItem->item->setGeometry(QRectF(QPointF(x, y), QSizeF(width - primarySpacing,height - secondarySpacing)));
	}

	return y + lineHeight - rect.y();
    }
    else if (orientation == Vertical) {

	int x = rect.x();
	int y = rect.y();
	int lineWidth = 0;
	int row = -1;
	int column = 0;

	LayoutItem *myItem;
	foreach (myItem, itemList) {
	    QSizeF size = myItem->item->minimumSize();
	    int nextY = y + size.height() + sp;
	    if (nextY - sp > rect.bottom() && lineWidth > 0) {
		y = rect.y();
		x = x + lineWidth + sp;
		nextY = y + size.height() + sp;
		lineWidth = 0;
		row = -1;
		column++;
	    }

	    row++;

	    myItem->row = row;
	    myItem->column = column;

	    y = nextY;
	if (lineWidth < size.width())
	    lineWidth = size.width();
	}

	for (int i=1; i<columnCount();++i) {
	    int prevRows = rowsPerColumn(i-1);
	    int thisRows = rowsPerColumn(i);
	    int gap = prevRows - thisRows;

	    if ((gap/2) > 0) {
		for (int j=thisRows-1;j>-1;--j) {
		    LayoutItem *item = itemList[indexAt(j,i)];
		    item->row = j+1;
		}

		for (int j=(prevRows-(gap/2));j<prevRows;++j) {
		    LayoutItem *item = itemList[indexAt(j,i-1)];
		    item->row = j-gap;
		    item->column = i;
		}
	    }
	}

	int rows = rowCount();
	int columns = columnCount();
	qreal width = (rect.width() / columns);
	qreal height = (rect.height() / rows);

	foreach (myItem, itemList) {
	    int row = myItem->row;
	    int column = myItem->column;

	    qreal x = width * (column) + secondarySpacing/2.0;
	    qreal y = height * (row) + (rows - rowsPerColumn(column))*height/2 + sp;

	    myItem->item->setGeometry(QRectF(QPointF(x, y), QSizeF(width - secondarySpacing,height - primarySpacing)));
	}

	return x + lineWidth - rect.x();
    }

    return -1;
}

int FlowLayout::rowCount() const {

    int i = 0;
    LayoutItem *myItem;
    foreach (myItem, itemList) {    
	int row = myItem->row+1;  
	if (row > i)
	    i = row;
    }
    return i;
}

int FlowLayout::columnCount() const {

    int i = 0;
    LayoutItem *myItem;
    foreach (myItem, itemList) {    
    int column = myItem->column+1;
	if (column > i)
	    i = column;
    }
    return i;
}

int FlowLayout::columnsPerRow(int row) const {

    int i = 0;
    LayoutItem *myItem;
    foreach (myItem, itemList) {    
    int column = myItem->column+1;
	if (myItem->row == row)
	    if (column > i)
		i = column;	
    }
    return i;
}

int FlowLayout::rowsPerColumn(int column) const {

    int i = 1;
    LayoutItem *myItem;
    foreach (myItem, itemList) {    
    int row = myItem->row+1;
	if (myItem->column == column)
	    if (row > i)
		i = row;
    }
    return i;
}

void FlowLayout::setOrientation(int orient) {
    orientation = orient;
}