/***************************************************************************
 *   Copyright (C) 2009 by Kai Dombrowe <just89@gmx.de>                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .        *
 ***************************************************************************/

/// Qt
#include <QGraphicsLinearLayout>
#include <QTimer>

/// KDE
#include <KIcon>
#include <KConfigGroup>
#include <KProcess>
#include <KStandardDirs>
#include <KMessageBox>
#include <Plasma/Theme>
#include <KDesktopFile>

/// own
#include "pgame.h"
#include "gamemodel.h"
#include "gamelist.h"
#include "gameitem.h"


PGame::PGame( QObject *parent, QVariantList args )
	: Plasma::PopupApplet( parent, args ),
	_locateTmp( true ),
	_locateBinary( true ),
	_frame( true ),
	selectedGame( 0 )
{

	setHasConfigurationInterface( true );
	setAspectRatioMode( Plasma::IgnoreAspectRatio );
	Plasma::Applet::resize( 200, 125 );
	setAcceptDrops( true );
	setPopupIcon( KIcon( "applications-games" ) );

}


PGame::~PGame()
{

	delete model;

}



void PGame::init()
{

	loadAppearanceConfig();
	setBackgroundHints( _background );

	list = new GameList( this );
	list->setColor( _color );
	list->setFont( _font );
	list->setPaintFrame( _frame );

	connect( list, SIGNAL( addClicked() ), this, SLOT( setupAddDialog() ) );
	connect( list, SIGNAL( removeClicked() ), this, SLOT( removeSelectedGames() ) );
	connect( list, SIGNAL( editClicked() ), this, SLOT( editSelectedGame() ) );
	connect( list, SIGNAL( doubleClick(const QModelIndex&) ), this, SLOT( startGame(const QModelIndex&) ) );

	model = new GameModel;
	list->setModel( model );

	loadConfig();

}


void PGame::startGame( const QModelIndex& idx )
{

	int row = idx.row();
	kDebug() << "start >" << model->gameList().at( row )->data( 0 ).toString();

	if( !model->gameList().at( row )->startx() )
	{
		KProcess *p = new KProcess;
		p->setShellCommand( model->gameList().at( row )->cmd() );
		p->start();
		return;
	}

	QString home = QDir::homePath();
	QString cmd  = model->gameList().at( row )->cmd();

	//xbin + xinitrc
	KUrl xbin; 
	if( _locateBinary )
		xbin = KStandardDirs::findExe( "X" );
	else
		xbin = _xBinary.prettyUrl();

	if( !QFile( xbin.toLocalFile() ).exists() )
	{
		KMessageBox::sorry( 0, QString("cannot find X binary: \"%1\" check your PGame settings.").arg( xbin.toLocalFile() ),
			QString(), 0 );
		return;
	}


	QFile *rc = new QFile( (home + "/.xinitrc_pgame") );
	if( rc->exists() )
		rc->remove();

	if( !rc->open( QIODevice::WriteOnly | QIODevice::Text ) )
	{
		KMessageBox::sorry( 0, QString("cannot open \"%1\".").arg( (home + "/.xinitrc_pgame") ),
			QString(), 0 );
		return;
	}

	QTextStream out( rc );
	out << QString( "exec %1" ).arg( cmd ) << "\n";

	//xnumber
	KUrl tmp;
	if( _locateTmp )
	{
		tmp = getenv("TMPDIR");
		if( !QDir( tmp.toLocalFile() ).exists() || tmp.isEmpty() )
			tmp = "/tmp";
	}
	else
		tmp = _tmpDir.prettyUrl();

	if( !QDir( tmp.toLocalFile() ).exists() )
	{
		KMessageBox::sorry( 0, QString("Cannot find tmp Directory \"%1\", check your PGame settings.").arg( tmp.toLocalFile() ),
			QString(), 0 );
		return;
	}


	int xnumber = 0;
	while( true )
	{
		if( QFile( (tmp.toLocalFile() + QString( "/.X%1-lock" ).arg( xnumber )) ).exists() )
			xnumber++;
		else
			break;
	}
	//start x
	KProcess *p = new KProcess;
	QString start = QString("xinit %1/.xinitrc_pgame -- %3 :%2 -br %4" ).arg( home ).arg( xnumber ).arg( xbin.toLocalFile() ).arg( _extraCmd );
	p->setOutputChannelMode( KProcess::MergedChannels );
	p->setShellCommand( start );
	kDebug() << "cmd >" << start;
	p->start();

}


void PGame::loadConfig()
{

	KConfigGroup cfg = config();
	int games = cfg.readEntry( "Games", 0 );

	if( games == 0 )
	{
		QStringList defaultGames;
		defaultGames << "KBreakout" << "KBounce" << "KBlackbox"
		<< "Kapman" << "KSirk" << "KSudoku" << "LSkat" << "Kolf"
		<< "Bomber" << "KGoldrunner" << "KSpaceDuel";
		foreach( const QString &defaultGame, defaultGames )
		{
			KService::Ptr service = KService::serviceByStorageId( defaultGame );
			if( service && service->isValid() )
			{
				QString path = service->entryPath();
				if( !path.isEmpty() && QDir::isAbsolutePath( path ) )
				{
					QString name, icon, cmd;
					bool game;
					readFromDesktopFile( path, name, icon, cmd, game );
					if( !game || name.isEmpty() || cmd.isEmpty() || icon.isEmpty() )
						return;
					model->addGame( name, icon, cmd ,false );
				}
			}
		}
		return;
	}

	for( int i = 0; i < games; i++ )
	{
		QString name = cfg.readEntry( QString( "Name %1" ).arg( i ), QString() );
		QString icon = cfg.readEntry( QString( "Icon %1" ).arg( i ), QString() );
		QString cmd = cfg.readEntry( QString( "Cmd %1" ).arg( i ), QString() );
		bool startx = cfg.readEntry( QString( "Startx %1" ).arg( i ), false );

		model->addGame( name, icon, cmd, startx );
	}

	_extraCmd = cfg.readEntry( "Extra Cmd" );
	_tmpDir = cfg.readEntry( "Tmp" );
	_xBinary = cfg.readEntry( "X" );
	_locateTmp = cfg.readEntry( "Auto Locate Tmp", true );
	_locateBinary = cfg.readEntry( "Auto Locate X", true );

}


void PGame::loadAppearanceConfig()
{

	KConfigGroup cfg = config();
	_color = cfg.readEntry( "Color", Plasma::Theme::defaultTheme()->color( Plasma::Theme::TextColor ) );
	_font = cfg.readEntry( "Font", Plasma::Theme::defaultTheme()->font( Plasma::Theme::DesktopFont ) );
	int bg = cfg.readEntry( "Background", 1 );
	switch( bg )
	{
		case 1: _background = Plasma::Applet::StandardBackground; break;
		case 2: _background = Plasma::Applet::TranslucentBackground; break;
	}
	_frame = cfg.readEntry( "Show Frame", true );

}


void PGame::saveGameConfig()
{

	KConfigGroup cfg = config();
	int games = 0;
	foreach( GameItem *game, model->gameList() )
	{
		QString name = game->data( 0 ).toString();
		QString icon = game->icon();
		QString cmd = game->cmd();
		bool startx = game->startx();

		cfg.writeEntry( QString( "Name %1" ).arg( games ), name );
		cfg.writeEntry( QString( "Icon %1" ).arg( games ), icon );
		cfg.writeEntry( QString( "Cmd %1" ).arg( games ), cmd );
		cfg.writeEntry( QString( "Startx %1" ).arg( games ), startx );

		games++;
	}
	cfg.writeEntry( "Games", games );

}


void PGame::saveGeneralConfig()
{

	_extraCmd = ui_settings.cmdEdit->text();
	_tmpDir = ui_settings.tmpRequester->url();
	_xBinary = ui_settings.binaryRequester->url();
	_locateTmp = ui_settings.tmpCheck->isChecked();
	_locateBinary = ui_settings.binaryCheck->isChecked();

	KConfigGroup cfg = config();

	cfg.writeEntry( "Extra Cmd", _extraCmd );
	cfg.writeEntry( "Tmp", _tmpDir );
	cfg.writeEntry( "X", _xBinary );
	cfg.writeEntry( "Auto Locate Tmp", _locateTmp);
	cfg.writeEntry( "Auto Locate X", _locateBinary );

}


void PGame::saveAppearanceConfig()
{

	KConfigGroup cfg = config();
	_color = ui_appearance.colorButton->color();
	_font = ui_appearance.fontRequester->font();
	if( ui_appearance.normalRadio->isChecked() )
	{
		_background = Plasma::Applet::StandardBackground;
		cfg.writeEntry( "Background", 1 );
	}
	else
	{
		_background = Plasma::Applet::TranslucentBackground;
		cfg.writeEntry( "Background", 2 );
	}
	_frame = ui_appearance.frameCheck->isChecked();

	cfg.writeEntry( "Color", _color );
	cfg.writeEntry( "Font", _font );
	cfg.writeEntry( "Show Frame", _frame );

	setBackgroundHints( _background );
	list->setColor( _color );
	list->setFont( _font );
	list->setPaintFrame( _frame );

}


void PGame::createConfigurationInterface( KConfigDialog *p_parent )
{

	QWidget *page1 = new QWidget;
	QWidget *page2 = new QWidget;
	ui_settings.setupUi( page1 );
	ui_appearance.setupUi( page2 );

	connect( p_parent, SIGNAL( okClicked() ), SLOT( saveGameConfig() ) );
	connect( p_parent, SIGNAL( okClicked() ), SLOT( saveGeneralConfig() ) );
	connect( p_parent, SIGNAL( okClicked() ), SLOT( saveAppearanceConfig() ) );
	connect( ui_settings.binaryRequester, SIGNAL( textChanged(QString) ), this, SLOT( updateCmd() ) );
	connect( ui_settings.cmdEdit, SIGNAL( textChanged(QString) ), this, SLOT( updateCmd() ) );

	ui_settings.tmpRequester->setMode( KFile::Directory );
	ui_settings.cmdEdit->setText( _extraCmd );
	if( _tmpDir.isEmpty() )
		ui_settings.tmpRequester->setUrl( KUrl( "/tmp" ) );
	else
		ui_settings.tmpRequester->setUrl( _tmpDir );
	if( _xBinary.isEmpty() )
		ui_settings.binaryRequester->setUrl( KUrl( "/usr/bin/X" ) );
	else
		ui_settings.binaryRequester->setUrl( _xBinary );
	ui_settings.tmpCheck->setChecked( _locateTmp );
	ui_settings.binaryCheck->setChecked( _locateBinary );

	ui_appearance.colorButton->setColor( _color);
	ui_appearance.fontRequester->setFont( _font );
	ui_appearance.frameCheck->setChecked( _frame );

	switch( _background )
	{
		case 1: ui_appearance.normalRadio->setChecked( true ); break;
		case 2: ui_appearance.transRadio->setChecked( true ); break;
		default: ui_appearance.normalRadio->setChecked( true ); break;
	}

	p_parent->addPage( page1, i18n( "XServer Settings" ), "xorg" );
	p_parent->addPage( page2, i18n( "Appearance" ), "color" );

}


void PGame::createNewGame()
{

	QString name = ui_game.nameEdit->text();
	QString icon = ui_game.iconButton->icon();
	QString cmd = ui_game.cmdEdit->text();
	bool startx = ui_game.xCheck->isChecked();

	model->addGame( name, icon, cmd, startx );
	saveGameConfig();

}


void PGame::setupAddDialog()
{

	QDialog *dia = new QDialog;
	ui_game.setupUi( dia );
	connect( ui_game.addButton, SIGNAL( clicked() ), this, SLOT( createNewGame() ) );
	connect( ui_game.addButton, SIGNAL( clicked() ), dia, SLOT( close() ) );
	dia->show();

}


void PGame::removeSelectedGames()
{

	foreach( QModelIndex idx, list->selectedRows() )
		model->removeItem( idx.row(), idx.column(), idx.parent() );

	saveGameConfig();

}


void PGame::editSelectedGame()
{

	GameItem *game( 0 );
	foreach( QModelIndex idx, list->selectedRows() )
		game = model->item( idx.row(), idx.column(), idx.parent() );

	if( !game )
		return;

	selectedGame = game;
	QDialog *dia = new QDialog;
	ui_game.setupUi( dia );
	ui_game.addButton->setText( i18n( "Save" ) );

	ui_game.nameEdit->setText( game->data( 0 ).toString() );
	ui_game.iconButton->setIcon( game->icon() );
	ui_game.cmdEdit->setUrl( game->cmd() );
	ui_game.xCheck->setChecked( game->startx() );

	connect( ui_game.addButton, SIGNAL( clicked() ), this, SLOT( saveSelectedGame() ) );
	connect( ui_game.addButton, SIGNAL( clicked() ), dia, SLOT( close() ) );
	connect( ui_game.cancelButton, SIGNAL( clicked() ), this, SLOT( dialogCancel() ) );

	dia->show();

}


void PGame::dialogCancel()
{

	selectedGame = 0;

}


void PGame::saveSelectedGame()
{

	if( !selectedGame )
		return;

	selectedGame->setName( ui_game.nameEdit->text() );
	selectedGame->setIcon( ui_game.iconButton->icon() );
	selectedGame->setCmd( ui_game.cmdEdit->text() );
	selectedGame->setStartx( ui_game.xCheck->isChecked() );
	saveGameConfig();
	selectedGame = 0;

}


void PGame::updateCmd()
{

	QString ui_bin = ui_settings.binaryRequester->url().path();
	QString ui_ecmd = ui_settings.cmdEdit->text();
	QString home = QDir::homePath();

	ui_settings.textBrowser->setText( QString("xinit %1/.xinitrc_pgame -- %2 :$display -br %3" )
		.arg( home ).arg( "<font color='red'>" + ui_bin + "</font>" ).arg( "<font color='red'>" + ui_ecmd ) );

}


QGraphicsWidget* PGame::graphicsWidget() 
{

	return list;

}


void PGame::dragEnterEvent( QGraphicsSceneDragDropEvent *event )
{

	Q_UNUSED( event );

}


void PGame::dropEvent( QGraphicsSceneDragDropEvent *event )
{

	QList<KUrl> urls = KUrl::List::fromMimeData( event->mimeData() );
	/// intern
	if( !event->mimeData()->data( "row" ).isEmpty() )
	{

		int row = event->mimeData()->data( "row" ).toInt();
		if( row == -1 )
			return;

		GameItem *game = model->item( row, 0, QModelIndex() );
		if( !game )
			return;

		QString name = game->data( 0 ).toString();
		QString icon = game->data( 1 ).toString();
		QString cmd = game->cmd();
		bool startx = game->startx();


		GameItem *ngame = new GameItem( name, icon, model->parentGame() );
		ngame->setCmd( cmd );
		ngame->setStartx( startx );
		model->insertItem( ngame, list->row( this, event->pos() ) );
		model->removeGame( game );
		saveGameConfig();
		event->accept();
		return;
	}/// extern
	else if( urls.count() < 1 )
		return;

	KUrl file = urls.first();

	if( !file.isLocalFile() )
		return;

	QString name, icon, cmd;
	bool isGame;
	readFromDesktopFile( file.path(), name, icon, cmd, isGame );
	kDebug() << isGame << name << cmd << icon;
	if( !isGame || name.isEmpty() || cmd.isEmpty() || icon.isEmpty() )
		return;

	GameItem *ngame = new GameItem( name, icon, 0 );
	ngame->setCmd( cmd );
	ngame->setStartx( false );
	model->insertItem( ngame, list->row( this, event->pos() ) );
	saveGameConfig();
	event->accept();

}


void PGame::readFromDesktopFile( const QString& path, QString &name, QString &icon, QString &cmd, bool &game )
{

	KConfig cfg( path );
	KConfigGroup desktop = KConfigGroup( &cfg, "Desktop Entry" );
	name = desktop.readEntry( "Name" );
	cmd = desktop.readEntry( "Exec" );
	icon = desktop.readEntry( "Icon" );
	game = desktop.readEntry( "Categories" ).split( ";" ).contains( "Game" );

}


#include "pgame.moc"

K_EXPORT_PLASMA_APPLET( pgame, PGame );




