/***************************************************************************
 *   Copyright (C) 2009 by Mathias Rabe <masterunderlined [at] web.de>     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .        *
 ***************************************************************************/
#include "drop2ftp.h"

#include <QDrag>
#include <QGraphicsSceneDragDropEvent>
#include <QTimer>
#include <QFontMetrics>

#include <KUrl>
#include <KConfigDialog>
#include <KFileDialog>
#include <kio/copyjob.h>
#include <KIcon>
#include <KGlobalSettings>
#include <kicondialog.h>
 
#include <plasma/tooltipmanager.h>

//#include <solid/device.h>
//#include <solid/networking.h>

#define MINSIZE 75
 
using namespace Plasma;

Drop2FTP::Drop2FTP(QObject *parent, const QVariantList &args)
    : Plasma::Applet(parent, args), 
       m_serverAddress(0),
       m_serverTitle(0),
       m_hideProgressInfo(0),
       m_numActivJobs(0),
       m_newIcon(0),
       m_iconName(0),
       m_icon(0),
       m_label(0)
{
    setAcceptDrops(true);
    setHasConfigurationInterface(true);
    setBackgroundHints(NoBackground);
    setAspectRatioMode(Plasma::ConstrainedSquare);
    resize(100, 100);
} 
 
Drop2FTP::~Drop2FTP()
{
    delete m_icon;
    delete m_label;
}
 
void Drop2FTP::init()
{
    //TODO rightklick -> open destination host with konqueror
    layout = new QGraphicsLinearLayout(this);
    layout->setContentsMargins(0, 0, 0, 0);
    layout->setOrientation(Qt::Vertical);
    layout->setSpacing(0);

    KConfigGroup cg = config();
    m_serverAddress = cg.readEntry("Server", "");
    m_serverTitle = cg.readEntry("Title", "");
    m_hideProgressInfo = cg.readEntry("HideProgressInfo", false);
    m_iconName = cg.readEntry("IconName", "applications-internet");

    m_label = new Plasma::Label(this);
    m_label->setAlignment(Qt::AlignHCenter);

    m_icon = new Plasma::IconWidget(KIcon(m_iconName), QString(), this);
    registerAsDragHandle(m_icon);
    layout->addItem(m_icon);

    Plasma::ToolTipManager::self()->registerWidget(this);
    Plasma::ToolTipContent toolTipData(i18n("Drop2FTP"), i18n("Drop your files here to send them to your server"), //TODO: replace "to your server" with the specified name in the settings
                                       m_icon->icon().pixmap(IconSize(KIconLoader::Desktop))); //TODO the icon doesn't refresh if it changes
    Plasma::ToolTipManager::self()->setContent(this, toolTipData);

    connect(m_icon, SIGNAL(pressed(bool)), this, SLOT(openFileDialog(bool)));
}

void Drop2FTP::paintInterface(QPainter *p,
        const QStyleOptionGraphicsItem *option, const QRect &contentsRect)
{
    Q_UNUSED( p );
    Q_UNUSED( option );

    if (!m_serverTitle.isEmpty() && layout->count() == 1 && qRound(contentsRect.width()) >= MINSIZE) { //add new label
        layout->addItem(m_label);
        layout->setStretchFactor(m_label, 0);
    }
    if (!m_serverTitle.isEmpty() && layout->count() == 2 && qRound(contentsRect.width()) >= MINSIZE) { //rename existing label
        QFontMetrics fontMetric = KGlobalSettings::smallestReadableFont();
        QString elidedServerTitle = fontMetric.elidedText(m_serverTitle, Qt::ElideRight, qRound(contentsRect.width()) - 13);
        m_label->setText(elidedServerTitle);
    } else { //remove label if there is no m_serverTitle or if the icon is to small
        m_label->setText(""); //otherwise fragments are the result
        layout->removeItem(m_label);
    }
}

void Drop2FTP::openFileDialog(bool down)
{
    if (down) {
        return;
    }
    if (m_serverAddress.isEmpty()) {
        emit showConfigurationInterface();
        return;
    }
    KUrl::List fileNameListFromDialog = KFileDialog::getOpenFileNames(); //http://api.kde.org/4.x-api/kdelibs-apidocs/kio/html/hierarchy.html
    if (!fileNameListFromDialog.isEmpty()) {
        sendToServer(fileNameListFromDialog);
    }
}

void Drop2FTP::createConfigurationInterface(KConfigDialog *parent)
{
    //Destination address-line
    QWidget *widgetDest = new QWidget;
    uiDest.setupUi(widgetDest);
    uiDest.serverAddress->setToolTip(i18n("Specify here the destination host. Use an entire address like ftp://192.168.0.1:21/folder"));
    uiDest.serverAddress->insert(m_serverAddress);

    //Destination bookmark-button
    bookmarkManager = KBookmarkManager::userBookmarksManager();
    bookmarkManager->setEditorOptions("Drop2FTP", true);
    connect(uiDest.editBookmarkButton, SIGNAL(clicked()), bookmarkManager, SLOT(slotEditBookmarks()));

    //Destination bookmark-treeview
    model = new QStandardItemModel;
    parentItem = model->invisibleRootItem();
    uiDest.bookmarkTree->setModel(model);
    connect(uiDest.bookmarkTree, SIGNAL(doubleClicked(const QModelIndex &)), this, SLOT(copyBookmarkAddress(const QModelIndex &)));

    traverser = new BookmarkTraverser;
    connect(traverser, SIGNAL(newBookmarkGroup(QString, QString)), this, SLOT(addBookmarkGroup(QString, QString)));
    connect(traverser, SIGNAL(newBookmark(QString, QString, KUrl)), this, SLOT(addBookmark(QString, QString, KUrl)));
    connect(traverser, SIGNAL(leaveBookmarkGroup()), this, SLOT(leaveBookmarkGroup()));

    connect(bookmarkManager, SIGNAL(changed(const QString &, const QString &)), this, SLOT(traverseBookmarks(const QString &, const QString &)));
    traverseBookmarks();

    //Destination warningwidget
    if (m_serverAddress.isEmpty()) {
        //TODO: add WarnIcon
        uiDest.informationLabel->setText(i18n("Please specify your destination host under \"Address\"."));
    }

    //Appearance
    QWidget *widgetAppear = new QWidget;
    uiAppear.setupUi(widgetAppear);
    uiAppear.iconButton->setIcon(KIcon(m_iconName));
    connect(uiAppear.iconButton, SIGNAL(clicked()), this, SLOT(openIconDialog()));
    uiAppear.serverTitle->setToolTip(i18n("Here you can specify which text should be displayed under the Icon"));
    uiAppear.serverTitle->insert(m_serverTitle);
    uiAppear.hideProgressInfo->setToolTip(i18n("Activate this, if you don't want to see the progress info."));
    uiAppear.hideProgressInfo->setChecked(m_hideProgressInfo);

    connect(parent, SIGNAL(okClicked()), this, SLOT(configAccepted()));
    parent->addPage(widgetDest, i18n("Destination"), Applet::icon());
    parent->addPage(widgetAppear, i18n("Appearance"), QString("video-display"));
}

void Drop2FTP::configAccepted()
{
    KConfigGroup cg = config();
    m_serverAddress = uiDest.serverAddress->text();
    m_serverTitle = uiAppear.serverTitle->text();
    m_hideProgressInfo = uiAppear.hideProgressInfo->checkState();
    if (!m_newIcon.isEmpty()) {
        m_iconName = m_newIcon;
        m_icon->setIcon(KIcon(m_iconName));
    }

    cg.writeEntry("Server", m_serverAddress);
    cg.writeEntry("Title", m_serverTitle);
    cg.writeEntry("HideProgressInfo", m_hideProgressInfo);
    cg.writeEntry("IconName", m_iconName);

    emit configNeedsSaving();
    emit update();
}

void Drop2FTP::traverseBookmarks(const QString &groupAddress, const QString &caller)
{
    Q_UNUSED( groupAddress );
    Q_UNUSED( caller );

    traverseBookmarks();
}

void Drop2FTP::traverseBookmarks()
{
    model->clear();
    QStringList headerLabels;
    headerLabels << i18n("Bookmark") << i18n("URL");
    model->setHorizontalHeaderLabels(headerLabels);
    traverser->traverse( bookmarkManager->root() );
    uiDest.bookmarkTree->expandAll();
}

void Drop2FTP::addBookmarkGroup(QString icon, QString text)
{
    QStandardItem *item = new QStandardItem(KIcon(icon), text);
    item->setEditable(false);
    QStandardItem *dummy = new QStandardItem(); //now I can set editable to false
    dummy->setEditable(false);

    parentItem->appendRow(item);
    parentItem->setChild(item->row(), 1, dummy);
    parentItem = item;
}

void Drop2FTP::addBookmark(QString icon, QString text, KUrl url)
{
    QStandardItem *item = new QStandardItem(KIcon(icon), text);
    item->setEditable(false);
    QStandardItem *urlItem = new QStandardItem(url.url());
    urlItem->setEditable(false);

    parentItem->appendRow(item);
    parentItem->setChild(item->row(), 1, urlItem);
}

void Drop2FTP::leaveBookmarkGroup()
{
    if (parentItem->parent() == 0) {
        parentItem = model->invisibleRootItem();
    } else {
        parentItem = parentItem->parent();
    }
}

void Drop2FTP::copyBookmarkAddress(const QModelIndex &index)
{
    QStandardItem *currentItem = model->itemFromIndex(index);

    //we only want to copy the stuff in the second column
    if(currentItem->column() != 1) {
        int row = currentItem->row();
        QModelIndex childIndex = index.sibling(row, 1);
        currentItem = model->itemFromIndex(childIndex);
    }

    if (currentItem->text().isEmpty()) {
        return;
    }
    emit uiDest.serverAddress->setText(currentItem->text());
}

void Drop2FTP::openIconDialog()
{
    KIconDialog *iDialog = new KIconDialog();
    m_newIcon = iDialog->getIcon();
    if (!m_newIcon.isEmpty()) {
        uiAppear.iconButton->setIcon(KIcon(m_newIcon));
    }
}

void Drop2FTP::slotResult(KJob *job)
{
    m_numActivJobs--;
    if (job->error()) {
        m_icon->setIcon(KIcon("dialog-close"));
    } else {
        m_icon->setIcon(KIcon("dialog-ok"));
    }
    QTimer::singleShot(3000, this, SLOT(resetIcon()));
    emit update();
}

void Drop2FTP::resetIcon()
{
    if (m_numActivJobs == 0) {
        m_icon->setIcon(KIcon(m_iconName));
    } else {
        m_icon->setIcon(KIcon("view-history"));
    }
    emit update();
}

void Drop2FTP::sendToServer(KUrl::List &src)
{
    const KUrl dest = m_serverAddress;
    /*FIXME check if url is a remote server and if the computer is online
    if (!dest.isLocalFile()) {
        if (Solid::Networking::Status() == Solid::Networking::Connected) {
            kDebug() << "Computer is online";
        } else {
            kDebug() << "Computer is offline";
        }
    }*/
    m_icon->setIcon(KIcon("view-history")); //TODO an animation would be nicer, wouldn't it? (BusyWidget)
    KJob *cJob;
    m_numActivJobs++;
    if (m_hideProgressInfo) {
        cJob = KIO::copy(src, dest, KIO::HideProgressInfo);
    } else {
        cJob = KIO::copy(src, dest);
    }
    connect(cJob, SIGNAL(result(KJob *)), this, SLOT(slotResult(KJob *))); //http://api.kde.org/4.x-api/kdelibs-apidocs/kio/html/classKIO_1_1CopyJob.html
}

void Drop2FTP::dragEnterEvent(QGraphicsSceneDragDropEvent *event)
{
    if (event->mimeData()->hasUrls()) {
        event->acceptProposedAction();
    }
}

void Drop2FTP::dropEvent(QGraphicsSceneDragDropEvent *event)
{
    if (m_serverAddress.isEmpty()) {
        emit showConfigurationInterface();
        return;
    }
    if (event->mimeData()->hasUrls()) {
        //The URLs are in QList<QUrl> but we need a KUrl::List
        KUrl::List urlList;
        foreach (KUrl fileName, event->mimeData()->urls()) {
            if (fileName.isLocalFile()) {
                urlList << fileName;
            }
        }
        if (!urlList.isEmpty()) {
            sendToServer(urlList);
        }
    }
    event->acceptProposedAction();
}

#include "drop2ftp.moc"
