/***************************************************************************
*   Copyright (C) 2009 by Georg Hennig <georg.hennig@web.de>              *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 3 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .        *
***************************************************************************/

#include <QDomDocument>
#include <QProcess>
#include <QTemporaryFile>
#include <QTimer>
#include <QTextCodec>

#include <KDebug>

#include <KIO/Job>
#include <KIO/NetAccess>

#include "plasma-cwp-update-thread.h"

Update_Thread::Update_Thread( QObject *parent, const QString &path, const uint identifier ) :
	QThread( parent ), local_data_path( path ), local_data_identifier( identifier )
{
	locale = "C";
	encoding = "";

	stop_thread = false;
	force_update_triggered = false;

	data_update_time = "";
	data_provider_update_time = "";

	xmlFile = "";

	raw_data_urlc = new QByteArray();
	raw_data_url1 = new QByteArray();
	raw_data_url2 = new QByteArray();
	raw_data_url3 = new QByteArray();
	raw_data_url4 = new QByteArray();
	raw_data_url5 = new QByteArray();
	raw_data_url6 = new QByteArray();
	raw_data_url7 = new QByteArray();

	// load weather data saved inside <path> using <identifier>
	loadLocalData();
}

Update_Thread::~Update_Thread()
{
	// save weather data saved inside <path> using <identifier>
	saveLocalData();

	delete raw_data_urlc;
	delete raw_data_url1;
	delete raw_data_url2;
	delete raw_data_url3;
	delete raw_data_url4;
	delete raw_data_url5;
	delete raw_data_url6;
	delete raw_data_url7;
}

void Update_Thread::run()
{
	// delay for a few seconds, workaround for "empty folderview bug"
	msleep( 15000 );

	updateFrequency = "1";
	collecting_data = false;

	unsigned long i;
	int freq;
	bool error;
	for (;;)
	{
		mutex.lock();
		collecting_data = true;
		collecting_data_start_time = QTime::currentTime();
		mutex.unlock();
		collectAndParseData();
		mutex.lock();
		collecting_data = false;
		mutex.unlock();
		emit( data_fetched() );
		mutex.lock();
		freq = updateFrequency.toInt(&error)*60*20;
		mutex.unlock();
		for(i=0; i<(unsigned long)freq; i++)
		{
			mutex.lock();
			if ( stop_thread )
			{
				mutex.unlock();
				kDebug() << "Thread stopped, return from run() function.";
				return;
			}
			 // reload after one minute, when no data has been loaded yet, or if reload has been forced
			if ( force_update_triggered || ( data_update_time == "" && i >= 60*20 ) )
			{
				mutex.unlock();
				break;
			}
			mutex.unlock();
			msleep( 50 );
		}

		mutex.lock();
		force_update_triggered = false;
		mutex.unlock();
	}
}

void Update_Thread::stop()
{
	mutex.lock();
	stop_thread = true;
	mutex.unlock();
}

bool Update_Thread::isCollectingDataHanging()
{
	bool ret = false;
	mutex.lock();
	if ( collecting_data )
	{
		bool error;
		if ( collecting_data_start_time.secsTo( QTime::currentTime() ) > updateFrequency.toInt(&error)*60 ) ret = true;
	}
	mutex.unlock();

	return ret;
}

void Update_Thread::force_update()
{
	mutex.lock();
	force_update_triggered = true;
	mutex.unlock();
}

void Update_Thread::loadLocalData()
{
	mutex.lock();

	if ( local_data_path == "" || local_data_identifier == 0 )
	{
		mutex.unlock();
		return;
	}

	KConfig cf( local_data_path + QString( "/cwp_local_data_%1.cfg" ).arg( local_data_identifier ), KConfig::SimpleConfig );
	KConfigGroup cfg( &cf, "" );

	data_update_time = cfg.readEntry( "data_update_time", "" );
	data_provider_update_time = cfg.readEntry( "data_provider_update_time", "" );

	data_location_location = cfg.readEntry( "data_location_location", "" );
	data_location_country = cfg.readEntry( "data_location_country", "" );

	data_sun_sunrise = cfg.readEntry( "data_sun_sunrise", "" );
	data_sun_sunset = cfg.readEntry( "data_sun_sunset", "" );

	data_current_temperature = cfg.readEntry( "data_current_temperature", "" );
	data_current_temperature_felt = cfg.readEntry( "data_current_temperature_felt", "" );
	data_current_wind_code = cfg.readEntry( "data_current_wind_code", "" );
	data_current_wind_speed = cfg.readEntry( "data_current_wind_speed", "" );
	data_current_wind = cfg.readEntry( "data_current_wind", "" );
	data_current_humidity = cfg.readEntry( "data_current_humidity", "" );
	data_current_icon_code = cfg.readEntry( "data_current_icon_code", -1 );

	data_current_icon_text = cfg.readEntry( "data_current_icon_text", "" );

	data_current_rain = cfg.readEntry( "data_current_rain", "" );
	data_current_dew_point = cfg.readEntry( "data_current_dew_point", "" );
	data_current_visibility = cfg.readEntry( "data_current_visibility", "" );
	data_current_pressure = cfg.readEntry( "data_current_pressure", "" );
	data_current_uv_index = cfg.readEntry( "data_current_uv_index", "" );

	data_day_name[0] = cfg.readEntry( "data_day_name[0]", "" );
	data_day_temperature_high[0] = cfg.readEntry( "data_day_temperature_high[0]", "" );
	data_day_temperature_low[0] = cfg.readEntry( "data_day_temperature_low[0]", "" );
	data_day_icon_code[0] = cfg.readEntry( "data_day_icon_code[0]", -1 );
	data_day_icon_text[0] = cfg.readEntry( "data_day_icon_text[0]", "" );

	data_day_name[1] = cfg.readEntry( "data_day_name[1]", "" );
	data_day_temperature_high[1] = cfg.readEntry( "data_day_temperature_high[1]", "" );
	data_day_temperature_low[1] = cfg.readEntry( "data_day_temperature_low[1]", "" );
	data_day_icon_code[1] = cfg.readEntry( "data_day_icon_code[1]", -1 );
	data_day_icon_text[1] = cfg.readEntry( "data_day_icon_text[1]", "" );

	data_day_name[2] = cfg.readEntry( "data_day_name[2]", "" );
	data_day_temperature_high[2] = cfg.readEntry( "data_day_temperature_high[2]", "" );
	data_day_temperature_low[2] = cfg.readEntry( "data_day_temperature_low[2]", "" );
	data_day_icon_code[2] = cfg.readEntry( "data_day_icon_code[2]", -1 );
	data_day_icon_text[2] = cfg.readEntry( "data_day_icon_text[2]", "" );

	data_day_name[3] = cfg.readEntry( "data_day_name[3]", "" );
	data_day_temperature_high[3] = cfg.readEntry( "data_day_temperature_high[3]", "" );
	data_day_temperature_low[3] = cfg.readEntry( "data_day_temperature_low[3]", "" );
	data_day_icon_code[3] = cfg.readEntry( "data_day_icon_code[3]", -1 );
	data_day_icon_text[3] = cfg.readEntry( "data_day_icon_text[3]", "" );

	data_day_name[4] = cfg.readEntry( "data_day_name[4]", "" );
	data_day_temperature_high[4] = cfg.readEntry( "data_day_temperature_high[4]", "" );
	data_day_temperature_low[4] = cfg.readEntry( "data_day_temperature_low[4]", "" );
	data_day_icon_code[4] = cfg.readEntry( "data_day_icon_code[4]", -1 );
	data_day_icon_text[4] = cfg.readEntry( "data_day_icon_text[4]", "" );

	data_day_name[5] = cfg.readEntry( "data_day_name[5]", "" );
	data_day_temperature_high[5] = cfg.readEntry( "data_day_temperature_high[5]", "" );
	data_day_temperature_low[5] = cfg.readEntry( "data_day_temperature_low[5]", "" );
	data_day_icon_code[5] = cfg.readEntry( "data_day_icon_code[5]", -1 );
	data_day_icon_text[5] = cfg.readEntry( "data_day_icon_text[5]", "" );

	data_day_name[6] = cfg.readEntry( "data_day_name[6]", "" );
	data_day_temperature_high[6] = cfg.readEntry( "data_day_temperature_high[6]", "" );
	data_day_temperature_low[6] = cfg.readEntry( "data_day_temperature_low[6]", "" );
	data_day_icon_code[6] = cfg.readEntry( "data_day_icon_code[6]", -1 );
	data_day_icon_text[6] = cfg.readEntry( "data_day_icon_text[6]", "" );

	// png must be used, as images (icons nearly always) contain transparent parts. jpg doesn't support transparency.
	int i;
	data_custom_image_list.clear();
	for ( i=0; i<100; i++ ) // random maximum location image number
	{
		QFile file( local_data_path + QString( "/cwp_custom_image_%1_%2.img" ).arg( i ).arg( local_data_identifier ) );
		if ( !file.open(QIODevice::ReadOnly) ) break;
		data_custom_image_list.push_back( file.readAll() );
	}
	data_current_icon = QImage( local_data_path + QString( "/cwp_icon_current_%1.png" ).arg( local_data_identifier ) );
	data_day_icon[0] = QImage( local_data_path + QString( "/cwp_icon_0_%1.png" ).arg( local_data_identifier ) );
	data_day_icon[1] = QImage( local_data_path + QString( "/cwp_icon_1_%1.png" ).arg( local_data_identifier ) );
	data_day_icon[2] = QImage( local_data_path + QString( "/cwp_icon_2_%1.png" ).arg( local_data_identifier ) );
	data_day_icon[3] = QImage( local_data_path + QString( "/cwp_icon_3_%1.png" ).arg( local_data_identifier ) );
	data_day_icon[4] = QImage( local_data_path + QString( "/cwp_icon_4_%1.png" ).arg( local_data_identifier ) );
	data_day_icon[5] = QImage( local_data_path + QString( "/cwp_icon_5_%1.png" ).arg( local_data_identifier ) );
	data_day_icon[6] = QImage( local_data_path + QString( "/cwp_icon_6_%1.png" ).arg( local_data_identifier ) );

	if ( data_update_time != "" ) emit( data_fetched() );

	mutex.unlock();
}

void Update_Thread::saveLocalData()
{
	mutex.lock();

	if ( local_data_path == "" || local_data_identifier == 0 )
	{
		mutex.unlock();
		return;
	}

	KConfig cf( local_data_path + QString( "/cwp_local_data_%1.cfg" ).arg( local_data_identifier ), KConfig::SimpleConfig );
	KConfigGroup cfg( &cf, "" );

	cfg.writeEntry( "data_update_time", data_update_time );
	cfg.writeEntry( "data_provider_update_time", data_provider_update_time );

	cfg.writeEntry( "data_location_location", data_location_location );
	cfg.writeEntry( "data_location_country", data_location_country );

	cfg.writeEntry( "data_sun_sunrise", data_sun_sunrise );
	cfg.writeEntry( "data_sun_sunset", data_sun_sunset );

	cfg.writeEntry( "data_current_temperature", data_current_temperature );
	cfg.writeEntry( "data_current_temperature_felt", data_current_temperature_felt );
	cfg.writeEntry( "data_current_wind_code", data_current_wind_code );
	cfg.writeEntry( "data_current_wind_speed", data_current_wind_speed );
	cfg.writeEntry( "data_current_wind", data_current_wind );
	cfg.writeEntry( "data_current_humidity", data_current_humidity );
	cfg.writeEntry( "data_current_icon_code", data_current_icon_code );

	cfg.writeEntry( "data_current_icon_text", data_current_icon_text );

	cfg.writeEntry( "data_current_rain", data_current_rain );
	cfg.writeEntry( "data_current_dew_point", data_current_dew_point );
	cfg.writeEntry( "data_current_visibility", data_current_visibility );
	cfg.writeEntry( "data_current_pressure", data_current_pressure );
	cfg.writeEntry( "data_current_uv_index", data_current_uv_index );

	cfg.writeEntry( "data_day_name[0]", data_day_name[0] );
	cfg.writeEntry( "data_day_temperature_high[0]", data_day_temperature_high[0] );
	cfg.writeEntry( "data_day_temperature_low[0]", data_day_temperature_low[0] );
	cfg.writeEntry( "data_day_icon_code[0]", data_day_icon_code[0] );
	cfg.writeEntry( "data_day_icon_text[0]", data_day_icon_text[0] );

	cfg.writeEntry( "data_day_name[1]", data_day_name[1] );
	cfg.writeEntry( "data_day_temperature_high[1]", data_day_temperature_high[1] );
	cfg.writeEntry( "data_day_temperature_low[1]", data_day_temperature_low[1] );
	cfg.writeEntry( "data_day_icon_code[1]", data_day_icon_code[1] );
	cfg.writeEntry( "data_day_icon_text[1]", data_day_icon_text[1] );

	cfg.writeEntry( "data_day_name[2]", data_day_name[2] );
	cfg.writeEntry( "data_day_temperature_high[2]", data_day_temperature_high[2] );
	cfg.writeEntry( "data_day_temperature_low[2]", data_day_temperature_low[2] );
	cfg.writeEntry( "data_day_icon_code[2]", data_day_icon_code[2] );
	cfg.writeEntry( "data_day_icon_text[2]", data_day_icon_text[2] );

	cfg.writeEntry( "data_day_name[3]", data_day_name[3] );
	cfg.writeEntry( "data_day_temperature_high[3]", data_day_temperature_high[3] );
	cfg.writeEntry( "data_day_temperature_low[3]", data_day_temperature_low[3] );
	cfg.writeEntry( "data_day_icon_code[3]", data_day_icon_code[3] );
	cfg.writeEntry( "data_day_icon_text[3]", data_day_icon_text[3] );

	cfg.writeEntry( "data_day_name[4]", data_day_name[4] );
	cfg.writeEntry( "data_day_temperature_high[4]", data_day_temperature_high[4] );
	cfg.writeEntry( "data_day_temperature_low[4]", data_day_temperature_low[4] );
	cfg.writeEntry( "data_day_icon_code[4]", data_day_icon_code[4] );
	cfg.writeEntry( "data_day_icon_text[4]", data_day_icon_text[4] );

	cfg.writeEntry( "data_day_name[5]", data_day_name[5] );
	cfg.writeEntry( "data_day_temperature_high[5]", data_day_temperature_high[5] );
	cfg.writeEntry( "data_day_temperature_low[5]", data_day_temperature_low[5] );
	cfg.writeEntry( "data_day_icon_code[5]", data_day_icon_code[5] );
	cfg.writeEntry( "data_day_icon_text[5]", data_day_icon_text[5] );

	cfg.writeEntry( "data_day_name[6]", data_day_name[6] );
	cfg.writeEntry( "data_day_temperature_high[6]", data_day_temperature_high[6] );
	cfg.writeEntry( "data_day_temperature_low[6]", data_day_temperature_low[6] );
	cfg.writeEntry( "data_day_icon_code[6]", data_day_icon_code[6] );
	cfg.writeEntry( "data_day_icon_text[6]", data_day_icon_text[6] );

	// png must be used, as images (icons nearly always) contain transparent parts. jpg doesn't support transparency.
	int i;
	for ( i=0; i<data_custom_image_list.size(); i++ )
	{
		QFile file( local_data_path + QString( "/cwp_custom_image_%1_%2.img" ).arg( i ).arg( local_data_identifier ) );
		file.open( QIODevice::WriteOnly );
		file.write( data_custom_image_list.at( i ) );
	}
	data_current_icon.save( local_data_path + QString( "/cwp_icon_current_%1.png" ).arg( local_data_identifier ) );
	data_day_icon[0].save( local_data_path + QString( "/cwp_icon_0_%1.png" ).arg( local_data_identifier ) );
	data_day_icon[1].save( local_data_path + QString( "/cwp_icon_1_%1.png" ).arg( local_data_identifier ) );
	data_day_icon[2].save( local_data_path + QString( "/cwp_icon_2_%1.png" ).arg( local_data_identifier ) );
	data_day_icon[3].save( local_data_path + QString( "/cwp_icon_3_%1.png" ).arg( local_data_identifier ) );
	data_day_icon[4].save( local_data_path + QString( "/cwp_icon_4_%1.png" ).arg( local_data_identifier ) );
	data_day_icon[5].save( local_data_path + QString( "/cwp_icon_5_%1.png" ).arg( local_data_identifier ) );
	data_day_icon[6].save( local_data_path + QString( "/cwp_icon_6_%1.png" ).arg( local_data_identifier ) );

	mutex.unlock();
}

void Update_Thread::get_weather_values( QString &data_location_location_tmp, QString &data_location_country_tmp, QString &data_sun_sunrise_tmp,
	QString &data_sun_sunset_tmp, QString &data_current_temperature_tmp, QString &data_current_temperature_felt_tmp,
	QString &data_current_wind_code_tmp, QString &data_current_wind_speed_tmp, QString &data_current_wind_tmp,
	QString &data_current_humidity_tmp, QString &data_current_dew_point_tmp,
	QString &data_current_visibility_tmp, QString &data_current_pressure_tmp, QString &data_current_uv_index_tmp,
	int &data_current_icon_code_tmp, QImage &data_current_icon_tmp,
	QString &data_current_icon_text_tmp, QString &data_current_rain_tmp, QString *data_day_name_tmp,
	QString *data_day_temperature_high_tmp, QString *data_day_temperature_low_tmp,
	int *data_day_icon_code_tmp, QImage *data_day_icon_tmp, QString *data_day_icon_text_tmp, QString &data_update_time_tmp,
	QString &data_provider_update_time_tmp, QString &tempType_tmp, QList<QByteArray> &data_custom_image_list_tmp )
{
	mutex.lock();

	data_update_time_tmp = data_update_time;
	data_provider_update_time_tmp = data_provider_update_time;
	data_location_location_tmp = data_location_location;
	data_location_country_tmp = data_location_country;
	data_sun_sunrise_tmp = data_sun_sunrise;
	data_sun_sunset_tmp = data_sun_sunset;
	data_current_temperature_tmp = data_current_temperature;
	data_current_temperature_felt_tmp = data_current_temperature_felt;
	data_current_wind_code_tmp = data_current_wind_code;
	data_current_wind_speed_tmp = data_current_wind_speed;
	data_current_wind_tmp = data_current_wind;
	data_current_humidity_tmp = data_current_humidity;
	data_current_icon_code_tmp = data_current_icon_code;
	data_current_icon_tmp = data_current_icon;
	data_current_icon_text_tmp = data_current_icon_text;
	data_current_rain_tmp = data_current_rain;
	data_current_dew_point_tmp = data_current_dew_point;
	data_current_visibility_tmp = data_current_visibility;
	data_current_pressure_tmp = data_current_pressure;
	data_current_uv_index_tmp = data_current_uv_index;
	tempType_tmp = tempType;
	data_custom_image_list_tmp = data_custom_image_list;

	int i;
	for(i=0; i<7; i++)
	{
		data_day_name_tmp[i] = data_day_name[i];
		data_day_temperature_high_tmp[i] = data_day_temperature_high[i];
		data_day_temperature_low_tmp[i] = data_day_temperature_low[i];
		data_day_icon_code_tmp[i] = data_day_icon_code[i];
		data_day_icon_tmp[i] = data_day_icon[i];
		data_day_icon_text_tmp[i] = data_day_icon_text[i];
	}
	mutex.unlock();
}

void Update_Thread::set_config_values( const QString &updateFrequency_tmp, const QString &xmlFile_tmp, const QString &zip_tmp, const QString &feelsLike_tmp,
	const QString &humidity_tmp, const QString &wind_tmp, const QString &tempType_tmp, const QList<KUrl> &customImageList_tmp )
{
	mutex.lock();

	updateFrequency = updateFrequency_tmp;
	xmlFile = xmlFile_tmp;
	zip = zip_tmp;
	feelsLike = feelsLike_tmp;
	humidity = humidity_tmp;
	wind = wind_tmp;
	tempType = tempType_tmp;
	customImageList = customImageList_tmp;

	mutex.unlock();
}

void Update_Thread::collectAndParseData()
{
	if ( xmlFile == "" ) return;

	// parse xml file
	QString urlc_prefix_xml;
	QString urlc_suffix_xml;

	QString url1_prefix_xml;
	QString url1_suffix_xml;

	QString url2_prefix_xml;
	QString url2_suffix_xml;

	QString url3_prefix_xml;
	QString url3_suffix_xml;

	QString url4_prefix_xml;
	QString url4_suffix_xml;

	QString url5_prefix_xml;
	QString url5_suffix_xml;

	QString url6_prefix_xml;
	QString url6_suffix_xml;

	QString url7_prefix_xml;
	QString url7_suffix_xml;

	QString urlc_follow_xml;
	QString url1_follow_xml;
	QString url2_follow_xml;
	QString url3_follow_xml;
	QString url4_follow_xml;
	QString url5_follow_xml;
	QString url6_follow_xml;
	QString url7_follow_xml;

	QString locale_xml;
	QString encoding_xml;

	QString data_location_url_xml;
	QString data_location_location_xml;
	QString data_location_country_xml;

	QString data_sun_url_xml;
	QString data_sun_sunrise_xml;
	QString data_sun_sunset_xml;

	QString data_current_temperature_url_xml;
	QString data_provider_update_time_xml;
	QString data_current_temperature_xml;
	QString data_current_temperature_felt_xml;
	QString data_current_wind_url_xml;
	QString data_current_wind_code_xml;
	QString data_current_wind_speed_xml;
	QString data_current_wind_xml;
	QString data_current_icon_url_xml;
	QString data_current_icon_code_xml;
	QString data_current_icon_xml;
	QString data_current_icon_text_xml;
	QString data_current_additional_url_xml;
	QString data_current_humidity_xml;
	QString data_current_rain_xml;
	QString data_current_dew_point_xml;
	QString data_current_visibility_xml;
	QString data_current_pressure_xml;
	QString data_current_uv_index_xml;

	QString data_day_url_xml[7];
	QString data_day_name_xml[7];
	QString data_day_temperature_high_xml[7];
	QString data_day_temperature_low_xml[7];
	QString data_day_icon_code_xml[7];
	QString data_day_icon_xml[7];
	QString data_day_icon_text_xml[7];

	const int WEATHER_CODE_TRANSFORM_SIZE = 110;
	QString data_icon_transform_in_xml[WEATHER_CODE_TRANSFORM_SIZE];
	QString data_icon_transform_out_xml[WEATHER_CODE_TRANSFORM_SIZE];

	const int WIND_TRANSFORM_SIZE = 500;
	QString data_wind_transform_in_xml[WIND_TRANSFORM_SIZE];
	QString data_wind_transform_out_xml[WIND_TRANSFORM_SIZE];

	int i;
	for ( i=0; i<WEATHER_CODE_TRANSFORM_SIZE; i++ )
	{
		data_icon_transform_in_xml[i] = "";
		data_icon_transform_out_xml[i] = "";
	}

	for ( i=0; i<WIND_TRANSFORM_SIZE; i++ )
	{
		data_wind_transform_in_xml[i] = "";
		data_wind_transform_out_xml[i] = "";
	}

	QDomDocument doc("weather_xml");
	mutex.lock();
	QFile file(xmlFile);
	mutex.unlock();

	//read the data into the XML parser and close the file
	doc.setContent(&file);
	file.close();

	QDomElement root = doc.documentElement();

	QDomNode n = root.firstChild();
	QDomNode n_old;

	while(!n.isNull())
	{
		QDomElement e = n.toElement();

		if(e.tagName() == "locale_settings")
		{
			locale_xml = e.attribute("locale", "");
			encoding_xml = e.attribute("encoding", "");
		}

		if(e.tagName() == "urlc")
		{
			urlc_prefix_xml = e.attribute("urlc_prefix", "");
			urlc_suffix_xml = e.attribute("urlc_suffix", "");
			urlc_follow_xml = e.attribute("urlc_follow", "");
		}

		if(e.tagName() == "url1")
		{
			url1_prefix_xml = e.attribute("url1_prefix", "");
			url1_suffix_xml = e.attribute("url1_suffix", "");
			url1_follow_xml = e.attribute("url1_follow", "");
		}

		if(e.tagName() == "url2")
		{
			url2_prefix_xml = e.attribute("url2_prefix", "");
			url2_suffix_xml = e.attribute("url2_suffix", "");
			url2_follow_xml = e.attribute("url2_follow", "");
		}

		if(e.tagName() == "url3")
		{
			url3_prefix_xml = e.attribute("url3_prefix", "");
			url3_suffix_xml = e.attribute("url3_suffix", "");
			url3_follow_xml = e.attribute("url3_follow", "");
		}

		if(e.tagName() == "url4")
		{
			url4_prefix_xml = e.attribute("url4_prefix", "");
			url4_suffix_xml = e.attribute("url4_suffix", "");
			url4_follow_xml = e.attribute("url4_follow", "");
		}

		if(e.tagName() == "url5")
		{
			url5_prefix_xml = e.attribute("url5_prefix", "");
			url5_suffix_xml = e.attribute("url5_suffix", "");
			url5_follow_xml = e.attribute("url5_follow", "");
		}

		if(e.tagName() == "url6")
		{
			url6_prefix_xml = e.attribute("url6_prefix", "");
			url6_suffix_xml = e.attribute("url6_suffix", "");
			url6_follow_xml = e.attribute("url6_follow", "");
		}

		if(e.tagName() == "url7")
		{
			url7_prefix_xml = e.attribute("url7_prefix", "");
			url7_suffix_xml = e.attribute("url7_suffix", "");
			url7_follow_xml = e.attribute("url7_follow", "");
		}

		if(e.tagName() == "data:location")
		{
			data_location_url_xml = e.attribute("url", "");
			data_location_location_xml = e.attribute("location", "");
			data_location_country_xml = e.attribute("country", "");
		}

		if(e.tagName() == "data:sun")
		{
			data_sun_url_xml = e.attribute("url", "");
			data_sun_sunrise_xml = e.attribute("sunrise", "");
			data_sun_sunset_xml = e.attribute("sunset", "");
		}

		if(e.tagName() == "data:current:temperature")
		{
			data_current_temperature_url_xml = e.attribute("url", "");
			data_provider_update_time_xml = e.attribute("update_time", "");
			data_current_temperature_xml = e.attribute("temperature", "");
			data_current_temperature_felt_xml = e.attribute("temperature_felt", "");
		}

		if(e.tagName() == "data:current:wind")
		{
			data_current_wind_url_xml = e.attribute("url", "");
			data_current_wind_code_xml = e.attribute("wind_code", "");
			data_current_wind_speed_xml = e.attribute("wind_speed", "");
			data_current_wind_xml = e.attribute("wind", "");
		}

		if(e.tagName() == "data:current:icon")
		{
			data_current_icon_url_xml = e.attribute("url", "");
			data_current_icon_code_xml = e.attribute("icon_code", "");
			data_current_icon_xml = e.attribute("icon", "");
			data_current_icon_text_xml = e.attribute("icon_text", "");
		}

		if(e.tagName() == "data:current:additional")
		{
			data_current_additional_url_xml = e.attribute("url", "");
			data_current_humidity_xml = e.attribute("humidity", "");
			data_current_rain_xml = e.attribute("rain", "");
			data_current_dew_point_xml = e.attribute("dew_point", "");
			data_current_visibility_xml = e.attribute("visibility", "");
			data_current_pressure_xml = e.attribute("pressure", "");
			data_current_uv_index_xml = e.attribute("uv_index", "");
		}

		if(e.tagName() == "data:day1")
		{
			data_day_url_xml[0] = e.attribute("url", "");
			data_day_name_xml[0] = e.attribute("name", "");
			data_day_temperature_high_xml[0] = e.attribute("temperature_high", "");
			data_day_temperature_low_xml[0] = e.attribute("temperature_low", "");
			data_day_icon_code_xml[0] = e.attribute("icon_code", "");
			data_day_icon_xml[0] = e.attribute("icon", "");
			data_day_icon_text_xml[0] = e.attribute("icon_text", "");
		}

		if(e.tagName() == "data:day2")
		{
			data_day_url_xml[1] = e.attribute("url", "");
			data_day_name_xml[1] = e.attribute("name", "");
			data_day_temperature_high_xml[1] = e.attribute("temperature_high", "");
			data_day_temperature_low_xml[1] = e.attribute("temperature_low", "");
			data_day_icon_code_xml[1] = e.attribute("icon_code", "");
			data_day_icon_xml[1] = e.attribute("icon", "");
			data_day_icon_text_xml[1] = e.attribute("icon_text", "");
		}

		if(e.tagName() == "data:day3")
		{
			data_day_url_xml[2] = e.attribute("url", "");
			data_day_name_xml[2] = e.attribute("name", "");
			data_day_temperature_high_xml[2] = e.attribute("temperature_high", "");
			data_day_temperature_low_xml[2] = e.attribute("temperature_low", "");
			data_day_icon_code_xml[2] = e.attribute("icon_code", "");
			data_day_icon_xml[2] = e.attribute("icon", "");
			data_day_icon_text_xml[2] = e.attribute("icon_text", "");
		}

		if(e.tagName() == "data:day4")
		{
			data_day_url_xml[3] = e.attribute("url", "");
			data_day_name_xml[3] = e.attribute("name", "");
			data_day_temperature_high_xml[3] = e.attribute("temperature_high", "");
			data_day_temperature_low_xml[3] = e.attribute("temperature_low", "");
			data_day_icon_code_xml[3] = e.attribute("icon_code", "");
			data_day_icon_xml[3] = e.attribute("icon", "");
			data_day_icon_text_xml[3] = e.attribute("icon_text", "");
		}

		if(e.tagName() == "data:day5")
		{
			data_day_url_xml[4] = e.attribute("url", "");
			data_day_name_xml[4] = e.attribute("name", "");
			data_day_temperature_high_xml[4] = e.attribute("temperature_high", "");
			data_day_temperature_low_xml[4] = e.attribute("temperature_low", "");
			data_day_icon_code_xml[4] = e.attribute("icon_code", "");
			data_day_icon_xml[4] = e.attribute("icon", "");
			data_day_icon_text_xml[4] = e.attribute("icon_text", "");
		}

		if(e.tagName() == "data:day6")
		{
			data_day_url_xml[5] = e.attribute("url", "");
			data_day_name_xml[5] = e.attribute("name", "");
			data_day_temperature_high_xml[5] = e.attribute("temperature_high", "");
			data_day_temperature_low_xml[5] = e.attribute("temperature_low", "");
			data_day_icon_code_xml[5] = e.attribute("icon_code", "");
			data_day_icon_xml[5] = e.attribute("icon", "");
			data_day_icon_text_xml[5] = e.attribute("icon_text", "");
		}

		if(e.tagName() == "data:day7")
		{
			data_day_url_xml[6] = e.attribute("url", "");
			data_day_name_xml[6] = e.attribute("name", "");
			data_day_temperature_high_xml[6] = e.attribute("temperature_high", "");
			data_day_temperature_low_xml[6] = e.attribute("temperature_low", "");
			data_day_icon_code_xml[6] = e.attribute("icon_code", "");
			data_day_icon_xml[6] = e.attribute("icon", "");
			data_day_icon_text_xml[6] = e.attribute("icon_text", "");
		}

		if(e.tagName() == "icon:transform")
		{
			QString transform_number;
			int n;
			for ( n=0; n<WEATHER_CODE_TRANSFORM_SIZE; n++ )
			{
				transform_number.setNum( n+1 );
				data_icon_transform_in_xml[n] = e.attribute(QString( "i" ) + transform_number, "");
				data_icon_transform_out_xml[n] = e.attribute(QString( "o" ) + transform_number, "");
			}
		}

		if(e.tagName() == "wind:transform")
		{
			QString transform_number;
			int n;
			for ( n=0; n<WIND_TRANSFORM_SIZE; n++ )
			{
				transform_number.setNum( n+1 );
				data_wind_transform_in_xml[n] = e.attribute(QString( "i" ) + transform_number, "");
				data_wind_transform_out_xml[n] = e.attribute(QString( "o" ) + transform_number, "");
			}
		}

		n = n.nextSibling();
	}

	if ( stop_thread ) return;

	// retrive weather files
	bool download_success_url[8] = { true, true, true, true, true, true, true, true };

	if ( urlc_prefix_xml != "" )
		download_success_url[0] = downloadFile( urlc_prefix_xml + zip + urlc_suffix_xml, raw_data_urlc );
	if ( stop_thread ) return;

	if ( url1_prefix_xml != "" )
		download_success_url[1] = downloadFile( url1_prefix_xml + zip + url1_suffix_xml, raw_data_url1 );
	if ( stop_thread ) return;

	if ( url2_prefix_xml != "" )
		download_success_url[2] = downloadFile( url2_prefix_xml + zip + url2_suffix_xml, raw_data_url2 );
	if ( stop_thread ) return;

	if ( url3_prefix_xml != "" )
	 download_success_url[3] = downloadFile( url3_prefix_xml + zip + url3_suffix_xml, raw_data_url3 );
	if ( stop_thread ) return;

	if ( url4_prefix_xml != "" )
		download_success_url[4] = downloadFile( url4_prefix_xml + zip + url4_suffix_xml, raw_data_url4 );
	if ( stop_thread ) return;

	if ( url5_prefix_xml != "" )
		download_success_url[5] = downloadFile( url5_prefix_xml + zip + url5_suffix_xml, raw_data_url5 );
	if ( stop_thread ) return;

	if ( url6_prefix_xml != "" )
		download_success_url[6] = downloadFile( url6_prefix_xml + zip + url6_suffix_xml, raw_data_url6 );
	if ( stop_thread ) return;

	if ( url7_prefix_xml != "" )
		download_success_url[7] = downloadFile( url7_prefix_xml + zip + url7_suffix_xml, raw_data_url7 );
	if ( stop_thread ) return;

	// run url follow commands & retrive followed weather files
	if ( urlc_follow_xml != "" )
	{
		QString urlc_new = runCommand( urlc_follow_xml, raw_data_urlc );
		download_success_url[0] = downloadFile( urlc_new, raw_data_urlc );
	}
	if ( stop_thread ) return;

	if ( url1_follow_xml != "" )
	{
		QString url1_new = runCommand( url1_follow_xml, raw_data_url1 );
		download_success_url[1] = downloadFile( url1_new, raw_data_url1 );
	}
	if ( stop_thread ) return;

	if ( url2_follow_xml != "" )
	{
		QString url2_new = runCommand( url2_follow_xml, raw_data_url2 );
		download_success_url[2] = downloadFile( url2_new, raw_data_url2 );
	}
	if ( stop_thread ) return;

	if ( url3_follow_xml != "" )
	{
		QString url3_new = runCommand( url3_follow_xml, raw_data_url3 );
		download_success_url[3] = downloadFile( url3_new, raw_data_url3 );
	}
	if ( stop_thread ) return;

	if ( url4_follow_xml != "" )
	{
		QString url4_new = runCommand( url4_follow_xml, raw_data_url4 );
		download_success_url[4] = downloadFile( url4_new, raw_data_url4 );
	}
	if ( stop_thread ) return;

	if ( url5_follow_xml != "" )
	{
		QString url5_new = runCommand( url5_follow_xml, raw_data_url5 );
		download_success_url[5] = downloadFile( url5_new, raw_data_url5 );
	}
	if ( stop_thread ) return;

	if ( url6_follow_xml != "" )
	{
		QString url6_new = runCommand( url6_follow_xml, raw_data_url6 );
		download_success_url[6] = downloadFile( url6_new, raw_data_url6 );
	}
	if ( stop_thread ) return;

	if ( url7_follow_xml != "" )
	{
		QString url7_new = runCommand( url7_follow_xml, raw_data_url7 );
		download_success_url[7] = downloadFile( url7_new, raw_data_url7 );
	}
	if ( stop_thread ) return;

	// Find data file for different values
	QByteArray *raw_data_location = rawDataFromUrl( data_location_url_xml );
	QByteArray *raw_data_sun = rawDataFromUrl( data_sun_url_xml );
	QByteArray *raw_data_current_temperature = rawDataFromUrl( data_current_temperature_url_xml );
	QByteArray *raw_data_current_wind = rawDataFromUrl( data_current_wind_url_xml );
	QByteArray *raw_data_current_icon = rawDataFromUrl( data_current_icon_url_xml );
	QByteArray *raw_data_current_additional = rawDataFromUrl( data_current_additional_url_xml );

	QByteArray *raw_data_day[7];
	QByteArray *tmp_array = new QByteArray();
	for ( i=0; i<7; i++ ) raw_data_day[i] = rawDataFromUrl( data_day_url_xml[i] );

	// run commands from xml file on weather files
	locale = locale_xml;
	encoding = encoding_xml;
	kDebug() << "setting locale to " << locale << " and encoding to " << encoding;

	bool keep_old_value = false;
	QString data_location_location_tmp;
	QString data_location_country_tmp;
	if ( successFromUrl( data_location_url_xml, &download_success_url[0] ) )
	{
		data_location_location_tmp = runCommand( data_location_location_xml, raw_data_location );
		kDebug() << "data_location_location: " << data_location_location_tmp;

		data_location_country_tmp = runCommand( data_location_country_xml, raw_data_location );
		kDebug() << "data_location_country: " << data_location_country_tmp;
	}
	else
	{
		kDebug() << "Download of location url failed, keeping old values.";

		data_location_location_tmp = data_location_location;
		data_location_country_tmp = data_location_country;

		keep_old_value = true;
	}
	if ( stop_thread ) return;

	QString data_sun_sunrise_tmp;
	QString data_sun_sunset_tmp;
	if ( successFromUrl( data_sun_url_xml, &download_success_url[0] ) )
	{
		data_sun_sunrise_tmp = runCommand( data_sun_sunrise_xml, raw_data_sun );
		kDebug() << "data_sun_sunrise: " << data_sun_sunrise_tmp;

		data_sun_sunset_tmp = runCommand( data_sun_sunset_xml, raw_data_sun );
		kDebug() << "data_sun_sunset: " << data_sun_sunset_tmp;
	}
	else
	{
		kDebug() << "Download of sunrise/sunset url failed, keeping old values.";

		data_sun_sunrise_tmp = data_sun_sunrise;
		data_sun_sunset_tmp = data_sun_sunset;

		keep_old_value = true;
	}
	if ( stop_thread ) return;

	QString data_provider_update_time_tmp;
	QString data_current_temperature_tmp;
	QString data_current_temperature_felt_tmp;
	if ( successFromUrl( data_current_temperature_url_xml, &download_success_url[0] ) )
	{
		data_provider_update_time_tmp = runCommand( data_provider_update_time_xml, raw_data_current_temperature );
		kDebug() << "data_provider_update_time: " << data_provider_update_time_tmp;

		data_current_temperature_tmp = runCommand( data_current_temperature_xml, raw_data_current_temperature );
		kDebug() << "data_current_temperature: " << data_current_temperature_tmp;

		data_current_temperature_felt_tmp = runCommand( data_current_temperature_felt_xml, raw_data_current_temperature );
		kDebug() << "data_current_temperature_felt: " << data_current_temperature_felt_tmp;
	}
	else
	{
		kDebug() << "Download of temperature url failed, keeping old values.";

		data_current_temperature_tmp = data_current_temperature;
		data_current_temperature_felt_tmp = data_current_temperature_felt;

		keep_old_value = true;
	}
	if ( stop_thread ) return;

	QString data_current_wind_code_tmp;
	QString data_current_wind_speed_tmp;
	QString data_current_wind_tmp;
	if ( successFromUrl( data_current_wind_url_xml, &download_success_url[0] ) )
	{
		data_current_wind_code_tmp = runCommand( data_current_wind_code_xml, raw_data_current_wind );
		kDebug() << "data_current_wind_code: " << data_current_wind_code_tmp;

		data_current_wind_speed_tmp = runCommand( data_current_wind_speed_xml, raw_data_current_wind );
		kDebug() << "data_current_wind_speed: " << data_current_wind_speed_tmp;

		data_current_wind_tmp = runCommand( data_current_wind_xml, raw_data_current_wind );
		kDebug() << "data_current_wind: " << data_current_wind_tmp;
	}
	else
	{
		kDebug() << "Download of wind url failed, keeping old values.";

		data_current_wind_code_tmp = data_current_wind_code;
		data_current_wind_speed_tmp = data_current_wind_speed;
		data_current_wind_tmp = data_current_wind;

		keep_old_value = true;
	}
	if ( stop_thread ) return;

	QString data_current_icon_code_tmp;
	QString data_current_icon_tmp;
	QString data_current_icon_text_tmp;
	if ( successFromUrl( data_current_icon_url_xml, &download_success_url[0] ) )
	{
		data_current_icon_code_tmp = runCommand( data_current_icon_code_xml, raw_data_current_icon );
		kDebug() << "data_current_icon_code: " << data_current_icon_code_tmp;

		tmp_array->clear();
		data_current_icon_tmp = runCommand( data_current_icon_xml, raw_data_current_icon );
		downloadFile( data_current_icon_tmp, tmp_array );
		QImage icon_tmp = data_current_icon;
		if ( !data_current_icon.loadFromData( *tmp_array ) ) data_current_icon = icon_tmp;
		kDebug() << "data_current_icon: " << data_current_icon_tmp;

		data_current_icon_text_tmp = runCommand( data_current_icon_text_xml, raw_data_current_icon );
		kDebug() << "data_current_icon_text: " << data_current_icon_text_tmp;
	}
	else
	{
		kDebug() << "Download of icon url failed, keeping old values.";

		data_current_icon_code_tmp = data_current_icon_code;
		data_current_icon_text_tmp = data_current_icon_text;

		keep_old_value = true;
	}
	if ( stop_thread ) return;

	QString data_current_humidity_tmp;
	QString data_current_rain_tmp;
	QString data_current_dew_point_tmp;
	QString data_current_visibility_tmp;
	QString data_current_pressure_tmp;
	QString data_current_uv_index_tmp;
	if ( successFromUrl( data_current_additional_url_xml, &download_success_url[0] ) )
	{
		data_current_humidity_tmp = runCommand( data_current_humidity_xml, raw_data_current_additional );
		kDebug() << "data_current_humidity: " << data_current_humidity_tmp;

		data_current_rain_tmp = runCommand( data_current_rain_xml, raw_data_current_additional );
		kDebug() << "data_current_rain: " << data_current_rain_tmp;

		data_current_dew_point_tmp = runCommand( data_current_dew_point_xml, raw_data_current_additional );
		kDebug() << "data_current_dew_point: " << data_current_dew_point_tmp;

		data_current_visibility_tmp = runCommand( data_current_visibility_xml, raw_data_current_additional );
		kDebug() << "data_current_visibility: " << data_current_visibility_tmp;

		data_current_pressure_tmp = runCommand( data_current_pressure_xml, raw_data_current_additional );
		kDebug() << "data_current_pressure: " << data_current_pressure_tmp;

		data_current_uv_index_tmp = runCommand( data_current_uv_index_xml, raw_data_current_additional );
		kDebug() << "data_current_uv_index: " << data_current_uv_index_tmp;
	}
	else
	{
		kDebug() << "Download of additional url failed, keeping old values.";

		data_current_humidity_tmp = data_current_humidity;
		data_current_rain_tmp = data_current_rain;
		data_current_dew_point_tmp = data_current_dew_point;
		data_current_visibility_tmp = data_current_visibility;
		data_current_pressure_tmp = data_current_pressure;
		data_current_uv_index_tmp = data_current_uv_index;

		keep_old_value = true;
	}
	if ( stop_thread ) return;

	QString data_day_name_tmp[7];
	QString data_day_temperature_high_tmp[7];
	QString data_day_temperature_low_tmp[7];
	QString data_day_icon_code_tmp[7];
	QString data_day_icon_tmp[7];
	QString data_day_icon_text_tmp[7];

	for(i=0; i<7; i++)
	{
		if ( successFromUrl( data_day_url_xml[i], &download_success_url[0] ) )
		{
			data_day_name_tmp[i] = runCommand( data_day_name_xml[i], raw_data_day[i] );
			kDebug() << "data_day_name[" << i << "]: " << data_day_name_tmp[i];

			data_day_temperature_high_tmp[i] = runCommand( data_day_temperature_high_xml[i], raw_data_day[i] );
			kDebug() << "data_day_temperature_high[" << i << "]: " << data_day_temperature_high_tmp[i];

			data_day_temperature_low_tmp[i] = runCommand( data_day_temperature_low_xml[i], raw_data_day[i] );
			kDebug() << "data_day_temperature_low[" << i << "]: " << data_day_temperature_low_tmp[i];

			data_day_icon_code_tmp[i] = runCommand( data_day_icon_code_xml[i], raw_data_day[i] );
			kDebug() << "data_day_icon_code[" << i << "]: " << data_day_icon_code_tmp[i];

			data_day_icon_tmp[i] = runCommand( data_day_icon_xml[i], raw_data_day[i] );

			tmp_array->clear();
			QString number;
			number.setNum( i+1 );
			downloadFile( data_day_icon_tmp[i], tmp_array );
			QImage icon_tmp = data_day_icon[i];
			if ( !data_day_icon[i].loadFromData( *tmp_array ) ) data_day_icon[i] = icon_tmp;
			kDebug() << "data_day_icon[" << i << "]: " << data_day_icon_tmp[i];

			data_day_icon_text_tmp[i] = runCommand( data_day_icon_text_xml[i], raw_data_day[i] );
			kDebug() << "data_day_icon_text[" << i << "]: " << data_day_icon_text_tmp[i];
		}
		else
		{
			kDebug() << "Download of day[" << i << "] url failed, keeping old values.";

			data_day_name_tmp[i] = data_day_name[i];
			data_day_temperature_high_tmp[i] = data_day_temperature_high[i];
			data_day_temperature_low_tmp[i] = data_day_temperature_low[i];
			data_day_icon_code_tmp[i] = data_day_icon_code[i];
			data_day_icon_text_tmp[i] = data_day_icon_text[i];

			keep_old_value = true;
		}
		if ( stop_thread ) return;
	}

	bool custom_image_success[customImageList.size()];
	QList<QByteArray> data_custom_image_list_array_tmp;
	for ( i=0; i<customImageList.size(); i++ )
	{
		custom_image_success[i] = true;
		if ( !customImageList.at( i ).isLocalFile() )
		{
			QByteArray tmp_image_array;
			custom_image_success[i] = downloadFile( customImageList.at( i ).url(), &tmp_image_array );
			data_custom_image_list_array_tmp.push_back( tmp_image_array );
		}
		else
		{
			data_custom_image_list_array_tmp.push_back( QByteArray() );
		}
		if ( stop_thread ) return;
	}

	mutex.lock();

	if ( !keep_old_value )
	{
		data_location_location = data_location_location_tmp;
		data_location_country = data_location_country_tmp;
		data_sun_sunrise = data_sun_sunrise_tmp;
		data_sun_sunset = data_sun_sunset_tmp;
		data_provider_update_time = data_provider_update_time_tmp;
		data_current_temperature = data_current_temperature_tmp;
		data_current_temperature_felt = data_current_temperature_felt_tmp;
		data_current_wind_code = data_current_wind_code_tmp;
		int j;
		for ( j=0; j<WIND_TRANSFORM_SIZE; j++ )
		{
			if ( QString::compare( data_current_wind_code_tmp, data_wind_transform_in_xml[j], Qt::CaseInsensitive ) == 0 )
				data_current_wind_code = data_wind_transform_out_xml[j];
		}
		data_current_wind_speed = data_current_wind_speed_tmp;
		data_current_wind = data_current_wind_tmp;
		data_current_humidity = data_current_humidity_tmp;
		data_current_icon_code = data_current_icon_code_tmp.toInt();

		for ( j=0; j<WEATHER_CODE_TRANSFORM_SIZE; j++ )
		{
			if ( QString::compare( data_current_icon_code_tmp, data_icon_transform_in_xml[j], Qt::CaseInsensitive ) == 0 )
				data_current_icon_code = data_icon_transform_out_xml[j].toInt();
		}
		data_current_icon_text = data_current_icon_text_tmp;
		data_current_rain = data_current_rain_tmp;
		data_current_dew_point = data_current_dew_point_tmp;
		data_current_visibility = data_current_visibility_tmp;
		data_current_pressure = data_current_pressure_tmp;
		data_current_uv_index = data_current_uv_index_tmp;

		for( i=0; i<7; i++ )
		{
			data_day_name[i] = data_day_name_tmp[i];
			data_day_temperature_high[i] = data_day_temperature_high_tmp[i];
			data_day_temperature_low[i] = data_day_temperature_low_tmp[i];
			data_day_icon_code[i] = data_day_icon_code_tmp[i].toInt();

			for ( j=0; j<WEATHER_CODE_TRANSFORM_SIZE; j++ )
			{
				if ( QString::compare( data_day_icon_code_tmp[i], data_icon_transform_in_xml[j], Qt::CaseInsensitive ) == 0 )
					data_day_icon_code[i] = data_icon_transform_out_xml[j].toInt();
			}

			data_day_icon_text[i] = data_day_icon_text_tmp[i];
		}

		data_custom_image_list.clear();
		for ( i=0; i<customImageList.size(); i++ )
		{
			if ( customImageList.at( i ).isLocalFile() )
			{
				QFile file(customImageList.at( i ).toLocalFile());
				if ( file.open(QIODevice::ReadOnly) )
				{
					data_custom_image_list.append( file.readAll() );
				}
			}
			else
			{
				if ( custom_image_success[i] )
				{
					data_custom_image_list.append( data_custom_image_list_array_tmp[i] );
				}
			}
		}
	}

	delete tmp_array;
	tmp_array = NULL;

	if ( !keep_old_value )
	{
		data_update_time = KGlobal::locale()->formatTime( QTime::currentTime(), false, false );
	}

	// shot, quick and dirty sanity check of downloaded value, if they're empty we better display nothing than that
	if ( !data_current_temperature.contains(QRegExp("[0-9]")) || data_current_temperature.length() > 6 /* too long temperature string? -21.5 has length=5 */ )
	{
		kDebug() << "Current temperature is invalid: Download didn't fail, but (all?) values are obviously invalid.";
		data_update_time = "";
	}

	mutex.unlock();
}

bool Update_Thread::downloadFile( const QString url, QByteArray *data )
{
	kDebug() << "Downloading file from " << url;

	// This is buggy in KDE 4.2 (never reaches time out)
	QByteArray data_tmp;

	qRegisterMetaType<KIO::filesize_t>("KIO::filesize_t");
	qRegisterMetaType<KIO::MetaData>("KIO::MetaData");
	KIO::Job *getJob = KIO::get( KUrl(url), KIO::Reload, KIO::HideProgressInfo );
	if ( KIO::NetAccess::synchronousRun( getJob, 0, &data_tmp ) )
	{
		*data = data_tmp;
		kDebug() << "Download finished successfully.";
		return true;
	}

	kDebug() << "Download finished but failed.";

	return false;
}

QString Update_Thread::runCommand( const QString command, const QByteArray *data )
{
	QString ret = "";

	QProcess process;

	QStringList env = QProcess::systemEnvironment();
	if ( locale != "" ) env << QString("LC_ALL=\"" + locale + "\"") ;
	process.setEnvironment(env);

	process.start( "sh", QStringList() << "-c" << command );
	process.waitForStarted();
	process.write( *data );
	process.closeWriteChannel();
	process.waitForFinished( 5000 );

	QByteArray err;
	err = process.readAllStandardError();
	if ( !err.isEmpty() ) kDebug() << "error running command " << command << " on input data: " << err << endl;

	QByteArray arr;
	arr = process.readAllStandardOutput();

	if ( !arr.isEmpty() )
	{
		if ( encoding == "ascii" ) ret = QString::fromAscii( arr );
		else if ( encoding == "latin1" ) ret = QString::fromLatin1( arr );
		else if ( encoding == "local8bit" ) ret = QString::fromLocal8Bit( arr );
		else if ( encoding == "ucs4" ) ret = QString::fromUtf8( arr );
		else if ( encoding == "utf8" ) ret = QString::fromUtf8( arr );
		else if ( encoding == "utf16" ) ret = QString::fromUtf8( arr );
		else if ( encoding == "latin2" )
		{
			QTextCodec *inp = QTextCodec::codecForName( "ISO8859-2" );
			ret = inp->toUnicode(arr);
		}
		else ret = arr;
	}

	int pos;

	pos = ret.indexOf("\r\n");
	if ( pos != -1 ) ret = ret.left(pos);

	pos = ret.indexOf("\n");
	if ( pos != -1 ) ret = ret.left(pos);

	ret = ret.simplified();

	return ret;
}

QByteArray *Update_Thread::rawDataFromUrl( const QString &url )
{
	QByteArray *ret = NULL;

	if ( url == "urlc" ) ret = raw_data_urlc;
	else if ( url == "url1" ) ret = raw_data_url1;
	else if ( url == "url2" ) ret = raw_data_url2;
	else if ( url == "url3" ) ret = raw_data_url3;
	else if ( url == "url4" ) ret = raw_data_url4;
	else if ( url == "url5" ) ret = raw_data_url5;
	else if ( url == "url6" ) ret = raw_data_url6;
	else if ( url == "url7" ) ret = raw_data_url7;

	return ret;
}

bool Update_Thread::successFromUrl( const QString &url, const bool *url_success )
{
	bool ret = false;

	if ( url == "urlc" ) ret = url_success[0];
	else if ( url == "url1" ) ret = url_success[1];
	else if ( url == "url2" ) ret = url_success[2];
	else if ( url == "url3" ) ret = url_success[3];
	else if ( url == "url4" ) ret = url_success[4];
	else if ( url == "url5" ) ret = url_success[5];
	else if ( url == "url6" ) ret = url_success[6];
	else if ( url == "url7" ) ret = url_success[7];

	return ret;
}

#include "plasma-cwp-update-thread.moc"
