/***********************************************************************************
* Adjustable Clock: Plasmoid to show date and time in adjustable format.
* Copyright (C) 2008 - 2009 Michal Dutkiewicz aka Emdek <emdeck@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#define DEFAULT_FORMAT "<div style=\"text-align:center; margin:5px; white-space:pre;\"><big>%H:%M:%S</big>\n<small>%d.%m.%Y</small></div>"

#include "AdjustableClock.h"

#include <QRegExp>
#include <QWebFrame>
#include <QClipboard>
#include <QHeaderView>
#include <QColorDialog>
#include <QWebSettings>

#include <KLocale>
#include <KTimeZone>
#include <KMessageBox>
#include <KInputDialog>
#include <KConfigDialog>
#include <KCalendarSystem>
#include <KSystemTimeZones>

#include <Plasma/Theme>

K_EXPORT_PLASMA_APPLET(adjustableclock, AdjustableClock)

AdjustableClock::AdjustableClock(QObject *parent, const QVariantList &args) : ClockApplet(parent, args),
    m_controlsTimer(new QTimer(this)),
    m_toolTipVisible(false),
    m_predefinedFormat(false)
{
    KGlobal::locale()->insertCatalog("libplasmaclock");
    KGlobal::locale()->insertCatalog("timezones4");
    KGlobal::locale()->insertCatalog("adjustableclock");

    setHasConfigurationInterface(true);

    m_controlsTimer->setInterval(250);
    m_controlsTimer->setSingleShot(true);

    m_timeFormatStrings.append(DEFAULT_FORMAT);
    m_timeFormatStrings.append("<div style=\"text-align:center; margin:5px; white-space:pre;\"><big style=\"font-family:'Nimbus Sans L Condensed';\">%H:%M:%S</big>\n<span style=\"font-size:small; font-family:'Nimbus Sans L';\">%d.%m.%Y</small></div>");
    m_timeFormatStrings.append("<div style=\"text-align:center; white-space:pre; font-size:25px; margin:5px;\">%H:%M</div>");
    m_timeFormatStrings.append("<div style=\"text-align:center; white-space:pre; font-size:25px; margin:0 0 5px 5px;\">%H:%M<span style=\"font-size:30px; position:relative; left:-8px; top:4px; z-index:-1; opacity:0.5;\">%S</span></div>");
    m_timeFormatStrings.append("<div style=\"text-align:center; white-space:pre; font-size:25px; margin:-15px 0 5px 5px; -webkit-box-reflect:below -5px -webkit-gradient(linear, left top, left bottom, from(transparent), color-stop(0.5, transparent), to(white));\">%H:%M<span style=\"font-size:30px; position:relative; left:-8px; top:4px; z-index:-1; opacity:0.5;\">%S</span></div>");

    m_timeFormatNames.append(i18n("Default"));
    m_timeFormatNames.append(i18n("Flat"));
    m_timeFormatNames.append(i18n("Simple"));
    m_timeFormatNames.append(i18n("dbClock"));
    m_timeFormatNames.append(i18n("dbClock with reflection"));
}

void AdjustableClock::init()
{
    KConfigGroup configuration = config();

    ClockApplet::init();

    m_clipboardFormats.append("%x");
    m_clipboardFormats.append("%f");
    m_clipboardFormats.append("%H:%M:%S");
    m_clipboardFormats.append(QString());
    m_clipboardFormats.append("%X");
    m_clipboardFormats.append("%F");
    m_clipboardFormats.append(QString());
    m_clipboardFormats.append("%c");
    m_clipboardFormats.append("%C");
    m_clipboardFormats.append("%Y-%m-%d %H:%M:%S");
    m_clipboardFormats.append(QString());
    m_clipboardFormats.append("%t");

    m_timeFormat = configuration.readEntry("timeFormat", DEFAULT_FORMAT);
    m_toolTipFormat = configuration.readEntry("toolTipFormat", "%Y-%m-%d %H:%M:%S");
    m_fastCopyFormat = configuration.readEntry("fastCopyFormat", "%Y-%m-%d %H:%M:%S");
    m_timeDifference = configuration.readEntry("timeDifference", 0);
    m_rescaleContents = configuration.readEntry("rescaleContents", false);
    m_paintBackground = configuration.readEntry("paintBackground", true);
    m_clipboardFormats = configuration.readEntry("clipboardFormats", m_clipboardFormats);

    m_page.mainFrame()->setScrollBarPolicy(Qt::Horizontal, Qt::ScrollBarAlwaysOff);
    m_page.mainFrame()->setScrollBarPolicy(Qt::Vertical, Qt::ScrollBarAlwaysOff);
    m_page.mainFrame()->setHtml("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\"><html><head><style type=\"text/css\">html, body, table, td {margin:0; padding:0; height:100%; width:100%; vertical-align:middle;}</style></head><body><table><tr><td id=\"clock\">" + formatDateTime(currentDateTime(), m_timeFormat) + "</td></tr></table></body></html>");

    updateTheme();

    connectSource(currentTimezone());

    setText(formatDateTime(currentDateTime(), m_timeFormat));

    m_clipboardMenu = new KMenu;

    m_clipboardAction = new QAction(SmallIcon("edit-copy"), i18n("C&opy to Clipboard"), this);
    m_clipboardAction->setMenu(m_clipboardMenu);
    m_clipboardAction->setVisible(!m_clipboardFormats.isEmpty());

    updateClipboardMenu();

    constraintsEvent(Plasma::SizeConstraint);

    connect(this, SIGNAL(activate()), this, SLOT(copyToClipboard()));
    connect(this, SIGNAL(destroyed()), m_clipboardMenu, SLOT(deleteLater()));
    connect(m_controlsTimer, SIGNAL(timeout()), this, SLOT(updateControls()));
    connect(m_clipboardMenu, SIGNAL(aboutToShow()), this, SLOT(updateClipboardMenu()));
    connect(m_clipboardMenu, SIGNAL(triggered(QAction*)), this, SLOT(copyToClipboard(QAction*)));
    connect(Plasma::Theme::defaultTheme(), SIGNAL(themeChanged()), this, SLOT(updateTheme()));
}

void AdjustableClock::dataUpdated(const QString &source, const Plasma::DataEngine::Data &data)
{
    Q_UNUSED(source)

    m_dateTime = QDateTime(data["Date"].toDate(), data["Time"].toTime());
    m_sunrise = data["Sunrise"].toTime();
    m_sunset = data["Sunset"].toTime();

    if (m_timeDifference)
    {
        m_dateTime = m_dateTime.addSecs(m_timeDifference);
        m_sunrise = m_sunrise.addSecs(m_timeDifference);
        m_sunset = m_sunset.addSecs(m_timeDifference);
    }

    setText(formatDateTime(m_dateTime, m_timeFormat));

    if (m_toolTipVisible)
    {
        updateToolTipContent();
    }
}

void AdjustableClock::constraintsEvent(Plasma::Constraints constraints)
{
    if (constraints & Plasma::SizeConstraint)
    {
        updateSize();
    }

    setBackgroundHints(m_paintBackground?DefaultBackground:NoBackground);
}

void AdjustableClock::paintInterface(QPainter *painter, const QStyleOptionGraphicsItem *option, const QRect &contentsRect)
{
    Q_UNUSED(option)
    Q_UNUSED(contentsRect)

    m_page.mainFrame()->render(painter);
}

void AdjustableClock::createClockConfigurationInterface(KConfigDialog *parent)
{
    KConfigGroup configuration = config();
    QStringList timeFormatStrings;
    QStringList timeFormatNames;
    QString preview;
    int row;

    QWidget *appearanceConfiguration = new QWidget;
    m_appearanceUi.setupUi(appearanceConfiguration);

    QWidget *clipboardActions = new QWidget;
    m_clipboardUi.setupUi(clipboardActions);

    QWidget *advancedConfiguration = new QWidget;
    m_advancedUi.setupUi(advancedConfiguration);

    KMenu *placeholdersMenu = new KMenu(m_appearanceUi.placeholdersButton);

    QMap<QChar, QString> placeholders;

    placeholders['H'] = i18n("The hour in 24h format with two digits");
    placeholders['k'] = i18n("The hour in 24h format");
    placeholders['I'] = i18n("The hour in 12h format with two digits");
    placeholders['l'] = i18n("The hour in 12h format");
    placeholders['M'] = i18n("The minute number with two digits");
    placeholders['S'] = i18n("The seconds number with two digits");
    placeholders['s'] = i18n("The seconds number");
    placeholders['p'] = i18n("The pm or am string");
    placeholders['Y'] = i18n("The whole year");
    placeholders['y'] = i18n("The lower two digits of the year");
    placeholders['n'] = i18n("The month number");
    placeholders['m'] = i18n("The month number with two digits");
    placeholders['e'] = i18n("The day of the month number");
    placeholders['d'] = i18n("The day of the month number with two digits");
    placeholders['b'] = i18n("The short form of the month name");
    placeholders['B'] = i18n("The long form of the month name");
    placeholders['a'] = i18n("The short form of the weekday name");
    placeholders['A'] = i18n("The long form of the weekday name");
    placeholders['g'] = i18n("The timezone city string");
    placeholders['z'] = i18n("The timezone offset to UTC");
    placeholders['Z'] = i18n("The timezone abbreviation string");
    placeholders['j'] = i18n("The day of year number");
    placeholders['w'] = i18n("The weekday number");
    placeholders['W'] = i18n("The week number");
    placeholders['c'] = i18n("The default short date and time format");
    placeholders['C'] = i18n("The default long date and time format");
    placeholders['x'] = i18n("The default long date format");
    placeholders['X'] = i18n("The default long time format");
    placeholders['f'] = i18n("The default short date format");
    placeholders['F'] = i18n("The default short time format");
    placeholders['t'] = i18n("The UNIX timestamp");
    placeholders['o'] = i18n("The sunrise time");
    placeholders['O'] = i18n("The sunset time");

    QMap<QChar, QString>::const_iterator iterator = placeholders.constBegin();

    while (iterator != placeholders.constEnd())
    {
        QAction *action = placeholdersMenu->addAction(QString("%1 (%%2)\t%3").arg(iterator.value()).arg(iterator.key()).arg(formatDateTime(m_dateTime, (QString('%').append(iterator.key())))));
        action->setData(iterator.key());

        ++iterator;
    }

    timeFormatStrings = m_timeFormatStrings;
    timeFormatStrings.append(configuration.readEntry("timeFormatStrings", QStringList()));

    timeFormatNames = m_timeFormatNames;
    timeFormatNames.append(configuration.readEntry("timeFormatNames", QStringList()));

    for (int i = 0; i < timeFormatStrings.count(); ++i)
    {
        if (i == m_timeFormatStrings.count())
        {
            m_appearanceUi.timeFormatComboBox->insertSeparator(i);
        }

        m_appearanceUi.timeFormatComboBox->addItem(timeFormatNames.at(i), timeFormatStrings.at(i));
    }

    m_appearanceUi.addButton->setIcon(KIcon("list-add"));
    m_appearanceUi.removeButton->setIcon(KIcon("list-remove"));
    m_appearanceUi.placeholdersButton->setIcon(KIcon("chronometer"));
    m_appearanceUi.placeholdersButton->setMenu(placeholdersMenu);
    m_appearanceUi.boldButton->setIcon(KIcon("format-text-bold"));
    m_appearanceUi.italicButton->setIcon(KIcon("format-text-italic"));
    m_appearanceUi.underlineButton->setIcon(KIcon("format-text-underline"));
    m_appearanceUi.justifyLeftButton->setIcon(KIcon("format-justify-left"));
    m_appearanceUi.justifyCenterButton->setIcon(KIcon("format-justify-center"));
    m_appearanceUi.justifyRightButton->setIcon(KIcon("format-justify-right"));

    m_appearanceUi.fontSizeComboBox->addItem("9");
    m_appearanceUi.fontSizeComboBox->addItem("11");
    m_appearanceUi.fontSizeComboBox->addItem("14");
    m_appearanceUi.fontSizeComboBox->addItem("17");
    m_appearanceUi.fontSizeComboBox->addItem("21");
    m_appearanceUi.fontSizeComboBox->addItem("28");
    m_appearanceUi.fontSizeComboBox->addItem("42");

    QPalette palette = m_appearanceUi.webView->page()->palette();
    palette.setBrush(QPalette::Base, Qt::transparent);

    m_appearanceUi.webView->setAttribute(Qt::WA_OpaquePaintEvent, false);
    m_appearanceUi.webView->page()->setPalette(palette);
    m_appearanceUi.webView->page()->setContentEditable(true);

    m_advancedUi.toolTipFormat->setPlainText(m_toolTipFormat);
    m_advancedUi.fastCopyFormat->setText(m_fastCopyFormat);
    m_advancedUi.timeDifference->setValue(m_timeDifference);
    m_advancedUi.rescaleContentsCheckBox->setChecked(m_rescaleContents);
    m_advancedUi.paintBackgroundCheckBox->setChecked(m_paintBackground);

    m_clipboardUi.clipboardActionsTable->setAlternatingRowColors(true);
    m_clipboardUi.clipboardActionsTable->setSelectionMode(QAbstractItemView::SingleSelection);
    m_clipboardUi.clipboardActionsTable->setColumnCount(2);
    m_clipboardUi.clipboardActionsTable->setHorizontalHeaderItem(0, new QTableWidgetItem(i18n("Format")));
    m_clipboardUi.clipboardActionsTable->setHorizontalHeaderItem(1, new QTableWidgetItem(i18n("Preview")));
    m_clipboardUi.clipboardActionsTable->setColumnWidth(0, 150);

    QHeaderView *header = m_clipboardUi.clipboardActionsTable->horizontalHeader();
    header->setResizeMode(QHeaderView::Stretch);
    header->resizeSection(1, 150);

    m_clipboardUi.moveUpButton->setIcon(KIcon("arrow-up"));
    m_clipboardUi.moveDownButton->setIcon(KIcon("arrow-down"));

    for (int i = 0; i < m_clipboardFormats.count(); ++i)
    {
        row = m_clipboardUi.clipboardActionsTable->rowCount();
        m_clipboardUi.clipboardActionsTable->insertRow(row);
        m_clipboardUi.clipboardActionsTable->setItem(row, 0, new QTableWidgetItem(m_clipboardFormats.at(i)));

        preview = formatDateTime(m_dateTime, m_clipboardFormats.at(i));

        QTableWidgetItem *item = new QTableWidgetItem(preview);
        item->setFlags(0);
        item->setToolTip(preview);

        m_clipboardUi.clipboardActionsTable->setItem(row, 1, item);
    }

    updateControls();

    itemSelectionChanged();

    parent->addPage(appearanceConfiguration, i18n("Appearance"), "preferences-desktop-theme");
    parent->addPage(clipboardActions, i18n("Clipboard actions"), "edit-copy");
    parent->addPage(advancedConfiguration, i18n("Advanced"), "configure");

    connect(parent, SIGNAL(finished()), this, SLOT(resetTimeFormat()));
    connect(placeholdersMenu, SIGNAL(triggered(QAction*)), this, SLOT(insertPlaceholder(QAction*)));
    connect(m_appearanceUi.timeFormatComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(loadItem(int)));
    connect(m_appearanceUi.addButton, SIGNAL(clicked()), this, SLOT(addItem()));
    connect(m_appearanceUi.removeButton, SIGNAL(clicked()), this, SLOT(removeItem()));
    connect(m_appearanceUi.webView->page(), SIGNAL(selectionChanged()), this, SLOT(selectionChanged()));
    connect(m_appearanceUi.webView->page(), SIGNAL(contentsChanged()), this, SLOT(contentsChanged()));
    connect(m_appearanceUi.timeFormat, SIGNAL(textChanged()), this, SLOT(formatChanged()));
    connect(m_appearanceUi.boldButton, SIGNAL(clicked()), this, SLOT(toggleState()));
    connect(m_appearanceUi.italicButton, SIGNAL(clicked()), this, SLOT(toggleState()));
    connect(m_appearanceUi.underlineButton, SIGNAL(clicked()), this, SLOT(toggleState()));
    connect(m_appearanceUi.justifyLeftButton, SIGNAL(clicked()), this, SLOT(toggleState()));
    connect(m_appearanceUi.justifyCenterButton, SIGNAL(clicked()), this, SLOT(toggleState()));
    connect(m_appearanceUi.justifyRightButton, SIGNAL(clicked()), this, SLOT(toggleState()));
    connect(m_appearanceUi.colorButton, SIGNAL(clicked()), this, SLOT(selectColor()));
    connect(m_appearanceUi.fontSizeComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(selectFontSize(int)));
    connect(m_appearanceUi.fontFamilyComboBox, SIGNAL(currentFontChanged(QFont)), this, SLOT(selectFontFamily(QFont)));
    connect(m_clipboardUi.addButton, SIGNAL(clicked()), this, SLOT(insertRow()));
    connect(m_clipboardUi.deleteButton, SIGNAL(clicked()), this, SLOT(deleteRow()));
    connect(m_clipboardUi.moveUpButton, SIGNAL(clicked()), this, SLOT(moveRowUp()));
    connect(m_clipboardUi.moveDownButton, SIGNAL(clicked()), this, SLOT(moveRowDown()));
    connect(m_clipboardUi.clipboardActionsTable, SIGNAL(itemSelectionChanged()), this, SLOT(itemSelectionChanged()));
    connect(m_clipboardUi.clipboardActionsTable, SIGNAL(cellChanged(int, int)), this, SLOT(updateRow(int, int)));

    QString timeFormat = configuration.readEntry("timeFormat", DEFAULT_FORMAT);
    int currentFormat = m_appearanceUi.timeFormatComboBox->findData(timeFormat);

    if (currentFormat < 0)
    {
        m_appearanceUi.timeFormatComboBox->insertItem(0, i18n("Custom"), timeFormat);

        currentFormat = 0;
    }

    m_appearanceUi.timeFormatComboBox->setCurrentIndex(currentFormat);

    loadItem(currentFormat);
}

void AdjustableClock::configAccepted()
{
    KConfigGroup configuration = config();
    QStringList timeFormatStrings;
    QStringList timeFormatNames;

    dataEngine("time")->disconnectSource((currentTimezone() + "|Solar"), this);

    for (int i = 0; i < m_appearanceUi.timeFormatComboBox->count(); ++i)
    {
        if (m_timeFormatNames.contains(m_appearanceUi.timeFormatComboBox->itemText(i)) || m_appearanceUi.timeFormatComboBox->itemText(i).isEmpty())
        {
            continue;
        }

        timeFormatStrings.append(m_appearanceUi.timeFormatComboBox->itemData(i).toString());

        timeFormatNames.append(m_appearanceUi.timeFormatComboBox->itemText(i));
    }

    m_timeFormat = m_appearanceUi.timeFormat->toPlainText();
    m_toolTipFormat = m_advancedUi.toolTipFormat->toPlainText();
    m_fastCopyFormat = m_advancedUi.fastCopyFormat->text();
    m_timeDifference = m_advancedUi.timeDifference->value();
    m_rescaleContents = m_advancedUi.rescaleContentsCheckBox->isChecked();
    m_paintBackground = m_advancedUi.paintBackgroundCheckBox->isChecked();

    configuration.writeEntry("timeFormatStrings", timeFormatStrings);
    configuration.writeEntry("timeFormatNames", timeFormatNames);
    configuration.writeEntry("timeFormat", m_timeFormat);
    configuration.writeEntry("toolTipFormat", m_toolTipFormat);
    configuration.writeEntry("fastCopyFormat", m_fastCopyFormat);
    configuration.writeEntry("timeDifference", m_timeDifference);
    configuration.writeEntry("rescaleContents", m_rescaleContents);
    configuration.writeEntry("paintBackground", m_paintBackground);

    m_clipboardFormats.clear();

    int rowCount = m_clipboardUi.clipboardActionsTable->rowCount();

    for (int i = 0; i < rowCount; ++i)
    {
        m_clipboardFormats.append(m_clipboardUi.clipboardActionsTable->item(i, 0)->text());
    }

    configuration.writeEntry("clipboardFormats", m_clipboardFormats);

    m_clipboardAction->setVisible(!m_clipboardFormats.isEmpty());

    ClockApplet::configAccepted();

    connectSource(currentTimezone());

    m_controlsTimer->stop();

    emit configNeedsSaving();
}

void AdjustableClock::connectSource(const QString &timezone)
{
    QRegExp formatWithSeconds = QRegExp("%(S|c|C|t|F|X)");
    const bool alignToSeconds = (m_timeFormat.contains(formatWithSeconds) || m_toolTipFormat.contains(formatWithSeconds));

    dataEngine("time")->connectSource((timezone + "|Solar"), this, (alignToSeconds?1000:60000), (alignToSeconds?Plasma::NoAlignment:Plasma::AlignToMinute));

    m_timeZoneAbbreviation = KSystemTimeZones::zone(timezone).abbreviation(QDateTime::currentDateTime().toUTC());

    if (m_timeZoneAbbreviation.isEmpty())
    {
        m_timeZoneAbbreviation = i18n("UTC");
    }

    int seconds = KSystemTimeZones::zone(currentTimezone()).currentOffset();
    int minutes = abs(seconds / 60);
    int hours = abs(minutes / 60);

    minutes = (minutes - (hours * 60));

    m_timeZoneOffset = QString::number(hours);

    if (minutes)
    {
        m_timeZoneOffset.append(':');

        if (minutes < 10)
        {
            m_timeZoneOffset.append('0');
        }

        m_timeZoneOffset.append(QString::number(minutes));
    }

    m_timeZoneOffset = (QChar((seconds >= 0)?'+':'-') + m_timeZoneOffset);

    updateSize();
}

void AdjustableClock::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    if (event->buttons() == Qt::MidButton)
    {
        copyToClipboard();
    }

    ClockApplet::mousePressEvent(event);
}

void AdjustableClock::copyToClipboard()
{
    QApplication::clipboard()->setText(formatDateTime(currentDateTime(), m_fastCopyFormat));
}

void AdjustableClock::insertPlaceholder(QAction *action)
{
    QString placeholder = QString('%').append(action->data().toChar());

    if (m_appearanceUi.tabWidget->currentIndex() > 0)
    {
        m_appearanceUi.timeFormat->insertPlainText(placeholder);
    }
    else
    {
        m_appearanceUi.webView->page()->mainFrame()->evaluateJavaScript("document.execCommand('inserthtml', false, '" + placeholder + "')");
    }
}

void AdjustableClock::loadItem(int index)
{
    m_predefinedFormat = false;

    m_appearanceUi.timeFormat->setPlainText(m_appearanceUi.timeFormatComboBox->itemData(index).toString());

    m_predefinedFormat = m_timeFormatNames.contains(m_appearanceUi.timeFormatComboBox->itemText(index));

    m_appearanceUi.removeButton->setEnabled(!m_predefinedFormat);
}

void AdjustableClock::addItem(bool automatically)
{
    QString formatName = m_appearanceUi.timeFormatComboBox->itemText(m_appearanceUi.timeFormatComboBox->currentIndex());

    if (automatically)
    {
        int i = 2;

        while (m_appearanceUi.timeFormatComboBox->findText(QString("%1 %2").arg(formatName).arg(i)) >= 0)
        {
            ++i;
        }

        formatName = QString("%1 %2").arg(formatName).arg(i);
    }
    else
    {
        formatName = KInputDialog::getText(i18n("Add new format"), i18n("Format name:"), formatName);
    }

    if (m_appearanceUi.timeFormatComboBox->findText(formatName) >= 0)
    {
        KMessageBox::error(m_appearanceUi.timeFormatComboBox, i18n("Format with this name already exists!"));
    }
    else if (!formatName.isEmpty())
    {
        int index = (m_appearanceUi.timeFormatComboBox->currentIndex() + 1);

        if (index <= m_timeFormatNames.count())
        {
            index = m_appearanceUi.timeFormatComboBox->count();
        }

        m_predefinedFormat = false;

        disconnect(m_appearanceUi.timeFormatComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(loadItem(int)));

        m_appearanceUi.timeFormatComboBox->insertItem(index, formatName, m_appearanceUi.timeFormat->toPlainText());
        m_appearanceUi.timeFormatComboBox->setCurrentIndex(index);
        m_appearanceUi.removeButton->setEnabled(true);

        connect(m_appearanceUi.timeFormatComboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(loadItem(int)));
    }
}

void AdjustableClock::removeItem()
{
    m_appearanceUi.timeFormatComboBox->removeItem(m_appearanceUi.timeFormatComboBox->currentIndex());

    if (!m_appearanceUi.timeFormatComboBox->count())
    {
        m_appearanceUi.timeFormatComboBox->addItem(i18n("Default"), DEFAULT_FORMAT);
        m_appearanceUi.timeFormatComboBox->setCurrentIndex(0);
    }
}

void AdjustableClock::updateControls()
{
    m_appearanceUi.webView->page()->mainFrame()->evaluateJavaScript("boldButton.setChecked(document.queryCommandState('bold'));"
        "italicButton.setChecked(document.queryCommandState('italic'));"
        "underlineButton.setChecked(document.queryCommandState('underline'));"
        "designModeEditor.setColor(document.queryCommandValue('forecolor'));"
        "designModeEditor.setFontSize((document.queryCommandValue('fontsize') == '9px')?9:document.queryCommandValue('fontsize').substr(0, 2));"
        "designModeEditor.setFontFamily(document.queryCommandValue('fontname'))");
}

void AdjustableClock::toggleState()
{
    m_appearanceUi.webView->page()->mainFrame()->evaluateJavaScript("document.execCommand('" + sender()->objectName().remove("Button").toLower() + "')");
}

void AdjustableClock::selectColor()
{
    QColorDialog colorDialog;
    colorDialog.setOptions(QColorDialog::ShowAlphaChannel);
    colorDialog.setCurrentColor(m_appearanceUi.colorButton->palette().button().color());

    if (colorDialog.exec() == QDialog::Accepted)
    {
        QColor color = colorDialog.selectedColor();

        m_appearanceUi.webView->page()->mainFrame()->evaluateJavaScript("document.execCommand('forecolor', false, '" + color.name() + "')");

        QPalette palette = m_appearanceUi.colorButton->palette();
        palette.setBrush(QPalette::Button, color);

        m_appearanceUi.colorButton->setPalette(palette);
    }
}

void AdjustableClock::selectFontSize(int size)
{
    m_appearanceUi.webView->page()->mainFrame()->evaluateJavaScript("document.execCommand('fontsize', false, '" + QString::number(size + 1) + "')");
}

void AdjustableClock::selectFontFamily(QFont font)
{
    m_appearanceUi.webView->page()->mainFrame()->evaluateJavaScript("document.execCommand('fontname', false, '" + font.family() + "')");
}

void AdjustableClock::setColor(QString color)
{
    if (color != "false")
    {
        QRegExp expression = QRegExp("rgb\\((\\d+), (\\d+), (\\d+)\\)");
        expression.indexIn(color);

        QStringList rgb = expression.capturedTexts();

        QPalette palette = m_appearanceUi.colorButton->palette();
        palette.setBrush(QPalette::Button, QColor(rgb.at(1).toInt(), rgb.at(2).toInt(), rgb.at(3).toInt()));

        m_appearanceUi.colorButton->setPalette(palette);
    }
}

void AdjustableClock::setFontSize(QString size)
{
    m_appearanceUi.fontSizeComboBox->setCurrentIndex(m_appearanceUi.fontSizeComboBox->findText(size));
}

void AdjustableClock::setFontFamily(QString font)
{
    m_appearanceUi.fontFamilyComboBox->setCurrentFont(QFont(font));
}

void AdjustableClock::contentsChanged()
{
    QString text = m_appearanceUi.webView->page()->mainFrame()->toHtml().remove("<html><body>").remove("</body></html>");
    QRegExp fontSize = QRegExp(" class=\"Apple-style-span\"");
    QRegExp fontColor = QRegExp("<font color=\"(#?[\\w\\s]+)\">(.+)</font>");
    QRegExp fontFamily = QRegExp("<font face=\"'?([\\w\\s]+)'?\">(.+)</font>");

    fontColor.setMinimal(true);

    fontFamily.setMinimal(true);

    text.remove(fontSize).replace(fontColor, "<span style=\"color:\\1;\">\\2</span>").replace(fontFamily, "<span style=\"font-family:'\\1';\">\\2</span>");

    disconnect(m_appearanceUi.timeFormat, SIGNAL(textChanged()), this, SLOT(formatChanged()));

    QString toolTip = formatDateTime(m_dateTime, text, false);

    m_appearanceUi.timeFormat->setToolTip(toolTip);
    m_appearanceUi.webView->setToolTip(toolTip);
    m_appearanceUi.timeFormat->setPlainText(text);

    if (m_predefinedFormat)
    {
        addItem(true);
    }

    m_appearanceUi.timeFormatComboBox->setItemData(m_appearanceUi.timeFormatComboBox->currentIndex(), text);

    connect(m_appearanceUi.timeFormat, SIGNAL(textChanged()), this, SLOT(formatChanged()));
}

void AdjustableClock::formatChanged()
{
    if (m_appearanceUi.timeFormat->toPlainText() == m_appearanceUi.webView->page()->mainFrame()->toHtml())
    {
        return;
    }

    disconnect(m_appearanceUi.webView->page(), SIGNAL(contentsChanged()));

    QString toolTip = formatDateTime(m_dateTime, m_appearanceUi.timeFormat->toPlainText(), false);

    m_appearanceUi.timeFormat->setToolTip(toolTip);
    m_appearanceUi.webView->setToolTip(toolTip);
    m_appearanceUi.webView->page()->mainFrame()->setHtml(m_appearanceUi.timeFormat->toPlainText());
    m_appearanceUi.webView->page()->mainFrame()->addToJavaScriptWindowObject("boldButton", m_appearanceUi.boldButton);
    m_appearanceUi.webView->page()->mainFrame()->addToJavaScriptWindowObject("italicButton", m_appearanceUi.italicButton);
    m_appearanceUi.webView->page()->mainFrame()->addToJavaScriptWindowObject("underlineButton", m_appearanceUi.underlineButton);
    m_appearanceUi.webView->page()->mainFrame()->addToJavaScriptWindowObject("designModeEditor", this);

    if (m_predefinedFormat)
    {
        addItem(true);
    }

    m_appearanceUi.timeFormatComboBox->setItemData(m_appearanceUi.timeFormatComboBox->currentIndex(), m_appearanceUi.timeFormat->toPlainText());

    connect(m_appearanceUi.webView->page(), SIGNAL(contentsChanged()), this, SLOT(contentsChanged()));
}

void AdjustableClock::selectionChanged()
{
    m_controlsTimer->start();
}

void AdjustableClock::itemSelectionChanged()
{
    QList<QTableWidgetItem*> selectedItems = m_clipboardUi.clipboardActionsTable->selectedItems();

    m_clipboardUi.moveUpButton->setEnabled(!selectedItems.isEmpty() && m_clipboardUi.clipboardActionsTable->row(selectedItems.first()) != 0);
    m_clipboardUi.moveDownButton->setEnabled(!selectedItems.isEmpty() && m_clipboardUi.clipboardActionsTable->row(selectedItems.last()) != (m_clipboardUi.clipboardActionsTable->rowCount() - 1));
    m_clipboardUi.deleteButton->setEnabled(!selectedItems.isEmpty());
}

void AdjustableClock::insertRow()
{
    const int row = ((m_clipboardUi.clipboardActionsTable->rowCount() && m_clipboardUi.clipboardActionsTable->currentRow() >= 0)?m_clipboardUi.clipboardActionsTable->currentRow():0);

    m_clipboardUi.clipboardActionsTable->insertRow(row);
    m_clipboardUi.clipboardActionsTable->setItem(row, 0, new QTableWidgetItem(QString()));

    QTableWidgetItem *item = new QTableWidgetItem(QString());
    item->setFlags(0);

    m_clipboardUi.clipboardActionsTable->setItem(row, 1, item);
    m_clipboardUi.clipboardActionsTable->setCurrentCell(row, 0);
}

void AdjustableClock::deleteRow()
{
    m_clipboardUi.clipboardActionsTable->removeRow(m_clipboardUi.clipboardActionsTable->row(m_clipboardUi.clipboardActionsTable->selectedItems().at(0)));
}

void AdjustableClock::moveRow(bool up)
{
    int sourceRow = m_clipboardUi.clipboardActionsTable->row(m_clipboardUi.clipboardActionsTable->selectedItems().at(0));
    int destinationRow = (up?(sourceRow - 1):(sourceRow + 1));

    QList<QTableWidgetItem*> sourceItems;
    QList<QTableWidgetItem*> destinationItems;

    for (int i = 0; i < 2; ++i)
    {
        sourceItems.append(m_clipboardUi.clipboardActionsTable->takeItem(sourceRow, i));

        destinationItems.append(m_clipboardUi.clipboardActionsTable->takeItem(destinationRow, i));
    }

    for (int i = 0; i < 2; ++i)
    {
        m_clipboardUi.clipboardActionsTable->setItem(sourceRow, i, destinationItems.at(i));
        m_clipboardUi.clipboardActionsTable->setItem(destinationRow, i, sourceItems.at(i));
    }

    m_clipboardUi.clipboardActionsTable->setCurrentCell(destinationRow, 0);
}

void AdjustableClock::moveRowUp()
{
    moveRow(true);
}

void AdjustableClock::moveRowDown()
{
    moveRow(false);
}

void AdjustableClock::updateRow(int row, int column)
{
    Q_UNUSED(column)

    if (!m_clipboardUi.clipboardActionsTable->item(row, 1))
    {
        return;
    }

    const QString preview = formatDateTime(m_dateTime, m_clipboardUi.clipboardActionsTable->item(row, 0)->text());

    m_clipboardUi.clipboardActionsTable->item(row, 1)->setText(preview);
    m_clipboardUi.clipboardActionsTable->item(row, 1)->setToolTip(preview);
}

void AdjustableClock::toolTipAboutToShow()
{
    updateToolTipContent();

    m_toolTipVisible = true;
}

void AdjustableClock::toolTipHidden()
{
    Plasma::ToolTipManager::self()->clearContent(this);

    m_toolTipVisible = false;
}

void AdjustableClock::setText(QString text)
{
    m_page.mainFrame()->evaluateJavaScript("document.getElementById('clock').innerHTML = '" + text + '\'');

    update();
}

void AdjustableClock::copyToClipboard(QAction* action)
{
    QApplication::clipboard()->setText(action->text());
}

void AdjustableClock::updateClipboardMenu()
{
    qDeleteAll(m_clipboardMenu->actions());

    m_clipboardMenu->clear();
    m_clipboardMenu->addActions(clipboardActions());
}

void AdjustableClock::changeEngineTimezone(const QString &oldTimezone, const QString &newTimezone)
{
    dataEngine("time")->disconnectSource((oldTimezone + "|Solar"), this);

    connectSource(newTimezone);
}

void AdjustableClock::updateToolTipContent()
{
    Plasma::ToolTipContent toolTipData;

    if (!m_toolTipFormat.isEmpty())
    {
        toolTipData.setImage(KIcon("chronometer").pixmap(IconSize(KIconLoader::Desktop)));
        toolTipData.setMainText(formatDateTime(m_dateTime, m_toolTipFormat, false));
        toolTipData.setAutohide(false);
    }

    Plasma::ToolTipManager::self()->setContent(this, toolTipData);
}

void AdjustableClock::updateSize()
{
    QString string;
    QString longest;
    QString temporary;
    int placeholder;
    const int length = (m_timeFormat.length() - 1);
    int number = 0;

    for (int index = 0; index < length; ++index)
    {
        if (m_timeFormat.at(index) == '%' && m_timeFormat.at(index + 1) != '%')
        {
            ++index;

            placeholder = m_timeFormat.at(index).unicode();

            longest.clear();

            switch (placeholder)
            {
                case 'a':
                case 'A':
                    number = KGlobal::locale()->calendar()->daysInWeek(m_dateTime.date());

                    for (int i = 0; i <= number; ++i)
                    {
                        temporary = KGlobal::locale()->calendar()->weekDayName(i, ((placeholder == 'a')?KCalendarSystem::ShortDayName:KCalendarSystem::LongDayName));

                        if (temporary.length() > longest.length())
                        {
                            longest = temporary;
                        }
                    }

                    string.append(longest);
                break;
                case 'b':
                case 'B':
                    number = KGlobal::locale()->calendar()->monthsInYear(m_dateTime.date());

                    for (int i = 0; i < number; ++i)
                    {
                        temporary = KGlobal::locale()->calendar()->monthName(i, KGlobal::locale()->calendar()->year(m_dateTime.date()), ((KGlobal::locale()->nounDeclension() && KGlobal::locale()->dateMonthNamePossessive())?((placeholder == 'b')?KCalendarSystem::ShortNamePossessive:KCalendarSystem::LongNamePossessive):((placeholder == 'b')?KCalendarSystem::ShortName:KCalendarSystem::LongName)));

                        if (temporary.length() > longest.length())
                        {
                            longest = temporary;
                        }
                    }

                    string.append(longest);
                break;
                case 'c':
                    string.append(KGlobal::locale()->formatDateTime(m_dateTime, KLocale::LongDate));
                break;
                case 'C':
                    string.append(KGlobal::locale()->formatDateTime(m_dateTime, KLocale::ShortDate));
                break;
                case 'd':
                case 'e':
                case 'H':
                case 'I':
                case 'k':
                case 'l':
                case 'm':
                case 'M':
                case 'n':
                case 'S':
                case 'W':
                case 'U':
                case 'y':
                    string.append("00");
                break;
                case 'f':
                    string.append(KGlobal::locale()->formatDate(m_dateTime.date(), KLocale::ShortDate));
                break;
                case 'x':
                    string.append(KGlobal::locale()->formatDate(m_dateTime.date(), KLocale::LongDate));
                break;
                case 'o':
                    string.append(KGlobal::locale()->formatTime(m_sunrise, false));
                break;
                case 'O':
                    string.append(KGlobal::locale()->formatTime(m_sunset, false));
                break;
                case 'F':
                    string.append(KGlobal::locale()->formatTime(m_dateTime.time(), false));
                break;
                case 'X':
                    string.append(KGlobal::locale()->formatTime(m_dateTime.time(), true));
                break;
                case 'g':
                    string.append(prettyTimezone());
                break;
                case 'j':
                    string.append("000");
                break;
                case 'p':
                    string.append((i18n("pm").length() > i18n("am").length())?i18n("pm"):i18n("am"));
                break;
                case 't':
                    string.append(QString::number(m_dateTime.toTime_t()));
                break;
                case 'w':
                    string.append("0");
                break;
                case 'Y':
                    string.append("0000");
                break;
                case 'Z':
                    string.append(m_timeZoneAbbreviation);
                break;
                case 'z':
                    string.append(m_timeZoneOffset);
                default:
                    string.append(m_timeFormat.at(index));
                break;
            }
        }
        else
        {
            string.append(m_timeFormat.at(index));
        }
    }

    if (m_timeFormat.at(length - 1) != '%')
    {
        string.append(m_timeFormat.at(length));
    }

    setText(string);

    m_page.setViewportSize(QSize(0, 0));
    m_page.mainFrame()->setZoomFactor(1);

    if (m_rescaleContents)
    {
        qreal widthFactor = (boundingRect().width() / m_page.mainFrame()->contentsSize().width());
        qreal heightFactor = (boundingRect().height() / m_page.mainFrame()->contentsSize().height());

        m_page.mainFrame()->setZoomFactor((widthFactor > heightFactor)?heightFactor:widthFactor);
    }

    if (formFactor() == Plasma::Horizontal)
    {
        setMinimumWidth(m_page.mainFrame()->contentsSize().width());
        setMinimumHeight(0);
    }
    else if (formFactor() == Plasma::Vertical)
    {
        setMinimumHeight(m_page.mainFrame()->contentsSize().height());
        setMinimumWidth(0);
    }
    else if (!m_rescaleContents)
    {
        resize(m_page.mainFrame()->contentsSize());
    }

    m_page.setViewportSize(m_rescaleContents?boundingRect().size().toSize():m_page.mainFrame()->contentsSize());

    setText(formatDateTime(m_dateTime, m_timeFormat));
}

void AdjustableClock::updateTheme()
{
    QPalette palette = m_page.palette();
    palette.setBrush(QPalette::Base, Qt::transparent);

    m_page.setPalette(palette);
    m_page.mainFrame()->evaluateJavaScript("document.fgColor = '" + Plasma::Theme::defaultTheme()->color(Plasma::Theme::TextColor).name() + '\'');

    update();
}

void AdjustableClock::resetTimeFormat()
{
    KConfigGroup configuration = config();

    m_timeFormat = configuration.readEntry("timeFormat", DEFAULT_FORMAT);

    updateSize();
}

QDateTime AdjustableClock::currentDateTime()
{
    Plasma::DataEngine::Data data = dataEngine("time")->query(currentTimezone());
    QDateTime dateTime = QDateTime(data["Date"].toDate(), data["Time"].toTime());
    dateTime = dateTime.addSecs(m_timeDifference);

    return dateTime;
}

QString AdjustableClock::formatDateTime(const QDateTime dateTime, QString format, bool clean)
{
    if (format.isEmpty())
    {
        return QString();
    }

    QString string;
    const int length = (format.length() - 1);

    for (int index = 0; index < length; ++index)
    {
        if (format.at(index) == '%' && format.at(index + 1) != '%')
        {
            ++index;

            switch (format.at(index).unicode())
            {
                case 'a': // weekday, short form
                    string.append(KGlobal::locale()->calendar()->weekDayName(dateTime.date(), KCalendarSystem::ShortDayName));
                break;
                case 'A': // weekday, long form
                    string.append(KGlobal::locale()->calendar()->weekDayName(dateTime.date(), KCalendarSystem::LongDayName));
                break;
                case 'b': // month, short form
                    string.append(KGlobal::locale()->calendar()->monthName(KGlobal::locale()->calendar()->month(dateTime.date()), KGlobal::locale()->calendar()->year(dateTime.date()), ((KGlobal::locale()->nounDeclension() && KGlobal::locale()->dateMonthNamePossessive())?KCalendarSystem::ShortNamePossessive:KCalendarSystem::ShortName)));
                break;
                case 'B': // month, long form
                    string.append(KGlobal::locale()->calendar()->monthName(KGlobal::locale()->calendar()->month(dateTime.date()), KGlobal::locale()->calendar()->year(dateTime.date()), KCalendarSystem::LongName));
                break;
                case 'c': // date and time format, short
                    string.append(KGlobal::locale()->formatDateTime(dateTime, KLocale::LongDate));
                break;
                case 'C': // date and time format, long
                    string.append(KGlobal::locale()->formatDateTime(dateTime, KLocale::ShortDate));
                break;
                case 'd': // day of the month, two digits
                    string.append(KGlobal::locale()->calendar()->dayString(dateTime.date(), KCalendarSystem::LongFormat));
                break;
                case 'e': // day of the month, one digit
                    string.append(KGlobal::locale()->calendar()->dayString(dateTime.date(), KCalendarSystem::ShortFormat));
                break;
                case 'f': // date format, short
                    string.append(KGlobal::locale()->formatDate(dateTime.date(), KLocale::ShortDate));
                break;
                case 'F': // time format, short
                    string.append(KGlobal::locale()->formatTime(dateTime.time(), false));
                break;
                case 'g': // timezone city
                    string.append(prettyTimezone());
                break;
                case 'H': // hour, 24h format
                    if (dateTime.time().hour() < 10)
                    {
                        string.append('0');
                    }

                    string.append(QString::number(dateTime.time().hour()));
                break;
                case 'I': // hour, 12h format
                    if ((((dateTime.time().hour() + 11) % 12) + 1) < 10)
                    {
                        string.append('0');
                    }

                    string.append(QString::number(((dateTime.time().hour() + 11) % 12) + 1));
                break;
                case 'j': // day of the year
                    string.append(QString::number(KGlobal::locale()->calendar()->dayOfYear(dateTime.date())));
                break;
                case 'k': // hour, 24h format, one digit
                    string.append(QString::number(dateTime.time().hour()));
                break;
                case 'l': // hour, 12h format, one digit
                    string.append(QString::number(((dateTime.time().hour() + 11) % 12) + 1));
                break;
                case 'm': // month, two digits
                    string.append(KGlobal::locale()->calendar()->monthString(dateTime.date(), KCalendarSystem::LongFormat));
                break;
                case 'M': // minute, two digits
                    if (dateTime.time().minute() < 10)
                    {
                        string.append('0');
                    }

                    string.append(QString::number(dateTime.time().minute()));
                break;
                case 'n': // month, one digit
                    string.append(KGlobal::locale()->calendar()->monthString(dateTime.date(), KCalendarSystem::ShortFormat));
                break;
                case 'o': // sunrise time
                    string.append(KGlobal::locale()->formatTime(m_sunrise, false));
                break;
                case 'O': // sunset time
                    string.append(KGlobal::locale()->formatTime(m_sunset, false));
                break;
                case 'p': // pm or am
                    string.append((dateTime.time().hour() >= 12)?i18n("pm"):i18n("am"));
                break;
                case 's': // second, one digit
                    string.append(QString::number(dateTime.time().second()));
                break;
                case 'S': // second, two digits
                    if (dateTime.time().second() < 10)
                    {
                        string.append('0');
                    }

                    string.append(QString::number(dateTime.time().second()));
                break;
                case 't': // UNIX timestamp
                    string.append(QString::number(dateTime.toTime_t()));
                break;
                case 'w': // day of week
                    string.append(QString::number(KGlobal::locale()->calendar()->dayOfWeek(dateTime.date())));
                break;
                case 'W': // week number
                case 'U':
                    string.append(QString::number(KGlobal::locale()->calendar()->weekNumber(dateTime.date())));
                break;
                case 'x': // date format, long
                    string.append(KGlobal::locale()->formatDate(dateTime.date(), KLocale::LongDate));
                break;
                case 'X': // time format, long
                    string.append(KGlobal::locale()->formatTime(dateTime.time(), true));
                break;
                case 'Y': // year, four digits
                    string.append(KGlobal::locale()->calendar()->yearString(dateTime.date(), KCalendarSystem::LongFormat));
                break;
                case 'y': // year, two digits
                    string.append(KGlobal::locale()->calendar()->yearString(dateTime.date(), KCalendarSystem::ShortFormat));
                break;
                case 'Z': // timezone abbreviation
                    string.append(m_timeZoneAbbreviation);
                break;
                case 'z': // timezone offset
                    string.append(m_timeZoneOffset);
                break;
                default:
                    string.append(format.at(index));
                break;
            }
        }
        else if (clean && format.at(index) == '\n')
        {
            string.append("\\n");
        }
        else if (clean && format.at(index) == '\'')
        {
            string.append("\\'");
        }
        else
        {
            string.append(format.at(index));
        }
    }

    if (format.at(length - 1) != '%')
    {
        string.append(format.at(length));
    }

    return string;
}

QList<QAction*> AdjustableClock::clipboardActions()
{
    QList<QAction*> actions;
    QDateTime dateTime = currentDateTime();

    for (int i = 0; i < m_clipboardFormats.count(); ++i)
    {
        if (m_clipboardFormats.at(i).isEmpty())
        {
            QAction *action = new QAction(NULL);
            action->setSeparator(true);

            actions.append(action);
        }
        else
        {
            actions.append(new QAction(formatDateTime(dateTime, m_clipboardFormats.at(i)), NULL));
        }
    }

    return actions;
}

QList<QAction*> AdjustableClock::contextualActions()
{
    QList<QAction*> actions = ClockApplet::contextualActions();

    for (int i = 0; i < actions.count(); ++i)
    {
        if (actions.at(i)->text() == i18n("C&opy to Clipboard"))
        {
            actions.removeAt(i);
            actions.insert(i, m_clipboardAction);
        }
    }

    return actions;
}

#include "AdjustableClock.moc"
