/*
 *   Copyright (C) 2007 Teemu Rytilahti <tpr@iki.fi>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License version 2 as
 *   published by the Free Software Foundation
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <QAction>
#include <QStringList>
#include <QIcon>
#include <QDBusInterface>
#include <QDBusReply>

#include <KRun>
#include <KUrl>
#include <KLocale>
#include <KService>
#include <KSharedPtr>
#include <KMimeType>
#include <KIconLoader>
#include <KStandardDirs>

#include <kservicetypetrader.h>

#include "webshortcutrunner.h"


WebshortcutRunner::WebshortcutRunner(QObject *parent, const QVariantList& args)
    : Plasma::AbstractRunner(parent)
{
    Q_UNUSED(args);
    // set the name shown after the result in krunner window
    setObjectName(i18n("Web shortcut"));
    // query ktrader for all available searchproviders and preload the default icon
    m_offers = KServiceTypeTrader::self()->query("SearchProvider");
    m_icon = QIcon(KIconLoader::global()->loadIcon("konqueror", K3Icon::Small));
}

WebshortcutRunner::~WebshortcutRunner() {}

QAction *WebshortcutRunner::accepts(const QString& term) {
    QString searchTerm = term.toLower();
    foreach(KSharedPtr<KService> service, m_offers) {
        QStringList list;
        // hmm, how about getting the keys for the localized sites?
        list << service->property("Keys").toStringList();

        foreach(QString key, list) {
            // FIXME? should we look for the used separator from the konqi's settings?
            key = key.toLower() + ":";
            if(searchTerm.startsWith(key)) {
                m_url = getSearchQuery(service->property("Query").toString(),searchTerm);
                QAction *action = new QAction(QString("Search from %1").arg(service->name()), this);

                // let's try if we can get a proper icon from the favicon cache
                QIcon icon = getFavicon(m_url);
                if(!icon.isNull())
                    action->setIcon(icon);
                else
                    action->setIcon(m_icon);

                return action;
            }
        }
    }
    return 0;
}

KUrl WebshortcutRunner::getSearchQuery(const QString &query, const QString &term) {
    // FIXME delimiter check like for above?
    QStringList tempList = term.split(":");
    if(tempList.count() > 0) {
        QString searchWord(tempList[1]);
        QString finalQuery(query);
        // FIXME? currently only basic searches are supported
        finalQuery.replace("\\{@}", searchWord);
        KUrl url(finalQuery);
        return url;
    }

    return KUrl();
}

QIcon WebshortcutRunner::getFavicon(const KUrl &url) {
    // query the favicons module
    QDBusInterface favicon("org.kde.kded", "/modules/favicons", "org.kde.FavIcon");
    QDBusReply<QString> reply = favicon.call("iconForUrl", url.url());

    if(!reply.isValid())
        return QIcon();

    // locate the favicon
    QString iconFile = KGlobal::dirs()->findResource("cache",reply.value()+".png");
    QIcon icon = QIcon(iconFile);

    if(!icon.isNull())
        return icon;

    return QIcon();
}

bool WebshortcutRunner::exec(QAction* action, const QString& command) {
    kDebug() << "command: " << command;
    kDebug() << "url: " << m_url.url();
    return( KRun::runUrl ( m_url, "text/html", 0 ) != false );
}

#include "webshortcutrunner.moc"
