/*
 *   Copyright (C) 2007 Michael Olbrich <michael-olbrich@web.de>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License version 2 as
 *   published by the Free Software Foundation
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "sensorsengine.h"

#include <QtCore/QTimer>
#include <QtCore/QDir>
#include <QtCore/QFile>

#include <KDebug>
#include <plasma/datacontainer.h>

SensorsEngine::SensorsEngine(QObject* parent, const QVariantList& args)
    : Plasma::DataEngine(parent)
{
    Q_UNUSED(args)

    QDir hwmon("/sys/class/hwmon/");
    QStringList list = hwmon.entryList(QDir::Dirs|QDir::NoDotAndDotDot);
    QStringList inputfilter;
    inputfilter << "*_input";
    foreach (QString devicename, list) {
        QDir device(hwmon);
        if (!device.cd(devicename)) {
            continue;
        }
        if (!device.cd("device")) {
            continue;
        }
        device.setPath(device.canonicalPath());
        QStringList sensors = device.entryList(inputfilter, QDir::Files);
        foreach (QString sensor, sensors) {
            QFile file(device.path() + QDir::separator() + sensor);
            if (!file.open(QIODevice::ReadOnly)) {
                continue;
            }
            QByteArray content = file.readAll();
            if (content.length() == 0) {
                continue;
            }
            file.close();
            sensor = sensor.left(sensor.indexOf("_input"));
            SensorData data(device.path(), sensor);
            m_sensors[device.dirName()+ "/" + sensor] = data;
        }
    }

    m_timer = new QTimer(this);
    m_timer->setSingleShot(false);
    connect(m_timer, SIGNAL(timeout()), this, SLOT(timeout()));
}

SensorsEngine::~SensorsEngine()
{
    delete m_timer;
}

QStringList SensorsEngine::sources() const
{
    return m_sensors.keys();
}

bool SensorsEngine::sourceRequested(const QString &name)
{
    SensorMap::ConstIterator iter = m_sensors.find(name);
    if (iter == m_sensors.end()) {
        return false;
    }
    QDir device((*iter).path);
    QFileInfo inputInfo(device, (*iter).name + "_input");
    if (!inputInfo.isReadable()) {
        return false;
    }
    QStringList filter;
    filter << (*iter).name + "_*";
    QStringList params = device.entryList(filter, QDir::Files);
    foreach (QString param, params) {
        QFile file(device.path() + QDir::separator() + param);
        if (!file.open(QIODevice::ReadOnly)) {
            continue;
        }
        QString content = file.readAll();
        file.close();
        param = param.mid((*iter).name.length() + 1);
        bool ok;
        int value = content.toInt(&ok);
        if (ok) {
            setData(name, param, value);
        } else {
            setData(name, param, content);
        }
    }
    if (!m_timer->isActive()) {
        m_timer->start(1000);
    }
    return true;
}

void SensorsEngine::update(Plasma::DataContainer* source)
{
    SensorData &s = m_sensors[source->objectName()];
    QFile file(s.path + QDir::separator()
               + s.name + "_input");
    if (!file.open(QIODevice::ReadOnly)) {
        return;
    }
    QString content = file.readAll();
    file.close();
    bool ok;
    int value = content.toInt(&ok);
    QVariant data;
    if (ok) {
        data = value;
    } else {
        data = content;
    }
    if (source->data()["input"] != data) {
          source->setData("input", data);
    }
}

void SensorsEngine::timeout()
{
    SourceDict dict = sourceDict();
    for (SourceDict::ConstIterator it = dict.begin(); it != dict.end(); ++it) {
        update(*it);
    }
    checkForUpdates();
}

#include "sensorsengine.moc"
