/*
 *   Copyright (C) 2007 Zack Rusin <zack@kde.org>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License version 2 as
 *   published by the Free Software Foundation
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "rotator.h"

#include <QGraphicsSceneMouseEvent>
#include <QPainter>

Rotator::Rotator(QObject *parent, const QVariantList &args)
    : Plasma::Applet(parent, args)
{
    setHasConfigurationInterface(false);
    setAcceptsHoverEvents(true);
    m_degrees = 0;
    m_pixmap = QPixmap(":/images/sample.png");
    connect(&m_timer, SIGNAL(timeout()),
            this, SLOT(rotate()));
    m_timer.setInterval(1000/30);
}

Rotator::~Rotator()
{
}

QRectF Rotator::boundingRect() const
{
    return QRectF(QPointF(0, 0), m_pixmap.size());
}

void Rotator::paintInterface(QPainter *painter,
                             const QStyleOptionGraphicsItem *,
                             const QRect &contentsRect)
{
    if (m_degrees >= 270 || m_degrees <= 90) {// front side
        painter->drawPixmap(0, 0, m_pixmap);
    } else { //back side
        painter->setRenderHint(QPainter::Antialiasing);
        QLinearGradient grad(boundingRect().width()/2,
                             0,
                             boundingRect().width()/2,
                             boundingRect().height());
        grad.setColorAt(0, QColor(0, 0, 0, 127));
        grad.setColorAt(0.5, QColor(50, 50, 50, 127));
        grad.setColorAt(1, QColor(255, 255, 255, 127));
        painter->fillRect(boundingRect(), QBrush(grad));
        painter->setPen(Qt::white);
        painter->save();
        //here we undo the transformation
        // and set it like it would be the front
        // (we don't want the back contents to be flipped :) )
        QTransform trans;
        trans.translate(boundingRect().width()/2,
                        boundingRect().height()/2);
        int angle = m_degrees;
        if (m_degrees <= 180)
            angle += 180;
        else
            angle -= 180;
        trans.rotate(angle, Qt::YAxis);
        trans.translate(-boundingRect().width()/2,
                        -boundingRect().height()/2);
        painter->setTransform(m_transform.inverted(), true);
        painter->setTransform(trans, true);
        QFont font = painter->font();
        font.setBold(true);
        font.setUnderline(true);
        painter->setFont(font);
        painter->drawText(20, 20, "This is my back side!");
        font.setBold(false);
        font.setUnderline(false);
        painter->setFont(font);
        painter->drawText(30, 50, "Some content goes here :)");
        painter->restore();
    }
}

void Rotator::mousePressEvent(QGraphicsSceneMouseEvent *e)
{
    Applet::mousePressEvent(e);
    if (m_timer.isActive())
        m_timer.stop();
    else
        m_timer.start();
}

void Rotator::mouseReleaseEvent(QGraphicsSceneMouseEvent *e)
{
    Applet::mouseReleaseEvent(e);
}

void Rotator::mouseMoveEvent(QGraphicsSceneMouseEvent *e)
{
    Applet::mouseMoveEvent(e);
}

void Rotator::wheelEvent(QGraphicsSceneWheelEvent *e)
{
    Applet::wheelEvent(e);
}

void Rotator::rotate()
{
    m_degrees += 5;
    if (m_degrees == 360)
        m_degrees = 0;
    m_transform = QTransform();
    m_transform.translate(boundingRect().width()/2,
                          boundingRect().height()/2);
    m_transform.rotate(m_degrees, Qt::YAxis);
    m_transform.translate(-boundingRect().width()/2,
                          -boundingRect().height()/2);
    setTransform(m_transform);
}

#include "rotator.moc"
