/***************************************************************************
 *   Copyright (C) 2007 by Anne-Marie Mahfouf <annma@kde.org>              *
 *   Copyright (C) 2007 by Stephan Binner <binner@kde.org>                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .        *
 ***************************************************************************/

#include "groupphoto.h"

#include <QApplication>
#include <QBitmap>
#include <QGraphicsScene>
#include <QPaintEvent>
#include <QPainter>
#include <QPixmap>
#include <QStyleOptionGraphicsItem>
#include <QGridLayout>
#include <QRadioButton>
#include <QGroupBox>
#include <QGraphicsSceneDragDropEvent>
#include <QPen>
#include <QCheckBox>
#include <QSpinBox>
#include <QTimer>

#include <KDebug>
#include <kdialog.h>
#include <KLocale>
#include <KIcon>
#include <kcolorbutton.h>
#include <kcombobox.h>
#include <KSharedConfig>
#include <kstandarddirs.h>
#include <kurlrequester.h>
#include <kio/netaccess.h>
#include <kurl.h>
#include <QPicture>

#include <plasma/dataenginemanager.h>
#include <plasma/widgets/vboxlayout.h>
#include <plasma/phase.h>

using namespace Plasma;

HitBox::HitBox(QGraphicsItem *parent, const QString& name, int width)
    : QGraphicsItem(parent)
{
    m_width = width;

    m_label = new Plasma::LineEdit(this);
    m_label->setDefaultText(name);
    m_label->setTextInteractionFlags(Qt::NoTextInteraction);
    m_label->setPos(0, m_width);
    m_label->setTextWidth(100);
}

HitBox::~HitBox()
{
}

QRectF HitBox::boundingRect() const
{
    return QRectF(0,0,m_width,m_width);
}

void HitBox::paint(QPainter *painter, const QStyleOptionGraphicsItem */*option*/, QWidget */*widget*/)
{
    painter->setPen(QPen(QBrush(QColor(255,255,255,200)), 1,Qt::SolidLine, Qt::RoundCap));
    painter->drawRect(0, 0, m_width, m_width);
}

Groupphoto::Groupphoto(QObject *parent, const QVariantList &args)
    : Plasma::Applet(parent, args), 
      m_dialog(0)
{
    setHasConfigurationInterface(true);
    setAcceptDrops(true);
    m_dialog = 0;

    // Get config values
    KConfigGroup cg = globalConfig();
    m_frameColor = cg.readEntry("frameColor", QColor(70, 90, 130));
    m_frame = cg.readEntry("frame", false);
    m_shadow = cg.readEntry("shadow", true);
    m_squareCorners = cg.readEntry("squareCorners", true);
    m_roundCorners = cg.readEntry("roundCorners", false);
    m_pixelSize = cg.readEntry("size", 1000);
    m_rotation = cg.readEntry("rotation",0);
    m_year = cg.readEntry("year", 2007);
    m_layout = new Plasma::VBoxLayout(0);
    m_layout->setGeometry(QRectF(0, 0, 400, 800));
    m_layout->setMargin(12);

    //Initialize the picture
    m_cmpPicture = NULL;
    choosePicture();
    composePicture();

    readNames();
    setAcceptsHoverEvents(true);

    m_timer = new QTimer();
    m_timer->setInterval(500);
    m_timer->setSingleShot(true);
    connect(m_timer, SIGNAL(timeout()), this, SLOT(search()));

    m_searchEdit = new Plasma::LineEdit(this);
    m_searchEdit->setTextInteractionFlags(Qt::TextEditorInteraction);
    m_searchEdit->setDefaultText(i18n("Enter participant to search here..."));
    Phase::self()->animateItem(m_searchEdit, Phase::Appear);
    const int frameThickness = 16;
    m_searchEdit->setPos(frameThickness + 2, frameThickness + 2);
    m_searchEdit->setTextWidth(170);
    connect(m_searchEdit, SIGNAL(editingFinished()), this, SLOT(search()));
    connect(m_searchEdit, SIGNAL(textChanged(const QString&)), this, SLOT(autoSearch(const QString&)));
}

void Groupphoto::readNames()
{
    foreach (_person* person, m_personList)
       delete person;
    m_personList.clear();

    QFile file(KGlobal::dirs()->findAllResources("data", (m_year==2007) ? "groupphoto/group-photo.html" : "groupphoto/group-photo-2006.html")[0]);

    if (!file.open(QIODevice::ReadOnly | QIODevice::Text))
       return;

    QTextStream in(&file);
    while (!in.atEnd()) {
        QString line = in.readLine();
        if (line.startsWith("  <area shape=\"circle\" title=\"")) {
            QStringList parts = line.split("\"");
            QStringList coords = parts[5].split(",");
            _person* person = new _person;
            person->name = parts[3];
            person->x = coords[0].toInt();
            person->y = coords[1].toInt();
            person->width = coords[2].toInt();
            m_personList.append(person);
        }
    }

}

void Groupphoto::autoSearch(const QString &word)
{
    m_timer->start();
}

void Groupphoto::search()
{
    foreach (HitBox* hitbox, m_hitboxList)
       delete hitbox;
    m_hitboxList.clear();

    const QString searchTerm = m_searchEdit->toPlainText();

    foreach (_person* person, m_personList) {
       if (person->name.indexOf(searchTerm, 0, Qt::CaseInsensitive)!=-1) {

          HitBox* hitbox = new HitBox(this, person->name, person->width);
         const int frameThickness = 16;
         hitbox->setPos((contentSize().width()-(frameThickness*2))/myPicture.width()*person->x,
                      (contentSize().height()-(frameThickness*2))/myPicture.height()*person->y);
         hitbox->setZValue(1);
         m_hitboxList.append(hitbox);
       }
    }

}

void Groupphoto::hoverMoveEvent(QGraphicsSceneHoverEvent *event)
{ 
    if (m_oldPos==event->pos()) 
       return;

    m_oldPos = event->pos();

    foreach (HitBox* hitbox, m_hitboxList)
       delete hitbox;
    m_hitboxList.clear();

    const int frameThickness = 16;
    foreach (_person* person, m_personList) {
       QRect rect((contentSize().width()-(frameThickness*2))/myPicture.width()*person->x, (contentSize().height()-(frameThickness*2))/myPicture.height()*person->y, person->width, person->width);
       if (rect.contains(event->pos().toPoint()) ) {
         HitBox* hitbox = new HitBox(this, person->name, person->width);
         const int frameThickness = 16;
         hitbox->setPos((contentSize().width()-(frameThickness*2))/myPicture.width()*person->x,
                      (contentSize().height()-(frameThickness*2))/myPicture.height()*person->y);
         hitbox->setZValue(1);
         m_hitboxList.append(hitbox);
         break;
       }
    }
}

void Groupphoto::resizeEvent(QResizeEvent *)
{
    prepareGeometryChange();
}

void Groupphoto::updated(const QString& source, const Plasma::DataEngine::Data &data)
{
    Q_UNUSED(source);
    Q_UNUSED(data);
    QGraphicsItem::update();

}

void Groupphoto::choosePicture()
{
    QString currentUrl = KGlobal::dirs()->findAllResources("data", (m_year==2007) ? "groupphoto/group-photo.jp*" : "groupphoto/group-photo-2006.jp*" )[0];

    QString tmpFile;
    if ( !KIO::NetAccess::download( currentUrl, tmpFile, 0L ) ) {
	return;
    }
    myPicture.load(tmpFile);
    if (myPicture.isNull()) {
	return;
    }
}

void Groupphoto::showConfigurationInterface()
{
    if (m_dialog == 0) {
        m_dialog = new KDialog;
        m_dialog->setCaption( "Configure Group Photo Frame" );
	ui.setupUi(m_dialog->mainWidget());
        m_dialog->setButtons( KDialog::Ok | KDialog::Cancel | KDialog::Apply );
        connect( m_dialog, SIGNAL(applyClicked()), this, SLOT(configAccepted()) );
        connect( m_dialog, SIGNAL(okClicked()), this, SLOT(configAccepted()) );
    }
    ui.frameCheckBox->setChecked(m_frame);
    ui.shadowCheckBox->setChecked(m_shadow);
    ui.squareButton->setChecked(m_squareCorners);
    ui.roundButton->setChecked(m_roundCorners);
    ui.spinSize->setValue(m_pixelSize);
    ui.sliderRotation->setValue(m_rotation/5);
    ui.yearComboBox->setCurrentIndex( (m_year==2007) ? 1 : 0);
	
    m_dialog->show();
}

void Groupphoto::configAccepted()
{

    KConfigGroup cg = globalConfig();
    m_frameColor = ui.changeFrameColor->color();
    cg.writeEntry("frameColor", m_frameColor);
    m_frame = ui.frameCheckBox->isChecked();
    cg.writeEntry("frame", m_frame);
    m_shadow = ui.shadowCheckBox->isChecked();
    cg.writeEntry("shadow", m_shadow);
    m_pixelSize = ui.spinSize->value();
    cg.writeEntry("size", m_pixelSize);
    m_rotation = ui.sliderRotation->value()*5;
    cg.writeEntry("rotation", m_rotation);
    m_squareCorners = ui.squareButton->isChecked();
    cg.writeEntry("squareCorners", m_squareCorners);
    m_roundCorners = ui.roundButton->isChecked();
    cg.writeEntry("roundCorners", m_roundCorners);
    m_year = (ui.yearComboBox->currentIndex() ==0) ? 2006 : 2007;
    cg.writeEntry("year", m_year);

    foreach (HitBox* hitbox, m_hitboxList)
       delete hitbox;
    m_hitboxList.clear();

    choosePicture();
    composePicture();

    readNames();

    cg.config()->sync();
  
}

QSizeF Groupphoto::contentSizeHint() const
{
    //FIXME: this needs to be settable / adjustable
    // return m_layout->geometry();
    return QSizeF(m_cmpPicture->width(), m_cmpPicture->height());
}

 void Groupphoto::dragEnterEvent(QGraphicsSceneDragDropEvent *event)//not sure it's needed
 {
    if (event->mimeData()->hasImage()) {
         event->acceptProposedAction();
    }
 }

void Groupphoto::dropEvent(QGraphicsSceneDragDropEvent *event)
{
    kDebug () << "Groupphoto::dropEvent";
    event->setAccepted(event->mimeData()->hasFormat("image/png"));//hasImage());
    /*if (event->mimeData()->hasFormat("image/png")) {//hasImage()) {
	image = qvariant_cast<QImage>(event->mimeData()->imageData());
	kDebug() << "image " << image.rect();
    }*/
    update();
}

Groupphoto::~Groupphoto()
{
    delete m_layout;
}

void Groupphoto::composePicture()
{
    QPixmap tempPicture;
    QPicture paintRecorder;
    QPainter p;
    
    // scale picture
    if (myPicture.width() > myPicture.height()) {
	    tempPicture = myPicture.scaledToWidth(m_pixelSize, Qt::SmoothTransformation);
    }
    else {
	    tempPicture = myPicture.scaledToHeight(m_pixelSize, Qt::SmoothTransformation);
    }
    
    
    int roundingFactor = 12 * m_roundCorners;
    const int frameThickness = 16;
    double swRadious = 20;
    int swRound =  12 + m_roundCorners*6 + m_frame * 4;
    //int m_swOffset = 3;
    
    p.begin(&paintRecorder);
    p.setRenderHint(QPainter::Antialiasing);
    
    //shadow
    if (m_shadow) {
	int swWidth = (int) (tempPicture.width() + frameThickness*m_frame + swRadious) ;
	int swHeight = (int) (tempPicture.height() + frameThickness * m_frame + swRadious);
	p.setBrush(QBrush(Qt::NoBrush));
	p.setPen(QPen(Qt::black, 1, Qt::SolidLine, Qt::RoundCap,Qt::RoundJoin));	
	for (int r=1; r<=swRadious; r++) {
	    p.setOpacity((r/swRadious)*(r/swRadious));
	    if (r == swRadious) {
		p.setBrush(QBrush(Qt::black,Qt::SolidPattern)); //the last rect will be filled
	    }
	    p.drawRoundRect(r, r, swWidth - 2 * r, swHeight - 2 * r, swRound-r, swRound-r);
	}
	p.translate(8,8);    
    }

    // frame
    if (m_frame) {
	QPainterPath framePath;
	p.setOpacity(0.5);
	QBrush blueBrush(m_frameColor);
	framePath.addRoundRect(0, 0, tempPicture.width() + frameThickness, tempPicture.height()+ frameThickness, roundingFactor);
	p.fillPath(framePath, blueBrush);//it's easier to fill the base with the frame color
	framePath.closeSubpath();
	p.translate(frameThickness/2, frameThickness/2);
    }
    // black frame
    p.setOpacity(1);
    QPainterPath path;
    if (m_frame) {
	p.setPen(QPen(Qt::black, 4, Qt::SolidLine, Qt::RoundCap,Qt::RoundJoin));
    }
    else {
	p.setPen(QPen(Qt::black, 4, Qt::NoPen, Qt::RoundCap,Qt::RoundJoin));
    }
    path.addRoundRect(0, 0, tempPicture.width(), tempPicture.height(), roundingFactor-4);
    p.drawPath(path);
    // picture
    QBrush myBrush(tempPicture);
    p.fillPath( path, myBrush);
    path.closeSubpath();	
    p.end();
    
    //Draw to a pixmap
    delete m_cmpPicture;
    m_cmpPicture = new QPixmap(paintRecorder.boundingRect().size());
    m_cmpPicture->fill(Qt::transparent);
    QPainter p2;
    p2.begin(m_cmpPicture);
    p2.drawPicture( 0 - paintRecorder.boundingRect().x(), 0-paintRecorder.boundingRect().y(),paintRecorder);
    prepareGeometryChange();
    update();	
}

void Groupphoto::paintInterface(QPainter *p, const QStyleOptionGraphicsItem *option, const QRect &contentsRect)
{
    
    Q_UNUSED(option);

    setDrawStandardBackground(true);
    p->setRenderHint(QPainter::SmoothPixmapTransform, true);
    //Set transform matrix (rotation) 
    //FIXME where do I should put this?
    QTransform transformMatrix;
    transformMatrix.rotate(m_rotation);
    setTransform(transformMatrix,0);

    //Paint the composited pixmap
    p->drawPixmap (0,0, *m_cmpPicture);

}
	

#include "groupphoto.moc"
