/*
 *   Copyright (C) 2007 Matt Broadstone <mbroadst@kde.org>
 *   Copyright (C) 2007 Matias Costa <m.costacano@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License version 2 as
 *   published by the Free Software Foundation
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#ifndef STDDESKTOP_H
#define STDDESKTOP_H

#include <QHash>

#include <plasma/applet.h>


namespace Plasma
{
    class DataEngine;
}

class KDirLister;
class Launcher;
class FileLauncher;
class DeviceLauncher;
class KFileItemList;
class KFileItem;
class DesktopOrganizer;
class KToggleAction;

class StandardDesktop : public Plasma::Applet
{
    Q_OBJECT
public:
    StandardDesktop(QObject *parent, const QVariantList &args);
    ~StandardDesktop();

    QSizeF contentSizeHint() const;
    void paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
               const QRect& rect);

    bool isGridAligned() const;
    void setGridAligned(bool align);

    void configureMedia();

    QSizeF gridSize() const;

    // Temporarily public
    void alignToGrid(Launcher *item);
    void alignToGrid(Launcher *item, const QPointF &pos);

    inline QPoint mapToGrid(const QPointF &pos) const;
    inline QPointF mapFromGrid(const QPoint &pos) const;

    /**
        Sets the grid size to the nearest size which results in
        a integer number of grid squares. If the desktop width is 1024
        and the desired grid width is 100 the resulting gridSize will be
        102.4 because it's the nearest width with a integer number of
        divisions (10)

        @arg size The desired gridSize to be set.
     */

    void setGridSize(const QSizeF& gridSize);
    /**
        The width and height of the size returned by this method represents
        the grid places a Launcher can be placed.
     */
    QSize gridDimensions() const;


    void changeAlignment(bool horizontal);

    virtual QList<QAction*> contextActions();

public Q_SLOTS:
    void updated(const QString &, QHash<QString, QVariant> );

private Q_SLOTS:
    void clearDesktop();
    void listingCompleted();
    void newItemsFound(const KFileItemList &items);
    void itemDeleted(KFileItem *file);

    void sourceAdded(const QString &source);
   
    void slotAlignHorizontal();
    void slotAlignVertical();
    void slotLock(bool);
    void slotAlignToGrid(bool b);

    void slotSortByNameCaseSensitive();
    void slotSortByNameCaseInsensitive();
    void slotSortBySize();
    void slotSortByType();
    void slotSortByDate();

protected:
    bool sceneEventFilter(QGraphicsItem *watched, QEvent *event);
    void createMenu();
    void saveIconPosition();
    void readIconPosition();

private:
    KDirLister *m_dirLister;
    Plasma::DataEngine *m_solidEngine;

    QList<Launcher*> m_desktopItems;

    QHash<KFileItem*, FileLauncher*> m_fileItems;
    QHash<QString, DeviceLauncher*> m_solidDevices;

    QList<QAction*> actions;

    DesktopOrganizer *m_organizer;

    bool m_gridAlign;
    bool m_bLockIcon;
    bool m_enableMedia;
    QSizeF m_gridSize;
    Launcher *m_target;
    KToggleAction* m_lockIcon;
    KToggleAction* m_alignToGrid;
};

QPoint StandardDesktop::mapToGrid(const QPointF &pos) const
{
    return QPoint(int(pos.x()/m_gridSize.width()),
                  int(pos.y()/m_gridSize.height()));
}

QPointF StandardDesktop::mapFromGrid(const QPoint &pos) const
{
    return QPointF(pos.x()*m_gridSize.width() +m_gridSize.width()/2,
                  pos.y()*m_gridSize.height() +m_gridSize.height()/2);
}

#endif

